/* THIS FILE HAS BEEN REWRITTEN BY GENNADY MARGOLIN ON 17/04/2001
   Last changes: 03.02.02 */

/*-------------------------------------------------------------------------------------
  The CTRW software is produced by Brian Berkowitz, Weizmann Institute of Science.
  Permission to use the CTRW software is hereby granted to non-profit educational
  and research institutions, for educational and research purposes only, provided
  and as long as this Notice appears.

  THE CTRW SOFTWARE IS BEING DEVELOPED AS A TOOL FOR SCIENTIFIC RESEARCH. HENCE IT
  IS NOT PRESENTED AS ERROR FREE, ACCURATE, COMPLETE, OR USEFUL FOR ANY SPECIFIC
  APPLICATION, AND THE WEIZMANN INSTITUTE MAKES NO WARRANTY OR REPRESENTATION THERETO.

  THE WEIZMANN INSTITUTE SHALL NOT BE LIABLE FOR ANY CLAIMS, DEMANDS, LIABILITIES,
  COSTS, LOSSES, DAMAGES OR EXPENSE OF WHATSOEVER KIND OR NATURE CAUSED TO OR SUFFERED
  BY ANY PERSON OR ENTITY THAT DIRECTLY OR INDIRECTLY ARISE OUT OR RESULT FROM THE USE
  OF THE CTRW SOFTWARE OR IN CONNECTION THEREOF.

  For usage by and for commercial entities please contact Brian Berkowitz.
--------------------------------------------------------------------------------------*/
/*------------------------------------------------------------------------
 Module:        /home/ctrw/CTRW/ctrw/ctrw_01.c
 Author:        Georg Kosakowski
 Project:       ctrw-lib for grace
 State:         working
 Creation Date: 22.05.2000
 Description:   ctrw dynamic library for use with GRACE
                http://plasma-gate.weizmann.ac.il/GRACE/
                Version 0.1 - 17 May 1999 - G. Kosakowski g.kosakowski@gmx.net
                Version 0.2 - 26 March 2000 - G. Kosakowski g.kosakowski@gmx.net
------------------------------------------------------------------------*/


#include "./ctrw.h"

/*========================================================================
   STANDARD CTRW PROCEDURES FOR 0 < beta < 2 , beta<>1 and no backflow
========================================================================*/
double FPTD(double tm, double beta, double r, double t) {
  
  if (beta<1) return FPTD01(tm,beta,r,t);
  else return FPTD12(tm,beta,r,t);

}

double CFPTD(double tm, double beta, double r, double t) {
  
  if (beta<1) return CFPTD01(tm,beta,r,t);
  else return CFPTD12(tm,beta,r,t);

}

double SCD(double R, double beta, double kapa, double L) {

  
  if (beta<1) return SCD01(R,beta,kapa,L);
  else return SCD12(R,beta,kapa,L);

}

double CSCD(double R, double beta, double kapa, double L) {

  
  if (beta<1) return CSCD01(R,beta,kapa,L);
  else return CSCD12(R,beta,kapa,L);

}

/*------------------------------------------------------------------------
                         SPATIAL MOMENTS
--------------------------------------------------------------------------
C=c_beta/l_bar and C_1=c_1/l_bar are constants of motion indep. of l_bar.
For 0<beta<1 have psi*(u)=1-c_beta*u^beta+c_1*u+h.o.;
                  R=t^beta/C;
                  kapa=(C_1/C)*t^(beta-1);
For 1<beta<2 have psi*(u)=1-c_1*u+c_beta*u^beta+h.o, with c_1=t_bar;
                  R=t/C_1 (R=wt and w=l_bar/t_bar);
                  kapa=(C/C_1)*t^(1-beta);
------------------------------------------------------------------------*/
double MEAN(double beta, double C, double C_1, double t) {

  int j, m;
  double R, kapa, gama, eps, sum;

  if(t<=0.) return 0.;

  if(beta==1.) {
    if(C==C_1) return -1000.; /* should be infinity */
    return t/fabs(C-C_1);
  }

  if(beta<0.) beta=0.;
  if(beta>2.) beta=2.;

  if(beta<1.) {
    if(C==0.) {
      if(C_1==0.) return -1000.; /* infinity */
      return t/C_1;
    }
    gama=beta;
    R=pow(t,beta)/C;
    kapa=C_1/C*pow(t,beta-1.);
    if(beta==0.) eps=2.; /* any number >1, to get m=0 */
    else eps=1./beta-1.;
  }

  else {
    if(C_1==0.) return -1000.; /* infinity */
    gama=1.;
    R=t/C_1;
    kapa=C/C_1*pow(t,1.-beta);
    eps=beta-1.;
  }

  m = (int) floor(1./eps);

  sum=1./MyGamma(1.+gama,0);
  for(j=1 ; j<=m ; j++)
    sum+=pow(kapa,j)/MyGamma(1.+gama*(1.-eps*j),0);
  
  return sum*R;
}
/*----------------------------------------------------------------------------*/
double ST_D(double beta, double C, double C_1, double t) {

  int j, m, n;
  double R, kapa, gama, eps, sum, sum1;

  if(t<=0.||beta==1.) return 0.;

  if(beta<0.) beta=0.;
  if(beta>2.) beta=2.;

  if(beta<1.) {
    if(C<=0.) return 0.; /* must be non-negative */
    gama=beta;
    R=pow(t,beta)/C;
    kapa=C_1/C*pow(t,beta-1.); /* if kapa is large negative - might be sum<0 */
    if(beta==0.) eps=2.; /* any number >1, to get m=0 */
    else eps=1./beta-1.;
  }

  else {
    if(C_1==0.) return -1000.; /* infinity */
    gama=1.;
    R=t/C_1;
    kapa=C/C_1*pow(t,1.-beta);
    eps=beta-1.;
  }

  m = (int) floor(1./eps);

  sum=2./MyGamma(1.+2.*gama,0)-1./pow(MyGamma(1.+gama,0),2);
  for(j=1 ; j<=m ; j++) {
    for(n=0, sum1=0. ; n<=j ; n++)
      sum1+=1./MyGamma(1.+gama*(1.-eps*n),0)/MyGamma(1.+gama*(1.-eps*(j-n)),0);
    sum+=pow(kapa,j)*(2.*(j+1)/MyGamma(1.+gama*(2.-eps*j),0)-sum1);
  }
  
  if(sum<0.) return -1.; /* cannot have negative variance!! */
  return R*sqrt(sum);
}

/*========================================================================
                  end procedures for 0<beta<2, beta<>1.
==========================================================================
==========================================================================
                   PROCEDURES FOR 0 < beta < 1
========================================================================*/

/*------------------------------------------------------------------------
Procedure:     FPTD01 ID:1
Purpose:       calculation FPTD for beta < 1
Input:         tmeff: factor for shift along x-axis (effective mean time)
               beta: "dispersion" parameter
	       -r*tmeff: starting time (r=0 leads to the old formulas);
	           usually r>0 gives a correction; rigorously speaking,
	           correct to use only for beta>0.5; we don't set it to 0
		   for small betas because of the continuity needed by 
		   fitting routines. Should be |r|<1.
               t: real time for experimental data point
Output:        FPTD01: value of FPTD for the input parameters
Errors:        if beta < 0.1 -> beta = 0.1
               if beta > 0.97 -> beta=0.97
------------------------------------------------------------------------*/
double FPTD01 (double tmeff, double beta, double r, double t) {

  double x;
  if(t<=-r*tmeff) return 0.;

  if(beta>0.99) beta=0.99;
  else if(beta<0.1) beta=0.1;
	
  x=pow(t/tmeff+r,-beta);
  return f(x,beta)*beta*x/(t+r*tmeff);
}

/*------------------------------------------------------------------------
Procedure:     CFPTD01 ID:1
Purpose:       calculation of cumulative FTPD (or FTPD for step input)
Input:         tmeff: factor for shift along x-axis (effective mean time)
               beta: "dispersion parameter"
	       -r*tmeff: starting time (r=0 leads to the old formulas);
	           usually r>0 gives a correction; rigorously speaking,
	           correct to use only for beta>0.5; we don't set it to 0
		   for small betas because of the continuity needed by 
		   fitting routines. Should be |r|<1.
               t: time of data point
Output:        CFPTD01: value of CFPTD for inputs
Errors:        beta < 0.1 -> beta=0.1
               beta > 0.97 -> beta=0.97
------------------------------------------------------------------------*/
double CFPTD01 (double tmeff, double beta, double r, double t){

  if(t<=-r*tmeff) return 0.;

  if(beta>0.99) beta=0.99;
  else if(beta<0.1) beta=0.1;
	
  return cf(pow(t/tmeff+r,-beta),beta);
}

/*------------------------------------------------------------------------
Procedure:     SCD01
Purpose:       calculation of spatial profile (no-backflow approximation)
               L:    (positive!) distance from origin.
               kapa: correction parameter coming from r<>0;
	       drops with time like t^(beta-1).
Limitations:   R must be positive !!!
Errors:        beta < 0.1 -> beta=0.1
               beta > 0.97 -> beta=0.97
------------------------------------------------------------------------*/
double SCD01 (double R, double beta, double kapa, double L) {

  double tmp;
  if(R<=0.) return 0.; /* must be >0 */
  if(L<0.||(tmp=1+kapa*L/R)<=0.) return 0.;

  if(beta>0.99) beta=0.99;
  else if(beta<0.1) beta=0.1;
		
  return f(L/R/pow(tmp,beta),beta)/R*(tmp-beta*(tmp-1.))/pow(tmp,beta+1.);
}

/*------------------------------------------------------------------------
Procedure:     CSCD01
Purpose:       calculation of cumulative spatial profile
               (or SCD for step input; no-backflow approximation)
               L:    (positive!) distance from origin.
               kapa: correction parameter coming from r<>0;
	       drops with time like t^(beta-1).
Limitations:   R must be positive !!!
Errors:        beta < 0.1 -> beta=0.1
               beta > 0.97 -> beta=0.97
------------------------------------------------------------------------*/
double CSCD01 (double R, double beta, double kapa, double L) {

  double tmp;
  if(R<=0.) return 0.; /* must be >0 */
  if(L<=0.) return 1.;
  if((tmp=1+kapa*L/R)<=0.) return 0.;

  if(beta>0.99) beta=0.99;
  else if(beta<0.1) beta=0.1;
		
  return cf(L/R/pow(tmp,beta),beta);
}

/*========================================================================
                END OF PROCEDURES FOR 0 < beta < 1
==========================================================================
========================================================================*/

/*========================================================================
                   PROCEDURES FOR 1 < beta < 2
========================================================================*/

/* by b_beta we mean b_beta at distance L (where the FPTD is measured)!!!
Limitations:   t_mean and b_beta must be positive !!!
Errors:        beta < 1.02 -> beta=1.02
------------------------------------------------------------------------*/

double FPTD12(double t_mean,  double beta,  double r,  double t) {

  double dum1;

  if(beta>2.) beta=2.;
  else if(beta<1.01) beta=1.01;

  if((dum1=t_mean*r)<=0.) return 0.;

  return f((t_mean-t)/dum1,1./beta)/beta/dum1;
}

/*----------------------------------------------------------------------*/

double CFPTD12(double t_mean,  double beta,  double r,  double t) {

  double dum1;

  if(beta>2.) beta=2.;
  else if(beta<1.01) beta=1.01;

  if((dum1=t_mean*r)<=0.) return (double)(t>t_mean);

  return cf((t_mean-t)/dum1,1./beta)/beta;
}

/*----------------------------------------------------------------------*/

/* by definition for 1<beta<2, R=wt, with w the (asymptotic) velocity   */
/* negative R, kapa=B/R^(beta-1) and L are meaningless !!! */

double SCD12(double R, double beta, double kapa, double L) {

  double niu, A1, h;

  if(R<=0.||kapa<0.) return 0.; /* meaningless */
  if(kapa==0.) return (double)(L==R); /* should be a delta-function... */

  if(beta>2.) beta=2.;
  else if(beta<1.01) beta=1.01;

  if(L<0.) return 0.;
  if(L==0.) return (beta-1.)/MyGamma(2.-beta,0)*kapa/R;

  niu=1./beta;
  A1=pow(L*kapa*pow(R,beta-1.),-niu);  /* A1=1/A */
  h=(L-R)*A1;

  return f(h,niu)*niu*(A1-h*niu/L);
}

/*----------------------------------------------------------------------*/

double CSCD12(double R, double beta, double kapa, double L) {

  double niu;

  if(R<=0.||kapa<0.) return 0.; /* meaningless */
  if(kapa==0.) return (double)(L<R); /* a unit step. */
  if(L<=0.) return 1.;

  if(beta>2.) beta=2.;
  else if(beta<1.01) beta=1.01;

  niu=1./beta;

  return cf((L-R)/pow(L*kapa*pow(R,beta-1.),niu),niu)*niu;
}

/*----------------------------------------------------------------------*/

/*========================================================================
                END OF PROCEDURES FOR 1 < beta < 2

==========================================================================
                   SPECIAL FUNCTIONS f AND cf
--------------------------------------------------------------------------
Procedure:     f
Purpose:
Input:
Output:
Errors:
------------------------------------------------------------------------*/
double f(double x, double niu) {
  int j, BigNf; /* ideally, should be infinity. */
  int n, k=25;   /* relates to number of terms in asymptotic series  */
  double theta=0.8, alpha=0.55; /* weights of last terms in expansions  */
  double eps=0.2;  /* length of the linear interpolation interval (>0) */
  double dum1, dum2, dum3;

  if (x>xupper(niu)) {

    dum1 = (1.-niu)*pow(niu*x,1./(1.-niu)); /* the big parameter */
    dum2 = 1.+theta*(2.-niu)*(2.*niu-1.)/24./dum1;
    return pow(dum1,niu-.5)*exp(-dum1/niu)*dum2/sqrt(2.*PI)/pow(1.-niu,niu);
  }

  if (x<0.) {                /* this is necessary to be sure that we are
                               in the case 1<beta<2 and then niu=1/beta;
                               otherwise, niu=beta can be very small and
                               xlower large positive, which is wrong. */
    if (x<(dum2=xlower(niu))) {

      /* for beta close to 1 need to interpolate between exact and
         approximate solutions, since they don't converge good:
         I do a linear interpolation between the f values at points
         xlower=dum2 and xlower-eps=dum1 */
      if(niu>0.87) if(x-0.0001*eps > (dum1=dum2-eps)) {
	dum3=f(dum2,niu);
	return dum3+(f(dum1,niu)-dum3)*(dum2-x)/eps;
      }
      
      dum1=pow(-x,-1./niu);
      n=(int) floor(niu*(.5+k))+1.;
      for(j=1, dum2=0. ; j<n ;  j++) {
	dum2+=pow(dum1,j)*sin(PI*j/niu)*MyGamma(j/niu,0)/MyGamma(j,0);
      }
      dum2+=alpha*pow(dum1,n)*sin(PI*n/niu)*MyGamma(n/niu,0)/MyGamma(n,0);
      return dum2/PI/niu/niu/x;
    }
  }

  if(x==0.) return MyGamma(niu,0)*sin(PI*niu)/PI;

  BigNf=(int) floor(250.+3./(1.01-niu));
  for(j=1, dum3=0. ; j<=BigNf ; j++) {
    dum2=exp(MyGamma(j*niu,1)-MyGamma(j,1));
    dum3 += pow(-x,j-1)*dum2*sin(PI*j*niu);
  }

  return dum3/PI;
}



/*------------------------------------------------------------------------
Procedure:     cf
Purpose:       integral of f

                                 x             infinity 
                                 /                /
                                |                |
                cf(x;niu) = 1 - | f(x';niu)dx' = | f(x';niu)dx'
                                |                |
                               /                /
                                0                x

Input:
Output:
Errors:
------------------------------------------------------------------------*/
double cf(double x, double niu) {
  int j, n, k=25, BigNcf=190;
  double thetac=0.75, eps_cl=0.1, eps_cu=0.01, alphac=0.53;
  double dum1, dum2, dum3;

  if (x>(dum3=xupper(niu))) {

    if (niu<0.2) thetac=0.67;

    else if ( niu>0.87 && x+0.0001*eps_cu < (dum2=dum3+eps_cu) ) {
      dum1=cf(dum3,niu);
      return dum1+(cf(dum2,niu)-dum1)*(x-dum3)/eps_cu;
    }

    dum1 = (1.-niu)*pow(niu*x,1./(1.-niu)); /* the big parameter */
    dum2 = 1.-thetac*(2.*niu*niu+7.*niu+2.)/24./dum1;
    return exp(-dum1/niu)*dum2/sqrt(2.*PI*dum1);
  }

  if (x<0) {                /* this is necessary to be sure that we are
                               in the case 1<beta<2 and then niu=1/beta;
                               otherwise, niu=beta can be very small and
                               xlower large positive, which is wrong. */
    if (x<(dum3=xlower(niu))) {

      if ( niu>0.94 && x-0.0001*eps_cl > (dum2=dum3-eps_cl) ) {
	dum1=cf(dum3,niu);
	return dum1+(cf(dum2,niu)-dum1)*(dum3-x)/eps_cl;
      }	
      
      dum1=pow(-x,-1./niu);
      n=(int) floor(niu*(.5+k))+1.;
      for(j=1, dum2=PI ; j<n ;  j++) {
	dum2+=pow(dum1,j)*sin(PI*j/niu)*MyGamma(j/niu,0)/MyGamma(j+1,0);
      }
      dum2+=alphac*pow(dum1,n)*sin(PI*n/niu)*MyGamma(n/niu,0)/MyGamma(n+1,0);
      return dum2/PI/niu;
    }
  }

  for(j=1, dum3=0. ; j<=BigNcf ; j++) {
    dum2=exp(MyGamma(j*niu,1)-MyGamma(j+1,1));
    dum3 += pow(-x,j)*dum2*sin(PI*j*niu);
  }

  return 1.+dum3/PI;
}


/*-------------------------------------------------------------------------
Procedure:     xupper
Purpose:       calcualting the crossing point between using exact and 
               approximate forms of the functions f(x,niu), cf(x,niu) 
               for large positive x >= xupper.
-------------------------------------------------------------------------*/

double xupper(double niu) {
  double x;

  if (niu < 0.9)
    x=2.5+18.443*niu-40.041*niu*niu+21.971*pow(niu,3)-1.8145*pow(niu,4);
  else x=163.12 - 485.58*niu + 489.72*niu*niu - 166.26*niu*niu*niu ;
  
  return x;
}

/*-------------------------------------------------------------------------
Procedure:     xlower
Purpose:       calcualting the crossing point between using exact and 
               approximate forms of the functions f(x,niu), cf(x,niu) 
               for large negative x <= xlower < 0.
-------------------------------------------------------------------------*/

double xlower(double niu) {
  
  /* The function xlower must be only used in the case  1<beta<2, 
     in which niu=1/beta.  This "if" below shouldn't be necessary
     since the check for negative x (which should happen only for
     1<beta<2) is done inside the functions f and cf,  but we put
     it for the case somebody calls these functions with negative
     x while in the interval 0<beta<1. */

  if(niu<0.5) {
    printf("EXITING: xlower: niu is out of range.");
    exit(-1);
  }

  return (40.*fabs(1.3-1./niu)-110.)/pow(100.,niu)+0.02;
}

/*========================================================================
========================================================================*/
