% DISTR([beta C C1]) calculates desired CTRW distributions for
% given constants of motion.
% Temporal and/or spatial points are given by the corresponding
% fields of the structure 'data', which is declared here global.
% The result is an array containing values of (C)FPTDs and/or
% (C)SCDs in accordance with the data.type.

function res = DISTR(const);

beta=const(1);
C=const(2);
C1=const(3);

global data;

if C <= 0
    disp('ERROR: nonpositive C.');
    return
end

if C1 < 0
    disp('ERROR: negative C1.');
    return
end

T=[data.time];
if min(T) < 0
    disp('ERROR: there are negative times from starting the experiment.');
    return
end

D=[data.distance];
if min(D) < 0
    disp('ERROR: there are negative distances from the inlet.');
    return
end

%%   calculations depending on value of beta:

if beta < 1     %   0 < beta < 1
    if beta > 0.99
        beta=0.99;
    elseif beta<0.1
        beta=0.1;
    end
    niu = beta;
    
    x = D*C./(T+D*C1).^beta;
    
    wc = 1;   % goes in front of function cf.
    wfptd = niu*x./(T+D*C1); % goes in front of function f for FPTDs. 
    wscd = (C*T+(1-niu)*D*C*C1)./(T+D*C1).^(niu+1); % in front of f for SCDs.
   
else            %   1 < beta < 2
    if beta > 2
        beta=2;
    elseif beta < 1.01
        beta=1.01;
    end
    niu = 1/beta;
    
    x = (D*C1-T)./(D*C).^niu;
    
    wc = niu;
    wfptd = niu./(D*C).^niu;
    wscd = (niu*T+(1-niu)*D*C1)*niu/C^niu./D.^(niu+1);
    
end

ind1=find(strcmp({data.type},'FPTD'));
ind2=find(strcmp({data.type},'SCD'));
ind3=find(strcmp({data.type},'CFPTD'));
ind3=[ind3, find(strcmp({data.type},'CSCD'))]; % CFPTD and CSCD are the same function of x.

if ~(length(ind1)+length(ind2)+length(ind3) == length(data))
    disp('ERROR: DISTR: have unknown data types.');
    return
end

res=zeros(size(data));  % just allocate memory.

if ~isempty(ind1)
    res(ind1) = T(ind1).*wfptd(ind1).*f(x(ind1),niu); % return T*FPTD.
    % Multiply by T to make returned values be close to 1, for equal 
    % weighting of different curves in the least squares fitting.
    % The concentration data points should be modified accordingly.
end

if ~isempty(ind2)
    res(ind2) = D(ind2).*wscd(ind2).*f(x(ind2),niu); % return D*SCD.
    % Multiply by t to make returned values be close to 1, for equal 
    % weighting of different curves in the least squares fitting.
    % The concentration data points should be modified accordingly.
end

if ~isempty(ind3)
    res(ind3) = wc*cf(x(ind3),niu);
end

% the end.