% bestguess = ctrw_multifit(guess, guesstype, DT, lb, ub, maxtrials)
% Fitting with constant beta distributions.
% 0.1 < beta <=2, beta ~= (0.99, 1.01)
% This program allows to determine the best CTRW fitting parameters
% to multiple curves (data) which can be of different type. Currently,
% it is possible to combine (C)FPTDs at different distances and (C)SCDs
% at different times together. (It is also possible to add MEANs and
% ST_Ds as additional types in a future.)
% For the fit to have a meaning all the curves should be measured for
% the same setup (the same flow system). Also, one-dimensional average
% flow and no-backflow is assumed.
% For the data curves, time should be measured from the start of injection
% (for (C)FPTDs) and distance should be measured from the injection point
% in the flow direction (for (C)SCDs). The injection is assumed to be 
% pointlike (in the one-dimensional model of the system).
% All the data curves should be put by user into the structure named "data",
% whose first field, 'concentration', contains either values of (C)FPTD
% (measured at a distance 'distance' from the inlet) as a function of time
% (field 'time'), or values of (C)SCD (measured at a time 'time' from the
% start of injection) as a function of distance (field 'distance').
% To distinguish between different types of curves, there is a field 'type',
% which currently accepts four values: 'FPTD', 'CFPTD', 'SCD' and 'CSCD'.
% Thus, each element of the structure array data contains a measurement
% specifying its value, type and time and location of the measurement.
% This data input should be done before calling this function. The data
% structure should be declared global before it is filled
% (it's better to make it global rather then pass it as an
% argument to functions which will be called many times):
%
% global data; 
% data=struct('concentration',{},'time',{},'distance',{},'type',{}); 
%
% The second line is needed to set matlab variable type of data to struct;
% Without this the following example doesn't work (at least in the matlabR12).
%
% EXAMLPE: adding FPTD curve to the data.
% Write the times into t, FPTD into y, provide the distance of the FPTD
% measurement, L, and set the curve type: type='FPTD'. Then issue
% data(datalength+1:datalength+length(y))=struct('concentration',num2cell(y),...
%        'time',num2cell(t),'distance',num2cell(L),'type',type);
% datalength=length(data);
%
% Note that the measurement units (absolute values) of times (and distances)
% are unimportant; what is important are their ratios.
%
% It's convenient to look for the best-fitting parameters relating to one of
% the curves. This choice should be done by user. All the parameters to
% other curves are uniquely determined from the chosen set of parameters.
% To get the two constants of motion (in addition to beta) one can use 
% function consts.
%
% %  EXAMPLE of defining parameters for this function (ctrw_multifit):
% %  Making initial guess. The convenient way to make a guess and also the
% %  limiting boundaries of it is to 
% %
% %  1. Choose whether it be a guess for (C)FPTD or (C)SCD. Set guesstype to
% %  to 'FPTD', 'CFPTD', 'SCD' or 'CSCD':
% guesstype = 'FPTD'
% 
% %  2. Specify distance (of (C)FPTD) or time (of (C)SCD) for which the guess is made:
% DT = 10
% 
% %  3. Provide initial guess;
% %  If guesstype = 'FPTD' or guesstype = 'FPTD', guess = [Tm(eff) beta r].
% %  If guesstype = 'SCD' or guesstype = 'CSCD', guess = [R beta kappa].
% guess = [5 1.3 .1]
% 
% %  4. Provide lower and upper boundaries (lb and ub) of the guesses.
% %  Note that the widest possible interval for beta is 0.1 < beta <= 2.
% lb=[.1 .2 0];             % lower bounds of guesses.
% ub=[100 2 1];             % upper bounds.
% 
% %  Finally, set maximal number of trials with rundom guesses (inside the bounds):
% maxtrials=3;
% %  and maximal relative error (to stop before making maxtrials:
% maxerr=1e-3;
%%%

function bestguess = ctrw_multifit(guess, guesstype, DT, lb, ub, maxtrials, maxerr);
% help lsqcurvefit  % sometimes need for first run to set path to lsqcurvefit.
global data; 

trial=0; % must start from zero for the "if" below.
besttrial=trial;
bestguess=guess;
relres=1;
bestrho=0;
opts=optimset(optimset('lsqcurvefit'),'MaxFunEvals',1000*length(guess)...
    ,'MaxIter',4000,'TolFun',1e-14,'Display','off');

% making concentration data to fit:
ind1=find(strcmp({data.type},'FPTD'));
ind2=find(strcmp({data.type},'SCD'));
ind3=find(strcmp({data.type},'CFPTD'));
ind3=[ind3, find(strcmp({data.type},'CSCD'))]; % CFPTD and CSCD are the same function of x.
if ~(length(ind1)+length(ind2)+length(ind3) == length(data))
    disp('ERROR: ctrw_multifit: have unknown data types.');
    return
end
y(ind1)=[data(ind1).time].*[data(ind1).concentration];
y(ind2)=[data(ind2).distance].*[data(ind2).concentration];
y(ind3)=[data(ind3).concentration];

% fitting
fid = fopen('multifit.out','w');
while (trial < maxtrials) & (relres > maxerr)
    
    if trial
        lb(1)=log10(lb(1));
        ub(1)=log10(ub(1));
        guess=lb+(ub-lb).*rand(size(guess));
        guess(1)=10^guess(1);
        lb(1)=10^lb(1);
        ub(1)=10^ub(1);
    end 
    
    [guess,resnorm] = lsqcurvefit('lsqDISTR',guess,1,y,lb,ub,opts,guesstype,DT);
    guess=params(consts(guess,guesstype,DT),guesstype,DT); % move beta out of (0.99, 1.01).
    
    rr=resnorm/sum(y.^2);
    
    % calculating correlation coeff.
    ydistr=DISTR(consts(guess, guesstype, DT));
    rho=((y-mean(y))*(ydistr-mean(ydistr))')/norm(y-mean(y))/norm(ydistr-mean(ydistr));
    
    trial=trial+1;
    disp([num2str(trial),': ',num2str(guess(1)),'   ',num2str(guess(2)),'   ',num2str(guess(3))...
            ,'; rel. res. =  ',num2str(rr),', corr. = ',num2str(rho)]);
    fprintf(fid,'%d:\t %7g\t %7g\t %7g;\t rel. res. = %f,\t corr. = %f\n',trial,guess,rr,rho);
    
    if relres > rr
        relres=rr;
        bestrho=rho;
        besttrial=trial;
        bestguess=guess;
    end
    
end
fclose(fid);

disp([num2str(trial),' trials. The best is #',num2str(besttrial),' and the best guess is']);
disp([' ',num2str(bestguess(1)),'   ',num2str(bestguess(2)),'   '...
        ,num2str(bestguess(3))]);
disp(['Its relative norm of residue is ',num2str(relres),' and correlation coefficient is ',num2str(bestrho)]); 

% the end.
