function y = L_conc(u_vec,varpar,options)
% ------------------------------------------------------------
% L_CONC
% Solution for the Laplace transformed total flux averaged
% (or resident concentration) for a 1D statiionary domain, 
% at a given position 'x'
% with constant velocity ( v ) and dispersivity ( D/v ) .
%
% Y = L_CONC(U_VEC,VARPAR,OPTIONS);
%
% u_vec  := vector of Laplace variables
% varpar := vector containing the velocity,
%           the dispersion and the psi parameters
% options:= cell array containing information on the model,
%           the outlet boundary condition, the inlet input,
%           the kind of concentration, and the section.
% y      := vector of Laplace transformed concentrations
%
% Example:
% y = L_conc([1+i 1+0.1*i],[1 0.05 [0.75 log10(1) ...
%             log10(100)]],{'TPL','N','step','c_f_norm',1,1,'R'})
%
% See also t_conc
% ------------------------------------------------------------

% This file is part of the CTRW toolbox v4.0
% For the usage of this file see Legal_notice.m
% Author   : Andrea Cortis
% $Original version : 26-Jun-2004$
% Modified by: Simon Emmanuel
% $Modified version : 27-Nov-2006$
% Modified by: Shira Rubin
% $Modified version : May-2010$
% Modified by: Rami Ben-Zvi
% $Modified version : 09-Sep-2015$
% Modified by: Alon Nissan
% $Modified version : 20-March-2017$
% e-mail   : brian.berkowitz@weizmann.ac.il
%           (see http://weizmann.ac.il/ESER/people/Brian/CTRW)
% ------------------------------------------------------------

v         = varpar(1);
D         = varpar(2);
param     = varpar(3:length(varpar));
str_bou   = options{2};
str_inlet = options{3};
str_conc  = options{4};
x         = options{5};
inlet_BC  = options{7};

y = zeros(size(u_vec));

if v<0 || D<0
  fprintf('ERROR: v = % 14.7e or D = %14.7e are negative \n', v, D);
  error('STOP')
end

if ~strcmp(str_conc, 'c_r_norm') && ~strcmp(str_conc, 'c_res') && ...
   ~strcmp(str_conc, 'c_f_norm') && ~strcmp(str_conc, 'c_flux')
  fprintf(' Variable str_conc is: %s \n',str_conc);
  fprintf('ERROR: valid options for variable "str_conc" are: \n')
  fprintf('       "res" : to calculate the resident      concentration  \n')
  fprintf('       "flux": to calculate the flux averaged concentration  \n')
  error('STOP')
end

for j=1:length(u_vec)
    u=u_vec(j);
    M = mem(u,param,options);

    switch str_inlet
	case {'step'}
	    y0 = 1/u;			% Response to a step   input
	case {'pulse'}
	    y0 = 1;			% Response to a Dirac  input
	case {'square'}
	    y0 = 1/u*(1-exp(-u));	% Response to a square input of duration t = 1
	otherwise
	    y0 = eval(str_inlet);	% Response to a user defined input
    end

    a = D/v;
    w = (1+4*u*a/M/v)^(1/2);
    z = (1+w)*x/2/a;
    k = (2*w-x*w+x)/2/a;
    p = exp(w/a);

    switch inlet_BC
    % ======================================================================
        case {'R'}
            switch str_bou
    % ----------------------------------------------------------------------
                case {'N'}	% Robin INLET, Neumann OUTLET
                switch str_conc
                    case {'c_res','c_r_norm'}		% Program OUTPUT: c_r
                        num = 2 * ( (w+1)*exp(k) + (w-1)*exp(z) ) / M;
                    case {'c_flux','c_f_norm'}		% Program OUTPUT: c_f
                        num = ( (w+1)^2*exp(k) - (w-1)^2*exp(z) ) / M;
                end
                A = (w+1)^2;
                B =-(w-1)^2;
    % ----------------------------------------------------------------------
                case {'D'}	% Robin INLET, Dirichlet OUTLET
                switch str_conc
                    case {'c_res','c_r_norm'}		% Program OUTPUT: c_r
                        num = 2 * ( exp(k) - exp(z) ) / M;
                    case {'c_flux','c_f_norm'}		% Program OUTPUT: c_f
                        num = ( (w+1)*exp(k) + (w-1)*exp(z) ) / M;
                end
                A = w+1;
                B = w-1;
            end
    % ======================================================================
        case {'D'}
            switch str_bou
    % ----------------------------------------------------------------------
                case {'N'}	% Dirichlet INLET, Neumann OUTLET
                switch str_conc
                    case {'c_res','c_r_norm'}		% Program OUTPUT: c_r
                        num = exp(k)*((w+1)+(w-1)*exp((x-1)*w/a));     
                    case {'c_flux','c_f_norm'}		% Program OUTPUT: c_f
                        num = exp(k)/2*((w+1)^2-(w-1)^2*exp((x-1)*w/a));                                                      
                end
                A = w+1;                    
                B = w-1;                    
    % ----------------------------------------------------------------------
            case {'D'}	% Dirichlet INLET, Dirichlet OUTLET
                switch str_conc
                    case {'c_res','c_r_norm'}		% Program OUTPUT: c_r
                        num = exp(k)*(1-exp(w*(x-1)/a));
                    case {'c_flux','c_f_norm'}		% Program OUTPUT: c_f
                        num = exp(k)/2*(w+1+(w-1)*exp(w*(x-1)/a));
                end    
                A = 1;
                B =-1;
            end            
    end

    if isinf(abs(p));
        den = A;
    else
        den = p*A+B;
    end
    y(j) = y0*num/den;
end;	% j
