function y=L_resid_2D(u_vec,varpar,options)
% ------------------------------------------------------------
% L_RESID_2D
% Solution for the Laplace transformed resident concentration
% for a 2D stationary domain, at a given position 'x1,x2'
% with constant velocity ( v ) and dispersivity ( D/v ) .
%
% Y = L_RESID_2D(U_VEC,VARPAR,OPTIONS)
%
% u_vec  := vector of Laplace variables
% varpar := vector containing the velocity, 
%           the dispersion and the psi parameters
% options:= cell array containing information on the model,
%           the outlet boundary condition, the inlet input,
%           the kind of concentration, and coordinates of 
%           the point at which the BTC is taken.
% y      := vector of Laplace transformed concentrations
% 
% Example
% y = L_resid_2D([1+i 1-i],[1 0.05 0.005 [0.75 log(1) ...
%                 log(100)]],{'TPL',0.2, 0.1,1})
%
% See also t_resid_2D
% ------------------------------------------------------------

% This file is part of the CTRW toolbox v4.0
% For the usage of this file see Legal_notice.m
% Author   : Andrea Cortis
% $Original version : 26-Jun-2004$
% Modified by: Simon Emmanuel
% $Modified version : 27-Nov-2006$
% Modified by: Rami Ben-Zvi
% $Modified version : 23-Mar-2016$
% Modified by: Alon Nissan
% $Modified version : 20-March-2017$
% e-mail   : brian.berkowitz@weizmann.ac.il
%           (see http://weizmann.ac.il/ESER/people/Brian/CTRW)

% ------------------------------------------------------------

v       = varpar(1);
DL      = varpar(2);
DT      = varpar(3);
param   = varpar(4:length(varpar));

str_mem   = options(1);
x1        = options{2};
x2        = options{3};

xbar   = v;
x2Lbar = 2*DL;
x2Tbar = 2*DT;

y = zeros(size(u_vec));
for j=1:length(u_vec)
    u=u_vec(j);    
    M = mem(u,param,str_mem);
    y0 = M*v              ; %%% Response to a Dirac  input  
    z  = xbar/x2Lbar*sqrt(x1^2+x2Lbar/x2Tbar*x2^2)*sqrt(1+2*u*x2Lbar./(M*xbar^2));   
    if real(z)<500
        yy   = 1./(M.*pi.*sqrt(x2Lbar*x2Tbar)).*exp(x1.*xbar./x2Lbar).*besselk(0,z)/M/v;
    else
        w = x1.*xbar./x2Lbar;
        y   = 1./(M.*pi.*sqrt(x2Lbar*x2Tbar)).*exp(w-z)/M/v;
        cc = sqrt(pi/2.0./z);
        yy = y.*cc;
    end    
    y(j) = y0*yy;    
end
