function f = ilap(F, t, P)
% ------------------------------------------------------------
% ILAP
% F :=  laplace-space function (string refering to an m-file), 
%           must have form F(s,P), where s is the 
%           Laplace parameter, and return column vector as result
% t :=  column vector of times 
% P :=   optional parameters to be passed on to F
% f :=  vector of real-space values f(t)
% 
% F = ILAP(F, t, P);
%
% Example
% f = ilap('L_psi' ,transpose(logspace(-1,1,10)), ...
%                      {[0.75 1 0.1] , {'ASY'}})
% Reference: 
% de Hoog, F. R., Knight, J. H., and Stokes, A. N. (1982). 
% An improved method for numerical inversion of Laplace transforms. 
% S.I.A.M. J. Sci. and Stat. Comput., 3, 357-366.
% ------------------------------------------------------------

% This file is part of the CTRW toolbox v4.0
% For the usage of this file see Legal_notice.m
% Author   : Andrea Cortis
% $Original version : 26-Jun-2004$
% e-mail   : brian.berkowitz@weizmann.ac.il
%           (see http://weizmann.ac.il/ESER/people/Brian/CTRW)
% ------------------------------------------------------------

alpha = 0         ; % largest pole of F 
tol   = 1e-9      ; % numerical tolerance of approaching pole
f     = []        ;
M     = 10        ;
run   = [0:1:2*M]';  

fprintf('ilap: Work in progress \n');
allt = t;
logallt = log10(allt);
iminlogallt = floor(min(logallt));
imaxlogallt = ceil(max(logallt));
for ilogt = iminlogallt:imaxlogallt, 

  t = allt(find((logallt>=ilogt) & (logallt<(ilogt+1))));
  if ~isempty(t), 

    T = max(t)*2;
    gamma = alpha-log(tol)/(2*T);

    nt = length(t);
    
    s_vec = gamma + i*pi*run/T;
    a     = zeros(size(s_vec));   
    for k=1:length(s_vec)
         fprintf('.')
        s=s_vec(k);
        c = ['a(k) = ' F '(s'];

        if nargin > 2,   
            for iarg = 1:length(P),
                c = [c ',P{' int2str(iarg),'}'];
            end
        end
        c = [c ');'];
        eval(c);
    end
    fprintf('\n')
    a(1) = a(1)/2;    
    
    e = zeros(2*M+1, M+1);
    q = zeros(2*M  , M+1);     
    e(:,1) = zeros(2*M+1,1);
    q(:,2) = a(2:2*M+1,1)./a(1:2*M,1);
    for r = 2:M+1,   
      e(1:2*(M-r+1)+1,r) = ...
      q(2:2*(M-r+1)+2,r) - q(1:2*(M-r+1)+1,r) + e(2:2*(M-r+1)+2,r-1);
      if r<M+1,    
    rq = r+1;
    q(1:2*(M-rq+1)+2,rq) = ...
     q(2:2*(M-rq+1)+3,rq-1).*e(2:2*(M-rq+1)+3,rq-1)./e(1:2*(M-rq+1)+2,rq-1);
      end
    end

    d = zeros(2*M+1,1);
    d(1,1) = a(1,1);
    d(2:2:2*M,1) = -q(1,2:M+1).'; 
    d(3:2:2*M+1,1) = -e(1,2:M+1).'; 

    A = zeros(2*M+2,nt);
    B = A;
    A(2,:) = d(1,1)*ones(1,nt);
    B(1:2,:) = ones(2,nt);
    z = exp(i*pi*t'/T);   
    
    for n = 3:2*M+2,
      A(n,:) = A(n-1,:) + d(n-1,1)*ones(1,nt).*z.*A(n-2,:);
      B(n,:) = B(n-1,:) + d(n-1,1)*ones(1,nt).*z.*B(n-2,:);
    end

    h2M = .5 * ( ones(1,nt) + ( d(2*M,1)-d(2*M+1,1) )*ones(1,nt).*z );
    R2Mz = -h2M.*(ones(1,nt) - ...
    (ones(1,nt)+d(2*M+1,1)*ones(1,nt).*z/(h2M).^2).^.5);
    A(2*M+2,:) = A(2*M+1,:) + R2Mz .* A(2*M,:);
    B(2*M+2,:) = B(2*M+1,:) + R2Mz .* B(2*M,:);

    fpiece = ( 1/T * exp(gamma*t') .* real(A(2*M+2,:)./B(2*M+2,:)) )';
    f = [f; fpiece]; 

  end 
  
end 
