%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Example of error minimization code for a single breakthrough curve
% using the TPL model. All five parameters are varied during the
% minimization process.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% UNITS IN THE INPUT FILE AND FOR ALL INPUT PARAMETERS
% MUST BE CONSISTENT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% This file is part of the CTRW toolbox v4.0
% For the usage of this file see Legal_notice.m
% Author   : Andrea Cortis
% $Original version : 26-Jun-2004$
% Modified by: Simon Emmanuel
% $Modified version : 27-Nov-2006$
% Modified by: Shira Rubin
% $Modified version : May-2010$
% Modified by: Rami Ben-Zvi
% $Modified version : 01-Sep-2015$
% Modified by: Alon Nissan
% $Modified version : 20-March-2017$
% e-mail   : brian.berkowitz@weizmann.ac.il
%           (see http://weizmann.ac.il/EPS/people/Brian/CTRW)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clc
clear all; close all; fclose all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 1
% Choice of model: ADE, TPL, ETA
% (see 'options{1}' in the user guide)
mod_type ='TPL';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 2
% Initial parameter guess
% Input of v and D must be in units of 1/TIME: i.e., divide the dimensional
% v and D by L (column length) and L^2, respectively.
v    =  0.01 ;                              % v_psi [T^-1]
D    = 2.2e-005 ;                           % D_psi [T^-1]
beta = 1.8;                                 % beta
t2   = 10^6;                                % cut-off time t2

%CHOOSE HERE: t1 specified or t1 calculated automatically
t1          = 10^-1.9;                        % t1 as an independent variable (specified)
% [t1, t2] = t1_calc(t2,v,D,beta);            % calculate t1 

p = [beta, log10(t1), log10(t2)];           % parameters must be consistent with model in Field 1

% NOTE: Do not forget to reconvert to real dimensional units of v and D;
%   multiply  v  and  D  by L (column length) and L^2, respectively.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 3
% Relative location (normalized) of measurement along the column
col_pos = 1;
% measurement position, should be in the range [0,1]
% (see 'options{5}' in the user guide)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 4
% Boundary condition at the outlet
% String may be 'N' (Neumann) or 'D'  (Dirichlet).
% (see 'options{2}' in the user guide)
outlet_BC = 'N';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 5
% Boundary condition at the inlet
% String may be 'R' (Robin) or 'D' (Dirichlet).
% (see 'options{7}' in the user guide)
inlet_BC = 'R';
% Time dependence f(t) at the inlet.
% String options include 'step', 'pulse', or 'square'.
% Or user defined functional dependence (in Laplace space).
% (see 'options{3}' in the user guide)
inlet_ft = 'step';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 6
% Solve for:
% - either the resident  concentration
% - or the flux-averaged concentration
% String may be 'c_f_norm' (flux-averaged) or 'c_r_norm' (resident).
% (see 'options{4}' in the user guide)
out_type = 'c_f_norm';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 7
% Name of input data file; The .m suffix must be omitted
data_1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 8
% Name of output data file
output = 'output.txt' ;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 9
% Generate the result for the first guess only [0]
% or initiate minimization [1]
minimize = 1;
optimization_type = 0; % [0]  Unconstrained Optimization [1] Constrained Optimization

% Note that use of the Constrained Optimization function type ('fminsearchbnd.m')
% requires lower and upper bounds (see lines 159-160, parameters LB, UB)
% Note also that to constrain a parameter with a constant value (that is not to be 
% optimized, simply multiply the parameter value by 1.0 in each of LB and UB. 


%__________________________________________________________________________
% MAIN SCRIPT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DO NOT CHANGE UNLESS YOU INTEND TO ALTER CODE

% Interpolation of data file
xa = a(:,1);
ya = a(:,2);
points = 1000;
minxa =  min(xa);
maxxa = max(xa);
minxa =  minxa+ (maxxa -minxa)/1000;
maxxa = maxxa- (maxxa -minxa)/1000;
t0 = transpose(linspace(minxa,maxxa,points));
data(:,1) = t0;
data(:,2) = interp1(xa,ya,t0);

% Calculation of additional parameters

inlet_f0 = 1;
% Note: inlet_f0 should NOT be modified in Toolbox version 3.1;
%    scaling factor f0 at the inlet (see 'options{6}' in the user guide.
col_length = 1;   % column length
pos = col_pos/col_length;
vDp_old = [v D p] ;
vDp = vDp_old;
vDp(1) = vDp_old(1)/col_length;
vDp(2) = vDp_old(2)/col_length^2;
options={ mod_type , outlet_BC , inlet_ft, out_type , pos, inlet_f0, inlet_BC};


% Calculation of normalized error
[err_old,  model] = diff_norm(vDp,{data,'t_conc',options});
err = err_old;

% Minimization algorithm
if minimize==1
    
    if optimization_type == 0
        OPTS = optimset('TolFun',1e-6,'TolX',1e-6,'display','iter');
        [vDp, FVAL , EXITFLAG] = my_fminsearch('diff_norm', vDp , OPTS ,{data,'t_conc',options});
        [err ,model] = diff_norm(vDp,{data,'t_conc',options});
        fprintf(1,' err = %6.4e \n',err);
        
        
    elseif  (optimization_type == 1)
        LB = [vDp(1)*0.1, vDp(2)*0.1, vDp(3)*0.1, vDp(4)*0.1, vDp(5)*0.1]; % values lower bounds
        UB = [vDp(1)*10,  vDp(2)*10,  vDp(3)*10,  vDp(4)*10,  vDp(5)*10];  % values lower bounds
        
        OPTS = optimset('TolFun',1e-6,'TolX',1e-6,'MaxIter',200,'display','iter');
        [vDp, fval] = fminsearchbnd('diff_norm', vDp ,LB,UB, OPTS ,{data,'t_conc',options});
        [err, model] = diff_norm(vDp,{data,'t_conc',options});
        fprintf(1,' err = %6.4e \n',err);
    end
   
    
end

vDp(1) = vDp(1)*col_length;
vDp(2) = vDp(2)*col_length^2;

% Output file
gen_output(vDp_old, vDp, err_old, err, mod_type, output)

% Graphics
figure(1)
plot(data(:,1),model, 'r', xa,ya,'b.')
hold on
xlabel('time');
if strcmp(out_type,'c_r_norm')
    ylabel('normalized c_r');
elseif strcmp(out_type,'c_f_norm')
    ylabel('normalized c_f');
end

if strcmp(out_type,'c_r_norm') || strcmp(out_type,'c_f_norm')
    figure(2)
    loglog(data(:,1),1-model,'r', xa,1-ya,'b.')
    title('On a Logarithmic Scale')
    xlabel('time');
    if strcmp(out_type,'c_r_norm')
    elseif strcmp(out_type,'')
        ylabel('1-normalized c_r');
    elseif strcmp(out_type,'c_f_norm')
        ylabel('1-normalized c_f');
    end
end
