%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Example of error minimization code for a single breakthrough curve
% using the TPL model. All five parameters are varied during the
% minimization process.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% UNITS IN THE INPUT FILE AND FOR ALL INPUT PARAMETERS
% MUST BE CONSISTENT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% This file is part of the CTRW toolbox v4.0
% For the usage of this file see Legal_notice.m
% Author   : Andrea Cortis
% $Original version : 26-Jun-2004$
% Modified by: Simon Emmanuel
% $Modified version : 27-Nov-2006$
% Modified by: Shira Rubin
% $Modified version : May-2010$
% Modified by: Rami Ben-Zvi
% $Modified version : 01-Sep-2015$
% Modified by: Alon Nissan
% $Modified version : 20-March-2017$
% e-mail   : brian.berkowitz@weizmann.ac.il
%           (see http://weizmann.ac.il/EPS/people/Brian/CTRW)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clc
clear all; close all; fclose all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 1
% Choice of model: ADE, TPL, ETA
% (see 'options{1}' in the user guide)
mod_type1 ='TPL';
mod_type2 ='ADE';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 2
% Initial parameter guess
% Input of v and D must be in units of 1/TIME: i.e., divide the dimensional
% v and D by L (column length) and L^2, respectively.
v =  0.0067 ; % v_psi  [T^-1]
D = 1.1e-4 ;% D_psi  [T^-1]
beta = 1.7;                                 % beta
t2   = 10^10;                                % cut-off time t2

%CHOOSE HERE: t1 specified or t1 calculated automatically
t1   = 10^-3;                             % t1 as an independent variable (specified)
% [t1, t2]   = t1_calc(t2,v,D,beta);                % calculate t1
p1= [beta, log10(t1), log10(t2)];   % parameters must be consistent with mod_type1 in Field 1
p2= [];                              % parameters must be consistent with mod_type2 in Field 1

% NOTE: Do not forget to reconvert to real dimensional units of v and D;
%   multiply  v  and  D  by L (column length) and L^2, respectively.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 3
% Relative location (normalized) of measurement along the column
col_pos = 1;
% measurement position, should be in the range [0,1]
% (see 'options{5}' in the user guide)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 4
% Boundary condition at the outlet
% String may be 'N' (Neumann) or 'D'  (Dirichlet).
% (see 'options{2}' in the user guide)
outlet_BC = 'N';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 5
% Boundary condition at the inlet
% String may be 'R' (Robin) or 'D' (Dirichlet).
% (see 'options{7}' in the user guide)
inlet_BC = 'R';
% Time dependence f(t) at the inlet.
% String options include 'step', 'pulse', or 'square'.
% Or user defined functional dependence (in Laplace space).
% (see 'options{3}' in the user guide)
inlet_ft = 'step';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 6
% Solve for:
% - either the resident  concentration
% - or the flux-averaged concentration
% String may be 'c_f_norm' (flux-averaged) or 'c_r_norm' (resident).
% (see 'options{4}' in the user guide)
out_type = 'c_f_norm';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 7
% Name of input data file; The .m suffix must be omitted
data_1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 8
% Name of output data file
output = 'output.txt' ;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 9
% Generate the result for the first guess only [0]
% or initiate minimization [1]
minimize = 0;
optimization_type = 0; % [0]  Unconstrained Optimization [1] Constrained Optimization

% Note that use of the Constrained Optimization function type ('fminsearchbnd.m')
% requires lower and upper bounds (see lines 178-179 ane lines 185-186, parameters LB, UB)
% Note also that to constrain a parameter with a constant value (that is not to be
% optimized, simply multiply the parameter value by 1.0 in each of LB and UB.


%__________________________________________________________________________
% MAIN SCRIPT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DO NOT CHANGE UNLESS YOU INTEND TO ALTER CODE

% Interpolation of data file
xa = a(:,1);
ya = a(:,2);
points = 1000;
minxa =  min(xa);
maxxa = max(xa);
minxa =  minxa+ (maxxa -minxa)/1000;
maxxa = maxxa- (maxxa -minxa)/1000;
t0 = transpose(linspace(minxa,maxxa,points));
data(:,1) = t0;
data(:,2) = interp1(xa,ya,t0);

% Calculation of additional parameters

inlet_f0 = 1;
% Note: inlet_f0 should NOT be modified in Toolbox version 3.1;
%    scaling factor f0 at the inlet (see 'options{6}' in the user guide.
col_length = 1;   % column length
pos = col_pos/col_length;
vDp1_old = [v D p1] ;
vDp1 = vDp1_old;
vDp1(1) = vDp1_old(1)/col_length;
vDp1(2) = vDp1_old(2)/col_length^2;

vDp2_old = [v D] ;
vDp2 = vDp2_old;
vDp2(1) = vDp2_old(1)/col_length;
vDp2(2) = vDp2_old(2)/col_length^2;

options1 = { mod_type1 , outlet_BC , inlet_ft, out_type , pos, inlet_f0, inlet_BC};
options2 = { mod_type2 , outlet_BC , inlet_ft, out_type , pos, inlet_f0, inlet_BC};


% Calculation of normalized error
[err1_old, model1] = diff_norm(vDp1,{data,'t_conc',options1});
err1 = err1_old;

[err2_old, model2] = diff_norm(vDp2,{data,'t_conc',options2});
err2 = err2_old;

% Minimization algorithm
if minimize==1
    
    if optimization_type == 0
        
        OPTS1 = optimset('TolFun',1e-6,'TolX',1e-6,'display','iter');
        [vDp1, FVAL1 , EXITFLAG1] = my_fminsearch('diff_norm', vDp1 , OPTS1 ,{data,'t_conc',options1});
        [err1, model1] = diff_norm(vDp1,{data,'t_conc',options1});
        fprintf(1,' err = %6.4e \n',err1);
        
        OPTS2 = optimset('TolFun',1e-6,'TolX',1e-6,'display','iter');
        [vDp2, FVAL2 , EXITFLAG2] = my_fminsearch('diff_norm', vDp2 , OPTS2 ,{data,'t_conc',options2});
        [err2, model2] = diff_norm(vDp2,{data,'t_conc',options2});
        fprintf(1,' err = %6.4e \n',err2);
        
        
        
    elseif (optimization_type == 1)
        
        LB1 = [vDp1(1)*0.1, vDp1(2)*0.1, vDp1(3)*0.1, vDp1(4)*0.1, vDp1(5)*0.1]; % values lower bounds
        UB1 = [vDp1(1)*10,  vDp1(2)*10,  vDp1(3)*10,  vDp1(4)*10,  vDp1(5)*10];  % values lower bounds
        OPTS1 = optimset('TolFun',1e-6,'TolX',1e-6,'MaxIter',200,'display','iter');
        [vDp1, fval1] = fminsearchbnd('diff_norm', vDp1 ,LB1,UB1, OPTS1 ,{data,'t_conc',options1});
        [err1, model1] = diff_norm(vDp1,{data,'t_conc',options1});
        fprintf(1,' err = %6.4e \n',err1);
        
        LB2 = [vDp2(1)*0.01, vDp2(2)*0.01]; % values lower bounds
        UB2 = [vDp2(1)*100,  vDp2(2)*100];  % values lower bounds
        OPTS2 = optimset('TolFun',1e-6,'TolX',1e-6,'MaxIter',200,'display','iter');
        [vDp2, fval2] = fminsearchbnd('diff_norm', vDp2 ,LB2,UB2, OPTS2 ,{data,'t_conc',options2});
        [err2, model2] = diff_norm(vDp2,{data,'t_conc',options2});
        fprintf(1,' err = %6.4e \n',err2);
        
        
    end
    
end



vDp1(1) = vDp1(1)*col_length;
vDp1(2) = vDp1(2)*col_length^2;
vDp2(1) = vDp2(1)*col_length;
vDp2(2) = vDp2(2)*col_length^2;

% Output file
gen_output(vDp1_old, vDp1, err1_old, err1, mod_type1, output)
fprintf(1,' err1 = %6.4e \n',err1);
fprintf(1,' err2 = %6.4e \n',err2);

% Graphics
figure(1)
plot(data(:,1),model1, 'r', data(:,1), model2, 'black', xa,ya,'b.')
legend(mod_type1, mod_type2, 'data','Location','SouthEast')
xlabel('time');
if strcmp(out_type,'c_r_norm')
    ylabel('normalized c_r');
elseif strcmp(out_type,'c_f_norm')
    ylabel('normalized c_f');
end

if strcmp(out_type,'c_r_norm') || strcmp(out_type,'c_f_norm')
    figure(2)
    loglog(data(:,1),(1-model1),'r', data(:,1),(1- model2), 'black',xa,1-ya,'b.');
    legend(mod_type1, mod_type2, 'data','Location','SouthEast');
    title('On a Logarithmic Scale');
    xlabel('time');
    if strcmp(out_type,'c_r_norm')
        ylabel('1-normalized c_r');
    elseif strcmp(out_type,'c_f_norm')
        ylabel('1-normalized c_f');
    end
end