%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Example of error minimization code using the ADE model for two breakthrough
% curves measured simultaneously at different sites in a 1D column
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% UNITS IN THE INPUT FILE AND FOR ALL INPUT PARAMETERS
% MUST BE CONSISTENT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% This file is part of the CTRW toolbox v4.0
% For the usage of this file see Legal_notice.m
% Author   : Andrea Cortis
% $Original version : 26-Jun-2004$
% Modified by: Simon Emmanuel
% $Modified version : 27-Nov-2006$
% Modified by: Shira Rubin
% $Modified version : May-2010$
% Modified by: Rami Ben-Zvi
% $Modified version : 01-Sep-2015$
% Modified by: Alon Nissan
% $Modified version : 20-March-2017$
% e-mail   : brian.berkowitz@weizmann.ac.il
%           (see http://weizmann.ac.il/EPS/people/Brian/CTRW)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clc
clear all; close all; fclose all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 1
% Choice of model: ADE, TPL, ETA
% (see 'options{1}' in the user guide)
mod_type ='ADE';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 2
% Initial parameter guess
% Input of v and D must be in units of 1/TIME: i.e., divide the dimensional
% v and D by L (column length) and L^2, respectively.
v =  5e-3 ; % v_psi [T^-1]
D = 5e-5 ;% D_psi [T^-1]
p= []; % parameters must be consistent with model chosen in Field 1


% NOTE: Do not forget to reconvert to real dimensional units of v and D;
%   multiply  v  and  D  by L (column length) and L^2, respectively.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 3
% Relative location (normalized) of measurement along the column
col_pos1 = 0.127;
col_pos2 = 0.762;
% measurement position, should be in the range [0,1]
% (see 'options{5}' in the user guide)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 4
% Boundary condition at the outlet
% String may be 'N' (Neumann) or 'D'  (Dirichlet).
% (see 'options{2}' in the user guide)
outlet_BC = 'N';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 5
% Boundary condition at the inlet
% String may be 'R' (Robin) or 'D' (Dirichlet).
% (see 'options{7}' in the user guide)
inlet_BC = 'R';
% Time dependence f(t) at the inlet.
% String options include 'step', 'pulse', or 'square'.
% Or user defined functional dependence (in Laplace space).
% (see 'options{3}' in the user guide)
inlet_ft = 'step';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 6
% Solve for:
% - either the resident  concentration
% - or the flux-averaged concentration
% String may be 'c_f_norm' (flux-averaged) or 'c_r_norm' (resident).
% (see 'options{4}' in the user guide)
out_type = 'c_f_norm';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 7
% Name of input data file; The .m suffix must be omitted
data_2
data_3

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 8
% Name of output data file
output = 'output.txt' ;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIELD 9
% Generate the result for the first guess only [0]
% or initiate minimization [1]
minimize = 0;
optimization_type = 0; % [0]  Unconstrained Optimization [1] Constrained Optimization

% Note that use of the Constrained Optimization function type ('fminsearchbnd.m')
% requires lower and upper bounds (see lines 176-177, parameters LB, UB)
% Note also that to constrain a parameter with a constant value (that is not to be
% optimized, simply multiply the parameter value by 1.0 in each of LB and UB.


%__________________________________________________________________________
% MAIN SCRIPT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DO NOT CHANGE UNLESS YOU INTEND TO ALTER CODE
% COMPARE TO INPUT_1.M TO SEE HOW CODE HAS BEEN
% CHANGED TO ALLOW MORE THAN ONE BREAKTHROUGH CURVE
% AS INPUT. NOTE THAT DIFF_NORM_MULTI.M IS USED TO CALCULATE
% THE ERROR. IF THE NUMBER OF BC'S IS INCREASED,
% DIFF_NORM_MULTI.M MUST ALSO BE CHANGED.

% Interpolation of data files
xa1 = a1(:,1);
ya1 = a1(:,2);
points = 1000;
minxa =  min(xa1);
maxxa = max(xa1);
minxa =  minxa+ (maxxa -minxa)/1000;
maxxa = maxxa- (maxxa -minxa)/1000;
t0 = transpose(linspace(minxa,maxxa,points));
data1(:,1) = t0;
data1(:,2) = interp1(xa1,ya1,t0);

xa2 = a2(:,1);
ya2 = a2(:,2);
points = 1000;
minxa =  min(xa2);
maxxa = max(xa2);
minxa =  minxa+ (maxxa -minxa)/1000;
maxxa = maxxa- (maxxa -minxa)/1000;
t0 = transpose(linspace(minxa,maxxa,points));
data2(:,1) = t0;
data2(:,2) = interp1(xa2,ya2,t0);

% Calculation of additional parameters

inlet_f0 = 1;
% Note: inlet_f0 should NOT be modified in Toolbox version 3.1;
%    scaling factor f0 at the inlet (see 'options{6}' in the user guide.
col_length = 1;   % column length
pos1 = col_pos1/col_length;
pos2 = col_pos2/col_length;
vDp = [v D p] ;
vDp_old = vDp;
vDp(1) = vDp_old(1)/col_length;
vDp(2) = vDp_old(2)/col_length^2;
options1={ mod_type , outlet_BC , inlet_ft, out_type , pos1, inlet_f0, inlet_BC};
options2={ mod_type , outlet_BC , inlet_ft, out_type , pos2, inlet_f0, inlet_BC};

% Calculation of normalized error
[err_old, model_1, model_2] = ...
    diff_norm_multi(vDp,...
    {data1,'t_conc',options1},...
    {data2,'t_conc',options2});
err = err_old;

% Minimization algorithm
if minimize==1
    
    if optimization_type == 0
        OPTS = optimset('TolFun',1e-6,'TolX',1e-6,'display','iter');
        [vDp, FVAL , EXITFLAG] = my_fminsearch('diff_norm_multi', vDp , OPTS ,...
            {data1,'t_conc',options1},...
            {data2,'t_conc',options2});
        fprintf(1,' err = %6.4e \n',err);
        
        
        
    elseif  (optimization_type == 1)
        LB = [vDp(1)*0.1, vDp(2)*0.1];  % values lower bounds
        UB = [vDp(1)*10 ,  vDp(2)*10];  % values lower bounds
        
        OPTS = optimset('TolFun',1e-6,'TolX',1e-6,'MaxIter',200,'display','iter');
        [vDp, fval] = fminsearchbnd('diff_norm_multi', vDp ,LB,UB, OPTS...
            ,{data1,'t_conc',options1},...
            {data2,'t_conc',options2});
        
        fprintf(1,' err = %6.4e \n',err);
    end
    
    
end


vDp(1) = vDp(1)*col_length;
vDp(2) = vDp(2)*col_length^2;

% Output file
gen_output(vDp_old, vDp, err_old, err, mod_type, output)
fprintf(1,' err = %6.4e \n',err);

% Graphics
figure(1)
str_title=['Breakthrough curve at column position:   ',num2str(col_pos1)];
plot(data1(:,1),model_1, 'r', xa1,ya1,'.r')
title(str_title)
xlabel('time');
if strcmp(out_type,'c_res')
    ylabel('c_r');
elseif strcmp(out_type,'c_flux')
    ylabel('c_f');
elseif strcmp(out_type,'c_r_norm')
    ylabel('normalized c_r');
elseif strcmp(out_type,'c_f_norm')
    ylabel('normalized c_f');
end

figure(2)
str_title=['Breakthrough curve at column position:   ',num2str(col_pos2)];
plot(data2(:,1),model_2, 'b', xa2,ya2,'.b')
title(str_title)
xlabel('time');
if strcmp(out_type,'c_res')
    ylabel('c_r');
elseif strcmp(out_type,'c_flux')
    ylabel('c_f');
elseif strcmp(out_type,'c_r_norm')
    ylabel('normalized c_r');
elseif strcmp(out_type,'c_f_norm')
    ylabel('normalized c_f');
end

figure(3)
plot(data1(:,1),model_1, 'r', xa1,ya1,'.r',....
    data2(:,1),model_2, 'b', xa2,ya2,'.b');
title('Comparison')
xlabel('time');
legend(['x=' num2str(col_pos1) ] ,'data\_2' , ['x=' num2str(col_pos2) ], 'data\_3', 'Location', 'SouthEast')
if strcmp(out_type,'c_r_norm')
    ylabel('normalized c_r');
elseif strcmp(out_type,'c_f_norm')
    ylabel('normalized c_f');
end

if strcmp(out_type,'c_r_norm') || strcmp(out_type,'c_f_norm')
    figure(4)
    loglog(data1(:,1),1-model_1, 'r', xa1,1-ya1,'.r',....
        data2(:,1),1-model_2, 'b', xa2,1-ya2,'.b');
    title('Comparison on a log-scale')
    xlabel('time');
    legend(['x=' num2str(col_pos1) ] ,'data\_2' , ['x=' num2str(col_pos2) ], 'data\_3', 'Location', 'SouthEast')
    if strcmp(out_type,'c_r_norm')
        ylabel('1-normalized c_r');
    elseif strcmp(out_type,'c_f_norm')
        ylabel('1-normalized c_f');
    end
end