function y = t_conc(t,param,  options)
% ------------------------------------------------------------
% T_CONC
% Solution for the total flux averaged
% (or resident concentration) for a 1D stationary domain,
% at a given position 'x'
% with constant velocity ( v ) and dispersivity ( D/v ) .
%
% Y = T_CONC(T,VARPAR,OPTIONS);
%
% t      := time vector
% varpar := vector containing the velocity,
%           the dispersion and the psi parameters
% options:= cell array containing information on the model,
%           the outlet boundary condition, the inlet input,
%           the kind of concentration, and the section.
% y      := vector of concentrations
%
% (see the User's Guide for a detailed description)
%
% Example:
% y = t_conc(transpose(logspace(-1,2,10)),[1 0.05 [0.75 ...
%            log(1) log(100)]],{'TPL','N','step','c_f_norm',1,1,'R'})
%
% See also L_conc
% ------------------------------------------------------------

% This file is part of the CTRW toolbox v4.0
% For the usage of this file see Legal_notice.m
% Author   : Andrea Cortis
% $Original version : 26-Jun-2004$
% Modified by: Simon Emmanuel
% $Modified version : 27-Nov-2006$
% Modified by: Shira Rubin
% $Modified version : May-2010$
% Modified by: Rami Ben-Zvi
% $Modified version : 16-Sep-2015$
% Modified by: Alon Nissan
% $Modified version : 20-March-2017$
% e-mail   : brian.berkowitz@weizmann.ac.il
%           (see http://weizmann.ac.il/ESER/people/Brian/CTRW)

% ------------------------------------------------------------

v         = param(1);
D         = param(2);
p         = param(3:length(param));

str_mem   = options{1};
str_bou   = options{2};
str_conc  = options{4};
x         = options{5};
inlet_BC  = options{7};

switch str_mem
    case 'ETA'
        eta = param(3:length(param));
        clencurt_coeff;
        calcola_coeff_eta(eta);
end

y = ilap('L_conc' ,t, {param , options});

if  strcmp(str_conc,'c_r_norm')  || strcmp(str_conc,'c_f_norm')
    switch str_mem
        case  'ADE'
            tmean=1;                    % first moment of psi(t)
            t1=1;
        case  'TPL'
            beta=p(1);
            t1=10^p(2);
            t2=10^p(3);
            tau=t1/t2;
                tmean=(((t2./t1).^beta) .* (-t1 + (exp(t1./t2)) .* ((t1./t2).^beta) .* (t1 + beta.*t2).*gammq(1 - beta, t1./t2)))...
            ./(exp(t1./t2).*(beta*gammq(-beta, t1./t2)));
            case 'ETA'
            eta=p;
            tt=logspace(-12,10,1e5)';
            calcola_coeff_eta(eta);
            psi=t_psi(tt,eta,{'ETA'});
            tmean  = trapz(tt,tt.*psi);
            load N
            load median_N
            t1 = interp1(N,median_N,eta);   % median of psi(t)
%         case 'EXP'
% 	    t1 = p(1);			% tbar
% 	    M0 = mem(0, p, {'EXP'});	% M0 is M(u=0)
% 	    tmean = t1 / M0;		% first moment of psi(t)
    end;	
end;

switch str_conc
    case {'c_r_norm'}
        switch inlet_BC
            case 'R'
                switch str_bou
                    case 'N'
                        y=y*t1/tmean;
                    case 'D'
                        y=y*t1/(1-exp((x-1)*v/D))/tmean;
                end;
            case 'D'
                switch str_bou
                    case 'N'
                        y=y;
                    case 'D'
                        y=y*(exp(v/D)-1)/exp(v/D)/(1-exp((x-1)*v/D));
                end;
        end;
    case {'c_f_norm'}
        switch inlet_BC
            case 'R'
                switch str_bou
                    case 'N'
                        y=y*t1/tmean;
                    case 'D'
                        y=y*t1/tmean;
                end;
            case 'D'
                switch str_bou
                    case 'N'
                        y=y;
                    case 'D'
                        y=y*(exp(v/D)-1)/exp(v/D);
                end;
        end;
end;

