% Conventional CW-CP with water exchange 

% jihyun-kim@weizmann.ac.il
% lucio.frydman@weizmann.ac.il
% i.kuprov@soton.ac.uk

function frydman_pump_cw()

% filename=[mfilename '_200water_CW3.5kHz_offset_0_-4_kex0'];

% Number of water protons
n_water_protons=20;

% Magnet field
sys.magnet=11.7;

% Core spin system
sys.isotopes={'1H','15N'};

% Add water protons
sys.isotopes(3:(3+n_water_protons-1))={'1H'};

% Chemical shifts, ppm
inter.zeeman.scalar={0.0 0.0};
inter.zeeman.scalar(3:(3+n_water_protons-1))={-9.0};

% Scalar couplings, Hz
inter.coupling.scalar=cell(n_water_protons+2);
inter.coupling.scalar{1,2}=-90;

% Estimated relaxation times, seconds
T1H=0.6; T1N=0.8; T1W=3.0;  
T2H=0.2; T2N=0.8; T2W=0.5; 

% Relaxation theory
inter.relaxation={'t1_t2'};
inter.r1_rates=num2cell(1./[T1H T1N T1W*ones(1,n_water_protons)]); %num2cell
inter.r2_rates=num2cell(1./[T2H T2N T2W*ones(1,n_water_protons)]); %num2cell
inter.rlx_keep='diagonal';
inter.equilibrium='IME';
inter.temperature=298;

% Basis set
bas.formalism='sphten-liouv';
bas.approximation='IK-1';
bas.connectivity='scalar_couplings';
bas.level=4;
bas.space_level=1;

% Exchange rates, Hz
nh_wt_exch_rate=300; % between NH and nearest water
wt_wt_exch_rate=1e4; % between all water protons

% Exchange rate matrix
inter.chem.flux_rate=zeros(n_water_protons+2);
inter.chem.flux_rate(1,3)=nh_wt_exch_rate;
inter.chem.flux_rate(3,1)=nh_wt_exch_rate;
inter.chem.flux_rate(3:(3+n_water_protons-1),...
                     3:(3+n_water_protons-1))=wt_wt_exch_rate;
inter.chem.flux_type='intermolecular';

% Spinach housekeeping
spin_system=create(sys,inter);
spin_system=basis(spin_system,bas);

% Simulation parameters
parameters.spins={'1H','15N'};
parameters.npoints=200; 
parameters.duration=200e-3; 
parameters.irr_powers=3e3; % Hz

% Get system trajectory (sequence is below)
traj=liquid(spin_system,@frydman_pump,parameters,'nmr');

% Observables - peptide bond spins
HNz=state(spin_system,{'Lz'},{1}); 
HNx=(state(spin_system,{'L+'},{1})+...
     state(spin_system,{'L-'},{1}))/2; 
Nz=state(spin_system,'Lz','15N'); 
Nx=(state(spin_system,'L+','15N')+...
    state(spin_system,'L-','15N'))/2;

% Observables - water spins
Hz=state(spin_system,'Lz','1H')-HNz; 
Hx=(state(spin_system,'L+','1H')+...
     state(spin_system,'L-','1H'))/2-HNx; 

% Project out the observables
Hz=real(Hz'*traj); Hx=real(Hx'*traj);
Nz=real(Nz'*traj); Nx=real(Nx'*traj);
HNz=real(HNz'*traj); HNx=real(HNx'*traj);

% Plotting
% figure(); scale_figure([1.0 2.0]);
subplot(3,1,1); plot([Nz',Nx']/2e-5,'LineWidth',1); hold on;
legend({'Nz','Nx'}); axis tight; grid on;
ylabel('rel. polarisation'); xlabel('time, ms'); ylim([0 1]);

subplot(3,1,2); plot([HNz',HNx']/2e-5,'LineWidth',1); hold on;
legend({'HNz','HNx'}); axis tight; grid on;
ylabel('rel. polarisation'); xlabel('time, ms'); ylim([0 1]);

subplot(3,1,3); plot([Hz',Hx']/2e-5,'LineWidth',1); hold on;
legend({'Hz','Hx'}); axis tight; grid on;
ylabel('rel. polarisation'); xlabel('time, ms'); ylim([0 n_water_protons]);

% save(filename)

end

% Pulse sequence
function traj=frydman_pump(spin_system,parameters,~,R,K)

% Get pulse operators
Hp=operator(spin_system,'L+','1H');
Hx=(Hp+Hp')/2; Hy=(Hp-Hp')/2i;
Np=operator(spin_system,'L+','15N');
Nx=(Np+Np')/2; Ny=(Np-Np')/2i;

% Lab frame Hamiltonian and equilibrium state
H_lab=hamiltonian(assume(spin_system,'labframe'),'left');
rho=equilibrium(spin_system,H_lab);

% Set H
spin_system=assume(spin_system,'nmr');
H=hamiltonian(spin_system); 
L=H+1i*R+1i*K;

% CW on 1H and 15N
L=L+2*pi*parameters.irr_powers*Nx+...
    2*pi*parameters.irr_powers*Hx;

% Get the trajectory going
time_step=parameters.duration/parameters.npoints;

% Nitrogen crusher
rho=step(spin_system,Ny,rho,+pi/2);
rho=homospoil(spin_system,rho,'destroy');

% Forward flip
rho=step(spin_system,Hy,rho,+pi/2);

% CP period
traj=krylov(spin_system,L,[],rho,time_step,...
            parameters.npoints-1,'trajectory');

end
