% TAPF-CP simulation with phase cycle r12

% jihyun-kim@weizmann.ac.il
% lucio.frydman@weizmann.ac.il
% i.kuprov@soton.ac.uk

function frydman_pump_xix_sg()

filename=[mfilename '_200water_offset_0_-4_kex1000'];

% Number of water protons 
n_water_protons=200;

% Magnet field
sys.magnet=11.7;

% Core spin system
sys.isotopes={'1H','15N'};

% Add water protons
sys.isotopes(3:(3+n_water_protons-1))={'1H'};

% Chemical shifts, ppm
inter.zeeman.scalar={0.0 0.0};
inter.zeeman.scalar(3:(3+n_water_protons-1))={-4.0};

% Scalar couplings, Hz
inter.coupling.scalar=cell(n_water_protons+2);
inter.coupling.scalar{1,2}=-90;

% Estimated relaxation times, seconds
T1H=0.6; T1N=0.8; T1W=3.0;  
T2H=0.2; T2N=0.8; T2W=0.5; 

% Relaxation theory
inter.relaxation={'t1_t2'};
inter.r1_rates=num2cell(1./[T1H T1N T1W*ones(1,n_water_protons)]); 
inter.r2_rates=num2cell(1./[T2H T2N T2W*ones(1,n_water_protons)]); 
inter.rlx_keep='diagonal';
inter.equilibrium='IME';
inter.temperature=298;

% Basis set
bas.formalism='sphten-liouv';
bas.approximation='IK-1';
bas.connectivity='scalar_couplings';
bas.level=4;
bas.space_level=1;

% Exchange rates, Hz
nh_wt_exch_rate=1000; % between NH and nearest water
wt_wt_exch_rate=1e4; % between all water protons

% Exchange rate matrix
inter.chem.flux_rate=zeros(n_water_protons+2);
inter.chem.flux_rate(1,3)=nh_wt_exch_rate;
inter.chem.flux_rate(3,1)=nh_wt_exch_rate;
inter.chem.flux_rate(3:(3+n_water_protons-1),...
                     3:(3+n_water_protons-1))=wt_wt_exch_rate;
inter.chem.flux_type='intermolecular';

% Spinach housekeeping
spin_system=create(sys,inter);
spin_system=basis(spin_system,bas);

% Simulation parameters
parameters.spins={'1H','15N'};
parameters.nblocks=500;
parameters.pulse_width=25e-6;
parameters.irr_pwr_n=10e3/3;     % Hz
parameters.irr_pwr_h=10e3;      % Hz


% Get system trajectory (sequence is below)
traj=liquid(spin_system,@frydman_pump,parameters,'nmr');

% Observables - peptide bond spins
HNz=state(spin_system,{'Lz'},{1}); 
HNx=(state(spin_system,{'L+'},{1})+...
     state(spin_system,{'L-'},{1}))/2; 
Nz=state(spin_system,'Lz','15N'); 
Nx=(state(spin_system,'L+','15N')+...
    state(spin_system,'L-','15N'))/2;

% Observables - water spins
Hz=state(spin_system,'Lz','1H')-HNz; 
Hx=(state(spin_system,'L+','1H')+...
     state(spin_system,'L-','1H'))/2-HNx; 

% Project out the observables
Hz=real(Hz'*traj); Hx=real(Hx'*traj);
Nz=real(Nz'*traj); Nx=real(Nx'*traj);
HNz=real(HNz'*traj); HNx=real(HNx'*traj);

% Handling x-axis as contact time
dt = parameters.pulse_width*12;
t = 0:dt:dt*(parameters.nblocks); t=t*1000; %ms unit

% Plotting
figure(); 
scale_figure([1.0 1.4]);
subplot(3,1,1); plot(t,Nx'/2e-5,'LineWidth',1); hold on; 
legend({'Nx'}); axis tight; grid on;
ylabel('rel. polarisation'); xlabel('time, ms'); ylim([0 1]); 
set(gca,'FontSize',15,'LineWidth',1); title('N magnetization')

subplot(3,1,2); plot(t,HNx'/2e-5,'LineWidth',1); hold on; 
legend({'HNx'}); axis tight; grid on;
ylabel('rel. polarisation'); xlabel('time, ms'); ylim([0 1]);
set(gca,'FontSize',15,'LineWidth',1); title('H^N magnetization')

subplot(3,1,3); plot(t,Hz'/2e-5,'LineWidth',1); hold on; 
legend({'Hz'}); axis tight; grid on;
ylabel('rel. polarisation'); xlabel('time, ms'); ylim([0 n_water_protons]);
set(gca,'FontSize',15,'LineWidth',1); title('H_{water} magnetization')

set(gcf,'Color','w');

% Save the figure
% savefig(gcf,'pump_xix_sg.fig');

save(filename) 

end


% Pulse sequence
function traj=frydman_pump(spin_system,parameters,~,R,K)

% Get pulse operators
Hp=operator(spin_system,'L+','1H');
Hx=(Hp+Hp')/2; Hy=(Hp-Hp')/2i;
Np=operator(spin_system,'L+','15N');
Nx=(Np+Np')/2; Ny=(Np-Np')/2i;

% Lab frame Hamiltonian and equilibrium state
H_lab=hamiltonian(assume(spin_system,'labframe'),'left');
rho=equilibrium(spin_system,H_lab);

% Set H
spin_system=assume(spin_system,'nmr');
H=hamiltonian(spin_system); 
L=H+1i*R+1i*K;

% Create pulse slice Liouvillians
L_p=L+2*pi*parameters.irr_pwr_n*Nx+...
      2*pi*parameters.irr_pwr_h*Hx;
L_m=L+2*pi*parameters.irr_pwr_n*Nx-...
      2*pi*parameters.irr_pwr_h*Hx;

% Nitrogen crusher
rho=step(spin_system,Ny,rho,+pi/2);
rho=homospoil(spin_system,rho,'destroy');

% Forward flip
rho=step(spin_system,Hy,rho,+pi/2);

% Start the trajectory
traj=rho;

% Loop over blocks
for n=1:parameters.nblocks

    % Loop over pulse pairs
    for k=1:4
        % Apply pulse pair
        rho=step(spin_system,L_p,rho,parameters.pulse_width);
        rho=step(spin_system,L_m,rho,parameters.pulse_width);
        rho=step(spin_system,L_p,rho,parameters.pulse_width);
    end

    % Append to trajectory
    traj=[traj rho];

end

end
