% L-WURST-CP simulation with water exchange 

% jihyun-kim@weizmann.ac.il
% lucio.frydman@weizmann.ac.il
% i.kuprov@soton.ac.uk
% m.g.concilio@soton.ac.uk

function frydman_pump_wurstcp()

filename=[mfilename '_200water_6kHzSW_PL20ms_CW3.5kHz_offset_0_-4_kex0'];

% Number of water protons
n_water_protons=200;

% Magnet field
sys.magnet=11.7;

% Core spin system
sys.isotopes={'1H','15N'};

% Add water protons
sys.isotopes(3:(3+n_water_protons-1))={'1H'};

% Chemical shifts, ppm
inter.zeeman.scalar={0.0,0.0};
inter.zeeman.scalar(3:(3+n_water_protons-1))={-4.0};

% Scalar couplings, Hz
inter.coupling.scalar=cell(n_water_protons+2);
inter.coupling.scalar{1,2}=-90;

% Estimated relaxation times, seconds
T1H=0.6; T1N=0.8; T1W=3.0;  
T2H=0.2; T2N=0.8; T2W=0.5;  

% Relaxation theory
inter.relaxation={'t1_t2'};
inter.r1_rates=num2cell(1./[T1H T1N T1W*ones(1,n_water_protons)]);
inter.r2_rates=num2cell(1./[T2H T2N T2W*ones(1,n_water_protons)]);
inter.rlx_keep='diagonal';
inter.equilibrium='IME';
inter.temperature=298;

% Basis set
bas.formalism='sphten-liouv';
bas.approximation='IK-1';
bas.connectivity='scalar_couplings';
bas.level=4;
bas.space_level=1;

% Exchange rates, Hz
nh_wt_exch_rate=1000; % sbetween NH and nearest water
wt_wt_exch_rate=1e4; % between all water protons

% Exchange rate matrix
inter.chem.flux_rate=zeros(n_water_protons+2);
inter.chem.flux_rate(1,3)=nh_wt_exch_rate;
inter.chem.flux_rate(3,1)=nh_wt_exch_rate;
inter.chem.flux_rate(3:(3+n_water_protons-1),...
                     3:(3+n_water_protons-1))=wt_wt_exch_rate;
inter.chem.flux_type='intermolecular';

% Spinach housekeeping
spin_system=create(sys,inter);
spin_system=basis(spin_system,bas);

% Simulation parameters
parameters.spins={'1H','15N'};
parameters.npoints=5000; parameters.duration=10e-3; 
parameters.bandwidth=10e3; parameters.smfactor=20;
parameters.irr_powers=3.5e3; % Hz 
parameters.B1=parameters.irr_powers; % Hz
parameters.iteration=15;
pulses=2;

% Get system trajectory (sequence is below)
traj=liquid(spin_system,@frydman_pump,parameters,'nmr');

% Observables - peptide bond spins
HNz=state(spin_system,{'Lz'},{1}); 
HNx=(state(spin_system,{'L+'},{1})+...
     state(spin_system,{'L-'},{1}))/2; 
Nz=state(spin_system,'Lz','15N'); 
Nx=(state(spin_system,'L+','15N')+...
    state(spin_system,'L-','15N'))/2;

% Observables - water spins
Hz=state(spin_system,'Lz','1H')-HNz; 
Hx=(state(spin_system,'L+','1H')+...
     state(spin_system,'L-','1H'))/2-HNx; 

% Project out the observables
Hz=real(Hz'*traj); Hx=real(Hx'*traj);
Nz=real(Nz'*traj); Nx=real(Nx'*traj);
HNz=real(HNz'*traj); HNx=real(HNx'*traj);

t=linspace(0,(pulses*((parameters.duration/parameters.npoints)*parameters.npoints))*parameters.iteration,...
              pulses*(parameters.npoints+1)*parameters.iteration);

% Plotting
figure(); scale_figure([1.0 2.0]);
subplot(3,1,1); plot(t,Nx./HNz(1)',t,Nz./HNz(1)','LineWidth',1.5); hold on;
legend({'Nx','Nz'}); axis tight; grid on;
ylabel('rel. polarisation'); xlabel('time, sec'); ax1=gca; ylim([0 1]);

subplot(3,1,2); plot(t,HNx./HNz(1)',t,HNz./HNz(1)','LineWidth',1.5); hold on;
legend({'HNx','HNz'}); axis tight; grid on;
ylabel('rel. polarisation'); xlabel('time, sec');  ax2=gca; ylim([-1 1]);

subplot(3,1,3); plot(t,Hx./HNz(1)',t,Hz./HNz(1)','LineWidth',1.5); hold on;
legend({'Hx','Hz'}); axis tight; grid on;
ylabel('rel. polarisation'); xlabel('time, sec');  ax3=gca; ylim([-n_water_protons n_water_protons]);
set(gcf,'Color','w'); set([ax1 ax2 ax3],'FontSize',14);

% savefig(gcf,'frydman_pump_wurstcp.fig');
save(filename)

end


function traj=frydman_pump(spin_system,parameters,~,R,K)

% Get pulse operators
Hp=operator(spin_system,'L+','1H');
Hx=(Hp+Hp')/2; Hy=(Hp-Hp')/2i;
Np=operator(spin_system,'L+','15N');
Nx=(Np+Np')/2; Ny=(Np-Np')/2i;

% Lab frame Hamiltonian and equilibrium state
Hz1=state(spin_system,'Lz','1H','exact'); % all H for normalization
parameters.rho0=Hz1;

% Set H
H_CP1=hamiltonian(assume(spin_system,'nmr')); 
L=H_CP1+1i*R+1i*K;

% CW
L = L+2*pi*parameters.irr_powers*Nx;

time_grid=parameters.duration*ones(1,parameters.npoints)/parameters.npoints;

% getting WURST pulse
[Cx,Cy]=chirp_pulse_xy(parameters.npoints,parameters.duration,parameters.bandwidth,parameters.smfactor,'wurst');
norm_factor=max(Cx); 
Cx=Cx/norm_factor; Cy=Cy/norm_factor;
Cx=2*pi*parameters.B1*Cx; Cy=2*pi*parameters.B1*Cy;

rho=parameters.rho0;

traj=[]; traja=[]; trajb=[];
for n=1:parameters.iteration
  
    % Direct wurst
    [~,traj_stack]=shaped_pulse_xy(spin_system,L,{Hx,Hy},{Cx,Cy},time_grid,rho);
    traj_stack=cell2mat(traj_stack);
    traja=[traja traj_stack]; %rho=traj_stack(:,end); %#ok<AGROW> 

    % wurst with inverse phase
    [~,traj_stack1]=shaped_pulse_xy(spin_system,L,{Hx,Hy},{Cx,-Cy},time_grid,traj_stack(:,end));
    traj_stack=cell2mat(traj_stack1);
    trajb=[trajb traj_stack]; rho=traj_stack(:,end); %#ok<AGROW>

    trajtot=[traja trajb];
    traj=[traj trajtot]; %#ok<AGROW> 

    clear('traja');  clear('trajb'); 

    traja=[]; trajb=[]; 
end


end






