function [TT, fid] = CreateTransitionTableJ(spins, acqType, timeAxis, T2)
% CreateTransitionTableJ  Calculates spin system transitions' amplitudes and phases.
%   TT = CreateTransitionTableJ(spins)
%   Returns an Nx3 matrix, TT, such that TT(i,1) is the transition's
%   frequency (in Hz), TT(i,2) its amplitude and TT(i,3) its phase.
%   It is assumed no pulse is applied during the creation of the table.
%
%   acqType is a string which can be 'combined' or 'separaate'. If set
%   to combined, all molecules' transitions would be stored in the same
%   table, while separated would generate a cell array of transition
%   tables, one for each of the molecules in the supplied spins.
%
%   timeAxis, T2 will generate the fid matrix. If not
%   supplied, fids will not be generated.

if nargin<2, acqType = 'combined'; end

isSecular = spins.isSecular;
numMolecules = numel(spins.molecule);

switch lower(acqType)
    case 'combined'
        TT = [];
    case 'separate'
        for idx=1:numMolecules
            TT{idx} = [];
        end
end

for idxMolecule=1:numMolecules
    gmRatio = spins.molecule(idxMolecule).gmRatio;
    numNuclei = numel(gmRatio);
    numSpins = numel(spins.molecule(idxMolecule).spin);
    
    % Step I: Prepare operators
    Ix_tot = zeros(2^numNuclei, 2^numNuclei);
    Iy_tot = zeros(2^numNuclei, 2^numNuclei);
    Iz_tot = zeros(2^numNuclei, 2^numNuclei);
    Iz_tot_grd = zeros(2^numNuclei, 2^numNuclei);
    for k=1:numNuclei
        Ix_tot = Ix_tot + IxN(k,numNuclei);
        Iy_tot = Iy_tot + IyN(k,numNuclei);
        Iz_tot = Iz_tot + IzN(k,numNuclei);
        Iz_tot_grd = Iz_tot_grd + IzN(k,numNuclei)*gmRatio(k);
    end;
    Ixy_tot = Ix_tot + 1i*Iy_tot;

    % ========================================================================
    % Compute CS Hamiltonian (sans B0 offsets)
    % ========================================================================

    Hcs = zeros(2^numNuclei, 2^numNuclei);
    for p=1:numNuclei
        cs = (spins.molecule(idxMolecule).csVec(p) - spins.csCenter)*spins.B0*gmRatio(p)/1000; % in kHz
        Hcs = Hcs - 2*pi*cs*IzN(p,numNuclei);
    end;

    % ========================================================================
    % Compute J-Coupling Hamiltonian, in 2*pi*kHz
    % ========================================================================

    HJ  = zeros(2^numNuclei, 2^numNuclei);
    for p=1:numNuclei
        for k=p+1:numNuclei
            HJ = HJ + 2*pi*spins.molecule(idxMolecule).JMatrix(p,k)*0.001*( ...
                 (1-isSecular)*(IxN(p,numNuclei)*IxN(k,numNuclei) + IyN(p,numNuclei)*IyN(k,numNuclei)) ...
                 + IzN(p,numNuclei)*IzN(k,numNuclei));
        end;
    end;
    HSpins = Hcs + HJ;

    % ========================================================================
    % Calculate the transition table.
    %
    % Let rho be the density matrix, H the Hamiltonian, O the observable. We can
    % diagonlize the Hamiltonian, H=V*D*V', such that 
    %   U(t)=expm(-1i*H*t) = V*expm(-1i*D*t)*V' = V*UD*V'
    % and hence
    %   s(t) = trace(rho(t)*O) 
    %        = trace(U*rho*U'*O)
    %        = trace(V*UD*V'*rho*V*UD'*V'*O)
    %        = trace(UD*rhoV*UD'*OV)           rhoV = V'*rho*V, OV = V'*O*V
    %        = sum_m         ( (UD*rhoV*UD'*OV)_{mm} )
    %        = sum_{m,n,p,q} ( UD(m,n)*rhoV(n,p)*UD'(p,q)*OV(q,m) )
    %        = sum_{m,p}     ( UD(m,m)*rhoV(m,p)*UD'(p,p)*OV(p,m) )
    %        = sum_{m,p}     exp(-1i*(Em-Ep)*t)*OV(p,m)*rhoV(m,p) 
    %        = sum_{m,p}     exp(-1i*(Em-Ep)*t)*(OV.')(m,p)*rhoV(m,p) 
    % ========================================================================

    [V, D] = eig(HSpins);
    D = real(diag(D)); % Eigenvalues of a Hermitian hamiltonian are always real.
    OV = V'*Ixy_tot*V;
    for idxSpin=1:numSpins
        rhoV = V'*spins.molecule(idxMolecule).spin(idxSpin).rho*V;
        for idx1=1:2^numNuclei
            for idx2=1:2^numNuclei
                amp = rhoV(idx1,idx2)*OV(idx2,idx1);
                if abs(amp)>1e-10
                    switch lower(acqType)
                        case 'combined'
                            TT(end+1, :) = [(D(idx2)-D(idx1))/2/pi  amp];
                        case 'separate'
                            TT{idxMolecule}(end+1, :) = [(D(idx2)-D(idx1))/2/pi  amp];
                    end
                end
            end
        end
    end;

end

if nargin>=4
    numAcqPoints = numel(timeAxis);
    switch lower(acqType)
        case 'separate'
            for idxMolecule=1:numMolecules
                fid{idxMolecule} = zeros(1, numAcqPoints);
                numLines = size(TT{idxMolecule},1);
                for idxLine=1:numLines
                    fid{idxMolecule} = fid{idxMolecule} + TT{idxMolecule}(idxLine,2)*exp(-timeAxis/T2).*exp(-1i*2*pi*TT{idxMolecule}(idxLine,1)*timeAxis);
                end
            end
        case 'combined'
            numLines = size(TT,1);
            fid = zeros(1, numAcqPoints);
            for idxLine=1:numLines
                fid = fid + TT(idxLine,2)*exp(-timeAxis/T2).*exp(-1i*2*pi*TT(idxLine,1)*timeAxis);
            end
    end
else
    fid = [];
end