% ========================================================================
% Unit Test: Pulse-Acquire using ApplySequenceJ
% ========================================================================
%
% Simulates a pulse-acquire experiment on a simple 2 spin (J-coupled)
% system, using the ApplySequenceJ routine and possibly the 
% CreateTransitionTableJ command.
%
% ========================================================================

clear all
close all
clc

% If set to 1, this simulation won't actually simulate the acquisition,
% but excite the spins and calculate the transition amplitudes for the
% density matrix and their respective frequencies.
% This is actually TOTALLY EQUIVALENT to simulating the full acquisition,
% but a lot faster.
isUseTransitionTable = 1;

% ========================================================================
% Define spin system
% ========================================================================

% Receiver/transmitter offset, in ppm. This gets subtracted from the cs
% of each spin, in ppm, so setting this to 4.7 would, e.g., make water
% resonate at 4.7 ppm
csCenter = 0; 

% Overall B0 field, in Tesla
B0 = 3;

% Global B1 scaling. Multiplies ALL RF pulses for all spins in all molecules,
% in addition to individual B1 scaling.
B1 = 1; 

% Set to 1 to enable the secular J-coupling Hamiltonian. This slightly
% speeds up the simulation, but in general you should keep it turned off
% to simulate the spin system fully.
isSecular = 0;

% Global linewidth, in Hz. The linewidth of each molecule is then 
% calculated using:
%   Rtot = R(global) (Hz)    + R(individual) (Hz)
%        = spins.linewidth   + spins.molecule(i).spin(j).linewidth
% Note that
%   T2   = 1/(pi*Rtot)
linewidth = 1; % Hz 

% Create the spin system. This does not populate the system with 
% specific molecules or spins
spins = InitSpinsJ(csCenter, B0, isSecular, linewidth, B1); 

% Adds a single "molecule" made of two J-coupled hydrogens.
% You can change the molecule easily from 'simple J' to anything that's
% allowed by SpinsJAddMolecule, such as 'GABA', 'Glu', etc ... (try it!)
spins = SpinsJAddMolecule(spins, 'simple J');

% ========================================================================
% Define sequence
% ========================================================================

Gx = 0;
Gy = 0;
Gz = 0;
SW = 1.2; % kHz
numAcqPoints = 1000; 

% We will use the full sequence if transition tables is turned off. 
% Otherwise, we will omit the acquisition and employ the 
% "CreateTransitionTableJ" command.
seqFull = {{'hard', 90, 270}, {'acquire', numAcqPoints, SW, Gx, Gy, Gz}};
seqExcite = {{'hard', 90, 270}};

% ========================================================================
% Apply sequence
% ========================================================================

tic

% Effective T2, in ms (for ad-hoc acquisition line broadening)
T2 = (1/(spins.linewidth + spins.molecule(1).spin(1).linewidth)/pi)*1000; 

if isUseTransitionTable
    % If we use transition tables, the command CreateTransitionTableJ will
    % calculate the frequencies and (complex) amplitudes of each peak in
    % the spectrum. It is then up to us to convolve these resonances with
    % the suitable FID, which is done here.
    [spins, ~] = ApplySequenceJ(spins, seqExcite);
    TT = CreateTransitionTableJ(spins);
    
    % Now add up the FID for each resonance in the transition table
    fid = zeros(1, numAcqPoints);
    numLines = size(TT,1);
    dt = 1/SW;
    timeAxis = [0:dt:(numAcqPoints-1)*dt];
    for idx=1:numLines
        fid = fid + TT(idx,2)*exp(-timeAxis/T2).*exp(-1i*2*pi*TT(idx,1)*timeAxis*1);
    end
    fid = {fid};
else
    [~, fid] = ApplySequenceJ(spins, seqFull);
end

fprintf('Total time simulating: %.2f sec\n', toc)

% ========================================================================
% Process and plot results
% ========================================================================

dv = SW/numAcqPoints;
freqAxis = linspace(-SW/2, SW/2-dv, numAcqPoints);

figure
subplot(2,1,1)
plot(real(fid{1}))
hold on
plot(imag(fid{1}),'r');
xlabel('ms');
ylabel('a.u.');
title('FID: Real (blue) and imaginary (red) parts');

spec = fftshift(fft(fid{1}));
subplot(2,1,2)
freqAxis = freqAxis*1000;
plot(freqAxis, real(spec))
hold on
plot(freqAxis, imag(spec),'r');
title('Spectrum: real (blue) and imaginary (red) parts');
xlim([-SW/2 SW/2]*1000);
xlabel('Hz');
ylabel('a.u.');

