% ========================================================================
% Unit Test: Pulse-Acquire using ApplySequenceJ
% ========================================================================
%
% Simulates a pulse-acquire experiment on a simple 2 spin (J-coupled)
% system, using the ApplySequenceJ routine.
%
% ========================================================================

clear all
close all
clc

% ========================================================================
% Define spin system
% ========================================================================

csCenter = 0; % ppm
B0 = 3; % Tesla
B1 = 1; % Scaling
isSecular = 1;
linewidth = 0; % Hz

spins = InitSpinsJ(csCenter, B0, isSecular, linewidth, B1); 
spins = SpinsJAddMolecule(spins, 'spin half');

ppmRange = 0.5;
numSpins = 101;
csVec = randn(1,numSpins)*ppmRange;
for idx=1:numSpins
    spins.molecule(1).spin(idx).rho = spins.molecule(1).spin(1).rho;
    spins.molecule(1).spin(idx).r = spins.molecule(1).spin(1).r;
    spins.molecule(1).spin(idx).B0 = csVec(idx);
    spins.molecule(1).spin(idx).B1 = spins.molecule(1).spin(1).B1;
    spins.molecule(1).spin(idx).linewidth = spins.molecule(1).spin(1).linewidth;
end


% ========================================================================
% Define sequence
% ========================================================================

Gx = 0;
Gy = 0;
Gz = 0;
SW = 0.5; % kHz
numAcqPoints = 1000; 
dv = SW/numAcqPoints;
freqAxis = linspace(-SW/2, SW/2-dv, numAcqPoints*2);

seq = {{'hard', 90, 90}, {'acquire', numAcqPoints, SW, Gx, Gy, Gz}, {'hard', 180, 90}, {'acquire', numAcqPoints*2, SW, Gx, Gy, Gz}};

% ========================================================================
% Apply sequence
% ========================================================================

[spins, fid] = ApplySequenceJ(spins, seq);

fid = {fid{2}, fid{1}};

% ========================================================================
% Process and plot results
% ========================================================================

figure
dt = 1/SW;
timeAxis = [0:dt:(numAcqPoints-1)*dt];
subplot(2,1,1)
plot(real(fid{1}))
hold on
plot(imag(fid{1}),'r');
xlabel('ms');
ylabel('a.u.');
title('FID: Real (blue) and imaginary (red) parts');

spec = fftshift(fft(fid{1}));
subplot(2,1,2)
freqAxis = freqAxis*1000;
plot(freqAxis, real(spec))
hold on
plot(freqAxis, imag(spec),'r');
title('Spectrum: real (blue) and imaginary (red) parts');
xlabel('Hz');
ylabel('a.u.');