function [pulse, pulseHeader] = PulseReadSiemensInclude(filename, pulseDuration, flipAngle, sliceThickness, gradientAxis, centerFreq)
% SYNTAX: 
%
%     [pulse, pulseHeader] = ReadSiemensIncludeFile(filename, [pulseDuration=1], 
%                                                   [flipAngle=90], [sliceThickness], [gradientAxis]) 
%
%
% Reads a pulse from a .h file.
%
% Inputs:
%
% Variable Name   Units    Description
% filename        -        Full filename, including directory. .PTA 
%                          extension not required.
% pulseDuration   ms       Duration of pulse. OPTIONAL (default: 1).
% flipAngle       deg.     Flip angle, in degrees. OPTIONAL (default: 90).
% variablePrefix  -        Optional. Prefix of variable names. If omitted,
%                          the filename will be used. For example, if 
%                          'variablePrefix' = 'myPulse', the function will
%                          look for myPulseRefGrad, myPulseMinSlice, etc.
%                          in the .h file
% sliceThickness  mm       Optional. Thickness of slice to be excited
% gradientAxis    -        Optional. 'x', 'y', or 'z' (case insensitive)
% centerFreq      kHz      Optional (default: 0). If provided, the center
%                          of the pulse will be shifted to the frequency
%                          in question.
%
%
% Outputs:
%
% Variable Name   Units    Description
% pulseHeader     -        Structure containing info from the first few
%                          lines of the .PTA file. The following fields
%                          are included:
%                          pulseName
%                          comment
%                          refGrad
%                          minSlice
%                          maxSlice
%                          ampInt
%                          powerInt
%                          absInt
%                          For their meaning, consult the IDEA manual.
%                          Furthermore, two additional fields are provided:
%                          ampVec   - normalized vec. containing amplitudes
%                          phaseVec - vector containing phases
% pulse           -        Pulse structure. 


% If pulseDuration and flipAngle are not specified, set to default
if nargin<6
    centerFreq = 0;
end

if (nargin<5)
    gradientAxis = 'z';
end

if (nargin<4)
    sliceThickness = 10;
end

if (nargin<3) 
    flipAngle = 90;
end

if (nargin<2)
    pulseDuration = 1;
end

% Check for .h extension. Append if missing
filename = RemoveExtensionFromFilename(filename);
filename = [filename, '.h'];

if ~exist(filename, 'file'),
    error('Cannot find file %s', filename);
end

% -----------------------------------------------------------------------
% Load pulse data
% ----------------------------------------------------------------------

% Initialize header
pulseHeader.pulseName = 'MyPulse.Pulse';
pulseHeader.comment = 'No comment.';
pulseHeader.refGrad = 1;
pulseHeader.minSlice = 1;  % mm
pulseHeader.maxSlice = 100;  % mm
pulseHeader.ampInt = 50.000; 
pulseHeader.powerInt = 0.000;
pulseHeader.absInt = 0.000;

% Load data
fid=fopen(filename, 'r'); 
pulseFileStr = fread(fid, inf, 'uint8=>char')';
fclose(fid);

[pulseHeader.refGrad, ok] = ReadParameterFromASCII(pulseFileStr, 'RefGrad', '=', 1);
[pulseHeader.minSlice, ok] = ReadParameterFromASCII(pulseFileStr, 'MinSlice', '=', 1);
[pulseHeader.maxSlice, ok] = ReadParameterFromASCII(pulseFileStr, 'MaxSlice', '=', 1);
[pulseHeader.ampInt, ok] = ReadParameterFromASCII(pulseFileStr, 'AmpInt', '=', 1);
[pulseHeader.powerInt, ok] = ReadParameterFromASCII(pulseFileStr, 'PowerInt', '=', 1);
[pulseHeader.absInt, ok] = ReadParameterFromASCII(pulseFileStr, 'AbsInt', '=', 1);

% Some of the parameters may be surrounded by text. E.g., ampint is usually
% written as ampint = float(32.41). We next strip the text.
if ischar(pulseHeader.refGrad),  idxDigits = regexp(pulseHeader.refGrad, '[0-9]');  pulseHeader.refGrad  = str2num(pulseHeader.refGrad(idxDigits(1):idxDigits(end)));  end
if ischar(pulseHeader.minSlice), idxDigits = regexp(pulseHeader.minSlice, '[0-9]'); pulseHeader.minSlice = str2num(pulseHeader.minSlice(idxDigits(1):idxDigits(end))); end
if ischar(pulseHeader.maxSlice), idxDigits = regexp(pulseHeader.maxSlice, '[0-9]'); pulseHeader.maxSlice = str2num(pulseHeader.maxSlice(idxDigits(1):idxDigits(end))); end
if ischar(pulseHeader.ampInt),   idxDigits = regexp(pulseHeader.ampInt, '[0-9]');   pulseHeader.ampInt   = str2num(pulseHeader.ampInt(idxDigits(1):idxDigits(end)));   end
if ischar(pulseHeader.powerInt), idxDigits = regexp(pulseHeader.powerInt, '[0-9]'); pulseHeader.powerInt = str2num(pulseHeader.powerInt(idxDigits(1):idxDigits(end))); end
if ischar(pulseHeader.absInt),   idxDigits = regexp(pulseHeader.absInt, '[0-9]');   pulseHeader.absInt   = str2num(pulseHeader.absInt(idxDigits(1):idxDigits(end)));   end

firstIndex = strfind(pulseFileStr, 'PulseArray[0].flAbs');
pulseDataStr = pulseFileStr(firstIndex:end);

indices = strfind(pulseDataStr, '(');
indicesClose = strfind(pulseDataStr, ')');
odds = indices(1:2:end); % The odd indices are for the "left column", i.e., amplitude
evens = indices(2:2:end); % The even indices are for the "right column", i.e., phase
numPoints = numel(odds);

% Calculate how many digits each point has.
% I am implicitly assuming all time points have the same number of digits,
% so I'm just using the first one.
numDigits = indicesClose(1)-indices(1)-1; 

ampVec = zeros(1, numPoints);
phaseVec = zeros(1, numPoints);


for idx=1:numPoints
    ampVec(idx) = str2double(pulseDataStr(odds(idx)+1:odds(idx)+numDigits));
    phaseVec(idx) = str2double(pulseDataStr(evens(idx)+1:evens(idx)+numDigits));
end

pulseHeader.ampVec = ampVec./max(abs(ampVec));  % Normalize to [0,1]
pulseHeader.phaseVec = phaseVec;

% -----------------------------------------------------------------------
% Create & Calibrate Pulse Structure
% ----------------------------------------------------------------------

pulse = PulseCreateFromSiemensParameters(...
    pulseHeader.ampVec, ...
    pulseHeader.phaseVec, ...
    pulseHeader.ampInt, ...
    pulseHeader.refGrad, ...
    gradientAxis, ...
    pulseDuration, ...
    sliceThickness, ...
    flipAngle);

pulseHeader.maxGradSlewRate = CalcMaxGradSlewRate(pulse);
pulseHeader.maxRFSlewRate = CalcMaxRFSlewRate(pulse);

% -----------------------------------------------------------------------
% Shift pulse center in frequency space
% ----------------------------------------------------------------------

if centerFreq~=0
    pulse = PulseShiftOffset(pulse, centerFreq); 
end
