function spins = ApplyPulseCycle(spins, pulse, pulsePhase, addCoeff)
% ApplyPulseCycle  Applies a shaped pulse with optional phase cycling
%   spins = ApplyPulseCycle(spins, pulse)  Applies the shaped pulse 
%   'pulse' to the input spin structure. 
%
%   spins = ApplyPulseCycle(spins, pulse, phaseCycle) Allows for phase
%   cycling. pulsePhase is a 1xN vector: N pulses are executed in parallel 
%   with phases pulsePhase(i), i=1,..,N, and the resulting magnetizations 
%   of the spins are added up. Phases are in degrees.
%
%   spins = ApplyPulseCycle(spins, pulse, phaseCycle, addCoeff)
%   Allows adding the different phase cycles with different weighting 
%   coefficients (including complex numbers!) in the 1xN vector addCoeff.

RFphase = pulse.RFphase;

if nargin<3, pulsePhase = 0; end
numCycles = numel(pulsePhase);
if nargin<4, addCoeff = ones(1, numCycles)/numCycles; end

numCycles = size(pulsePhase, 2);
numSpins = numel(spins);
if numCycles==1 % No phase cycle
    pulse.RFphase = pulse.RFphase + pulsePhase(1,1)/180*pi; 
    spins = ApplyPulseRelax(spins, pulse);
else % Phase cycle
    spinsInitial = spins;
    for idxSpin=1:numSpins
        spins(idxSpin).M = [0;0;0];
    end
    for idxCycle=1:numCycles
        curPhase = pulsePhase(idxCycle)/180*pi;
        curPulse = pulse;
        curPulse.RFphase = RFphase + curPhase;
        spinsOut = ApplyPulseRelax(spinsInitial, curPulse);
        for idxSpin=1:numSpins
            spins(idxSpin).M = spins(idxSpin).M + addCoeff(idxCycle)*spinsOut(idxSpin).M;
        end
    end
end



