% ========================================================================
% Unit Test: Pulse-Acquire for GABA using ApplySequenceJ
% ========================================================================
%
% Simulates an idealized MEGA-PRESS experiment on a simple GABA (J-coupled)
% system, using the ApplySequenceJ routine. The MEGA-PRESS sequence is 
% (following Waddell, MRI 25:1032-1038 (2007))
%
%
% FS - frequency selective editing pulse 
%
%             180     180(FS)     180      180(FS)
%      90      _        _          _          _
%       _     | |      | |        | |        | |
%      | |    | |      | |        | |        | |
% RF __| |____| |______| |________| |________| |_______[Acquire]
%       <--t1--><--t2---><---t3----><---t4----><---t5-->
%
% Following Waddell 2007, we put
%     t1 = shortest
%     t2 = TE/4
%     t3 = TE/2-t4
%     t4 = TE/4
%     t5 = TE/2-t1-t2
% such that
%     t1+t2+t3+t4+t5     = TE
%     t1-t2+t3-t4+t5     = 0   (ON)
%     t1-(t2+t3)+(t4+t5) = 0   (OFF)
%
% ========================================================================

clear all
close all
clc

% ========================================================================
% Define spin system
% ========================================================================

csCenter = 4.7; % ppm
B0 = 2.9; % Tesla
B1 = 1; % Scaling
isSecular = 0;
linewidth = 0; % Hz

spins = InitSpinsJ(csCenter, B0, isSecular, linewidth, B1); 
spins = SpinsJAddMolecule(spins, 'gaba');
spins = SpinsJAddMolecule(spins, 'glu');
spins = SpinsJAddMolecule(spins, 'naa singlet');
spins.molecule(1).spin(1).linewidth = 4;
T2 = (1/(spins.linewidth + spins.molecule(1).spin(1).linewidth)/pi)*1000; % Effective T2, in ms (for ad-hoc acquisition line broadening)
spins.molecule(2).spin(1).linewidth = 4;
spins.molecule(3).spin(1).linewidth = 4;

% ========================================================================
% Define editing sequence:
% ========================================================================

Gx = 0;
Gy = 0;
Gz = 0;
SW = 1.2; % kHz
numAcqPoints = 1000; 
dt = 1/SW;
timeAxis = [0:dt:(numAcqPoints-1)*dt];
TE = 68; % ms
t1 = 1; % Choose the "shortest possible" t1 & t2
t2 = TE/4;
t4 = TE/4;
t3 = TE/2-t4;
t5 = TE/2-t1-t2;

affectedNucleiExc = {[1 1 1 1 1 1], [1 1 1 1 1], [0]};
affectedNuclei = {[1 1 0 0 0 0], [1 1 1 1 0], [1]};

seqON  = {{'hard', 90, 270, affectedNucleiExc}, {'delay', t1}, ...
          {'hard', 180, 0},                     {'delay', t2}, ...
          {'hard', 180, 0, affectedNuclei},     {'delay', t3}, ...    % Editing pulse
          {'hard', 180, 0},                     {'delay', t4}, ...
          {'hard', 180, 0, affectedNuclei},     {'delay', t5}, ...    % Editing pulse
          };
seqOFF = {{'hard', 90, 270},                    {'delay', t1}, ...
          {'hard', 180, 0},                     {'delay', t2+t3}, ...
          {'hard', 180, 0},                     {'delay', t4+t5}, ...
          };

% ========================================================================
% Apply sequence
% ========================================================================

tic
[spinsON, ~] = ApplySequenceJ(spins, seqON);
[spinsOFF, ~] = ApplySequenceJ(spins, seqOFF);

TTON = CreateTransitionTableJ(spinsON);
TTOFF = CreateTransitionTableJ(spinsOFF);
fprintf('Total simulation time: %.2f sec\n', toc);

% ========================================================================
% Process and plot results
% ========================================================================

fidON = zeros(1, numAcqPoints);
fidOFF = zeros(1, numAcqPoints);

numLines = size(TTON,1);
for idx=1:numLines
    fidON = fidON + TTON(idx,2)*exp(-timeAxis/T2).*exp(-1i*2*pi*TTON(idx,1)*timeAxis);
end
numLines = size(TTOFF,1);
for idx=1:numLines
    fidOFF = fidOFF + TTOFF(idx,2)*exp(-timeAxis/T2).*exp(-1i*2*pi*TTOFF(idx,1)*timeAxis);
end
fid = {fidON, fidOFF};

spec2=SPT1DGUI('B0', B0, 'FID', fid, 'dwellTime', 0.001/SW);
