function [fid, spins] = PropagateJ(spins, pulse, acqType, affectedNuclei, freqRange)
% SYNTAX: [fid, spinsOut] = PropagateJ(spins, pulse, isAcquire, affectedNuclei)
%
% Input Variables
% Variable Name    Description
% spins            Input spin structure. For more details, see below.
% pulse            Input pulse structure. For more details, see below.
% affectedNuclei   A cell array indicating which nuclei and ppm ranges
%                  are to be affected. For example:
%                  affectedNuclei = {{'1h', 3, 4}, {'13c'}}
%                  This indicates all 13c nuclei will be acted on,
%                  as well as any protons with chemical shifts between
%                  3 and 4 ppm (before taking into account spins.csCenter).
%                  This is a "cheap" way of simulating (perfect) selective 
%                  pulses.
% acqType          0: No acquisition.
%                  1: Acquire on, sum FID for all molecules.
%                  2: Acquire on, produce cell array of FIDs for each of the molecules.
% freqRange        Used to crudely simulate frequency selective pulses. 
%                  Any nucleus falling outside the given frequency range
%                  will not be affected by the pulse. freqRange is a 
%                  1x2 vector of [lowest ppm, highest ppm]
% 
%
% The current function applies a pulse to the given spins, assuming 
% homonuclear case. The spins are a structure of the form specified in
% InitSpinsJ. The input pulse has a structure:
%
%   pulse.tp      = time of pulse, in ms (#)
%   pulse.RFamp   = amplitude of pulse, in microTesla (vector)
%   pulse.RFphase = phase of pulse, in radians (vector)
%   pulse.Gx      \
%   pulse.Gy       |-> Gradients, in mT/m (vector)
%   pulse.Gz      /

isLHRot = 1; % If set to 1, the left hand convention for rotations will be used throughout

isSecular = spins.isSecular;
numMolecules = numel(spins.molecule);
if (nargin<3), acqType = 0; end
if (nargin<4)
    affectedNuclei = [];
end
if (nargin<5)
    freqRange = [];
end

if isempty(affectedNuclei)
    affectedNuclei = {};
    for idxMolecule=1:numMolecules
        numNuclei = numel(spins.molecule(idxMolecule).gmRatio);
        affectedNuclei = [affectedNuclei, ones(1,numNuclei)];
    end
end

if ~isempty(freqRange)
    for idxMolecule=1:numMolecules
        affectedNuclei{idxMolecule} = affectedNuclei{idxMolecule} & (spins.molecule(idxMolecule).csVec<=freqRange(2)) & (spins.molecule(idxMolecule).csVec>=freqRange(1));       
    end
end


Nt = length(pulse.RFamp);
dt = pulse.tp/Nt; % in ms
switch acqType
    case 0
        fid = [];
    case 1
        fid = zeros(1,Nt);
    case 2
        for idx=1:numMolecules
            fid{idx} = zeros(1,Nt);
        end
end

for idxMolecule=1:numMolecules
    gmRatio = spins.molecule(idxMolecule).gmRatio;
    numNuclei = numel(gmRatio);
    numSpins = numel(spins.molecule(idxMolecule).spin);
    
    % Step I: Prepare operators
    Ix_tot = zeros(2^numNuclei, 2^numNuclei);
    Iy_tot = zeros(2^numNuclei, 2^numNuclei);
    Iz_tot = zeros(2^numNuclei, 2^numNuclei);
    Ix_tot_RF = zeros(2^numNuclei, 2^numNuclei);
    Iy_tot_RF = zeros(2^numNuclei, 2^numNuclei);
    Iz_tot_grd = zeros(2^numNuclei, 2^numNuclei);
    for k=1:numNuclei
        Ix_tot = Ix_tot + IxN(k,numNuclei);
        Iy_tot = Iy_tot + IyN(k,numNuclei);
        Ix_tot_RF = Ix_tot_RF + IxN(k,numNuclei)*affectedNuclei{idxMolecule}(k);
        Iy_tot_RF = Iy_tot_RF + IyN(k,numNuclei)*affectedNuclei{idxMolecule}(k);
        Iz_tot = Iz_tot + IzN(k,numNuclei);
        Iz_tot_grd = Iz_tot_grd + IzN(k,numNuclei)*gmRatio(k);
    end;
    Ixy_tot = Ix_tot + 1i*Iy_tot;
    

    % ========================================================================
    % Compute CS Hamiltonian (sans B0 offsets)
    % ========================================================================

    Hcs = zeros(2^numNuclei, 2^numNuclei);
    for p=1:numNuclei
        cs = (spins.molecule(idxMolecule).csVec(p) - spins.csCenter)*spins.B0*gmRatio(p)/1000; % in kHz
        Hcs = Hcs - 2*pi*cs*IzN(p,numNuclei); % rad*kHz
    end;
    
    % ========================================================================
    % Compute J-Coupling Hamiltonian, in 2*pi*kHz
    % ========================================================================

    HJ  = zeros(2^numNuclei, 2^numNuclei);
    for p=1:numNuclei
        for k=p+1:numNuclei
            HJ = HJ + 2*pi*spins.molecule(idxMolecule).JMatrix(p,k)*0.001*( ...
                 (1-isSecular)*(IxN(p,numNuclei)*IxN(k,numNuclei) + IyN(p,numNuclei)*IyN(k,numNuclei)) ...
                 + IzN(p,numNuclei)*IzN(k,numNuclei)); % rad*kHz
        end;
    end;
    H0 = Hcs + HJ; % rad*kHz

    
    % ========================================================================
    % Propagate each molecule through time
    % ========================================================================

    % Calculate the RF Hamiltonian WITHOUT B1 scaling
    for idxTime=1:Nt
        H_RF{idxTime} = 2*pi*pulse.RFamp(idxTime)*(cos(pulse.RFphase(idxTime))*Ix_tot_RF + sin(pulse.RFphase(idxTime))*Iy_tot_RF); % In kHz*rad
    end
    
    for idxSpin=1:numSpins
        % =====================================
        % Compute Time-Dependent RF Hamiltonian
        % =====================================
        B1Scaling = spins.molecule(idxMolecule).spin(idxSpin).B1*spins.B1; % (Local B1)*(Global B1)
        T2 = (1/(spins.linewidth + spins.molecule(idxMolecule).spin(idxSpin).linewidth)/pi)*1000; % Effective T2, in ms (for ad-hoc acquisition line broadening)
        for idxTime=1:Nt
            curTime = (idxTime-1)*dt; % ms
            % Acquire FID
            switch (acqType)
                case 1
                    fid(idxTime) = fid(idxTime) + trace(spins.molecule(idxMolecule).spin(idxSpin).rho*Ixy_tot)*exp(-curTime/T2); % Insert ad-hoc line broadening
                case 2
                    fid{idxMolecule}(idxTime) = fid{idxMolecule}(idxTime) + trace(spins.molecule(idxMolecule).spin(idxSpin).rho*Ixy_tot)*exp(-curTime/T2); % Insert ad-hoc line broadening
            end
            % =============================================
            % Compute gradient field & inhomogeneity offset
            % =============================================
            offset = spins.molecule(idxMolecule).spin(idxSpin).B0*spins.B0*gmRatio(1); % in Hz % <--- FIX THIS (gmRatio(1))
            
            HGrad = 2*pi*(pulse.Gx(idxTime)*spins.molecule(idxMolecule).spin(idxSpin).r(1) ...
                           + pulse.Gy(idxTime)*spins.molecule(idxMolecule).spin(idxSpin).r(2) ...
                           + pulse.Gz(idxTime)*spins.molecule(idxMolecule).spin(idxSpin).r(3) ...
                           + offset*0.001)*0.001*Iz_tot_grd;  % In rad*kHz
            % ==================
            % Compute propagator
            % ==================
            if ((acqType~=0) && isSecular) 
                P = (-1)^(isLHRot-1)*1i*(H0 + HGrad)*dt; % No RF. dt is in ms. HGrad, H0 is in rad*kHz.
                U = diag(exp(diag(P)));
            else
                U = expm((-1)^(isLHRot-1)*1i*(H0 + HGrad + B1Scaling*H_RF{idxTime})*dt); % <--- FIX THIS (gmRatio(1))
            end
            % =========
            % Propagate
            % =========
            spins.molecule(idxMolecule).spin(idxSpin).rho = U*spins.molecule(idxMolecule).spin(idxSpin).rho*U';
        end
    end;

end

