classdef Pulse
    % Class for storing RF pulses
    
    properties
        dtAxis      % 1xN time axis (does not have to be uniform), in ms
        amp         % 1xN amplitude of RF pulse, in uT
        phase       % 1xN phase of RF pulse, in radians
        offset      % Central transmit frequency of pulse, in kHz
    end
    
    properties (Dependent)
        timeAxis
        numSteps 
        dwellTime
        duration
    end
        
    
    methods
        function pulse = Pulse(varargin)
            % Constructor
            p = inputParser;
            p.addParameter('dwellTime', [], @(x) x>0); 
            p.addParameter('duration', [], @(x) x>0);
            p.addParameter('dtAxis', [], @(x) isvector(x') && prod(x>0) && isreal(x));
            p.addParameter('amp', 0, @(x) isvector(x') && prod(x>=0) && isreal(x));
            p.addParameter('phase', 0, @(x) isvector(x') && prod(x>=0) && isreal(x));
            p.addParameter('offset', 0, @(x) isreal(x));
            p.parse(varargin{:});
            pulse.amp = p.Results.amp;
            pulse.phase = p.Results.phase;
            pulse.offset = p.Results.offset;
            N = numel(pulse.amp);
            if ~isempty(p.Results.dtAxis)
                pulse.dtAxis = p.Results.dtAxis;
            else
                if ~isempty(p.Results.duration)
                    pulse.dtAxis = ones(1,pulse.numSteps)*p.Results.duration/pulse.numSteps; 
                else
                    error('wowowow');
                end
            end
            
        end

        
        
        function pulse = SetDuration(pulse, duration)
            pulse.dtAxis = ones(1, pulse.numSteps)*(duration/pulse.numSteps);
        end
        
        function duration = get.duration(pulse)
            % Returns total pulse duration, in ms
            duration = sum(pulse.dtAxis);
        end
        
        function numSteps = get.numSteps(pulse)
            % Returns total number of steps in pulse
            numSteps = numel(pulse.amp);
        end
        
        function Plot(pulse)
            % Plots pulse behavior as a function of time
            figure;
            
            ax(1) = subplot(1,4,1);
            plot(pulse.timeAxis, pulse.amp);
            title('RF Amplitude');
            xlabel('ms');
            ylabel('uT');

            ax(2) = subplot(1,4,2);
            plot(pulse.timeAxis, pulse.RFphase);
            title('RF Phase');
            xlabel('ms');
            ylabel('rads.');

            ax(3) = subplot(1,4,3);
            Ot = diff(pulse.amp)./diff(pulse.timeAxis)/2/pi; 
            plot(pulse.timeAxis(1:end-1),Ot);
            xlabel('ms');
            title('Frequency');
            ylabel('kHz');

            ax(4) = subplot(1,4,4);
            plot(pulse.timeAxis, real(pulse.amp.*exp(1i*pulse.phase)));
            plot(pulse.timeAxis, imag(pulse.amp.*exp(1i*pulse.phase)), 'r');
            title('Real & Imaginary');
            xlabel('ms');
            ylabel('kHz');
            
            linkaxes(ax, 'x');            
        end
        
        function pulse = Copy(pulse)
        end
        
        function PlotFreqResponse(pulse)
        end
        
        
    end
    
end