function spins = SpinsJAddMolecule(spins, molecule, relAmplitude, r)
% SYNTAX: spins = SpinsJAddMolecule(spins, molecule, relAmplitude)
%
% Adds a specific molecule type to the spin structure.
%
% Input Variables
% spins        The input spin structure
% molecule     The input molecule. A string. One of the following (case
%              insensitive):
%              'Spin Half': On resonance spin-1/2.
%              'Simple J': Two spin-1/2s, at +-1 ppm with J=10 Hz.
%              'NAA'
%              'GABA'
%              'Lac'
%              'mI'
%              'Cho'
%              'Cr'
%              'Cho singlet'
%              'Cr singlet'
%              'NAA singlet'
%              'Glu'
%              'Gln'
%              'Water'
% relAmplitude Relative amplitude of molecule's signal
% r            Three dimensional vector indicating the molecule's position
%              (in mm)
%
% Output Variables
% spins        The input spin structure with the appended molecule.

if isempty(spins), spins = InitSpinsJ; end
numMolecules = numel(spins.molecule);
if nargin<3, 
    switch lower(molecule)
        case 'acetate'
            relAmplitude = 1; 
            moleculeName = 'Acetate';
        case {'naa', 'naa singlet'}
            relAmplitude = 10; 
            moleculeName = 'NAA';
        case 'naag'
            relAmplitude = 1; 
            moleculeName = 'NAAG';
        case 'atp'
            relAmplitude = 3; 
            moleculeName = 'ATP';
        case 'alanine'
            relAmplitude = 0.5; 
            moleculeName = 'Ala';
        case 'gaba'
            relAmplitude = 1; 
            moleculeName = 'GABA';
        case 'asc'
            relAmplitude = 1; 
            moleculeName = 'Asc';
        case 'asp'
            relAmplitude = 1.5; 
            moleculeName = 'Asp';
        case {'cho', 'cho singlet'}
            relAmplitude = 1.5; 
            moleculeName = 'Cho';
        case {'cr', 'cr singlet'}
            relAmplitude = 6; 
            moleculeName = 'Cr';
        case 'eth'
            relAmplitude = 0.5; 
            moleculeName = 'Ethanol';
        case 'glc'
            relAmplitude = 1.5; 
            moleculeName = 'Glc';
        case 'glu'
            relAmplitude = 9.0; 
            moleculeName = 'Glu';
        case 'gln'
            relAmplitude = 4; 
            moleculeName = 'Gln';
        case 'gsh'
            relAmplitude = 2; 
            moleculeName = 'GSH';
        case 'gsh-glu'
            relAmplitude = 2;
            moleculeName = 'GSH-Glu';
        case 'gsh-gly'
            relAmplitude = 2;
            moleculeName = 'GSH-Gly';
        case 'gsh-cys'
            relAmplitude = 2;
            moleculeName = 'GSH-Cys';
        case 'gpc'
            relAmplitude = 1; 
            moleculeName = 'GPC';
        case 'gly'
            relAmplitude = 0.6; 
            moleculeName = 'Gly';
        case 'mi'
            relAmplitude = 6; 
            moleculeName = 'mI';
        case 'lac'
            relAmplitude = 0.2; 
            moleculeName = 'Lac';
        case 'pcr'
            relAmplitude = 4.0; 
            moleculeName = 'PCr';
        case 'pcho'
            relAmplitude = 0.5; 
            moleculeName = 'PCho';
        case 'pyr'
            relAmplitude = 0.2; 
            moleculeName = 'Pyr';
        case 'suc'
            relAmplitude = 0.2; 
            moleculeName = 'Suc';
        case 'tau'
            relAmplitude = 4.0; 
            moleculeName = 'Tau';
        otherwise
            relAmplitude = 1; 
            moleculeName = '';
    end
end
if nargin<4, r = [0;0;0]; end

switch lower(molecule)
    case 'spin half'
        csVec = 0;
        JMatrix = 0;
        relAmplitudes = relAmplitude;
        nuclei = {'1H'};
    case 'water'
        csVec = 4.7;
        JMatrix = 0;
        relAmplitudes = 2*relAmplitude;
        nuclei = {'1H'};
    case 'simple j'
        csVec = [-0.5 0.5]; % ppm
        J = 10; % Hz
        JMatrix = [0 J;
                   J 0]; % Hz
        relAmplitudes = [1 1].*relAmplitude;
        nuclei = {'1H', '1H'};
    case 'naa singlet'
        csVec = 2.0;
        JMatrix = 0;
        relAmplitudes = 3*relAmplitude;
        nuclei = {'1H'};
    case 'cr singlet'
        csVec = 3.0;
        JMatrix = 0;
        relAmplitudes = 3*relAmplitude;
        nuclei = {'1H'};
    case 'cho singlet'
        csVec = 3.2;
        JMatrix = 0;
        relAmplitudes = 9*relAmplitude;
        nuclei = {'1H'};
    case 'cr'
        csVec = [3.027 3.913];
        relAmplitudes = relAmplitude*[3 2];
        JMatrix = zeros(2,2);
        nuclei = {'1H','1H'};
    case 'acetate'
        csVec = 1.904;
        relAmplitudes = relAmplitude*3;
        JMatrix = 0;
        nuclei = {'1H'};
    case 'asp'
        %              2      3     3'
        csVec   = [3.891  2.801  2.653];
        JMatrix = [    0   3.65   9.11;    % 2
                    3.65      0 -17.43;    % 3
                    9.11 -17.43      0];   % 3'
        relAmplitudes = [1 1 1]*relAmplitude;
        nuclei = {'1H','1H','1H'};
    case 'lac'
        csVec   = [4.097  1.313  1.313  1.313];
        JMatrix = [    0   6.93   6.93   6.93;
                    6.93      0      0      0;
                    6.93      0      0      0;
                    6.93      0      0      0];
        relAmplitudes = [1 1 1 1]*relAmplitude;
        nuclei = {'1H','1H','1H','1H'};
    case 'pcr'
        csVec = [3.029 3.930];
        relAmplitudes = relAmplitude*[3 2];
        JMatrix = zeros(2,2);
        nuclei = {'1H','1H'};
    case 'cho'
        %                    1    1'     2    2'
        csVec   = [3.185 4.054 4.054 3.501 3.501];
        JMatrix = [    0     0     0     0     0;   
                       0     0     0  3.15  6.99;   % 1
                       0     0     0  6.99  3.15;   % 1'
                       0  3.15  6.99     0     0;   % 2
                       0  6.99  3.15     0     0];  % 2'
        relAmplitudes = relAmplitude*[9 1 1 1 1];
        nuclei = {'1H','1H','1H','1H','1H'};
    case 'gaba'
        csVec = [1.89 1.89 2.28 2.28 3.01 3.01]; % ppm
        J = 7.3; % Hz
        JMatrix = [0 J J J J J;
                   J 0 J J J J;
                   J J 0 J 0 0;
                   J J J 0 0 0;
                   J J 0 0 0 J;
                   J J 0 0 J 0];
        relAmplitudes = [1 1 1 1 1 1]*relAmplitude; 
        nuclei = {'1H','1H','1H','1H','1H','1H'};
    case 'gln'
        %           3        3b       4       4b      2
        csVec   = [2.135    2.115   2.434    2.456   3.757];
        JMatrix = [    0   -14.45    6.35     6.88    5.84;   % 3
                  -14.45        0    9.16     6.88    6.53;   % 3b
                    6.35     9.16       0   -15.92       0;   % 4
                    6.88     6.88  -15.92        0       0;   % 4b
                    5.84     6.53       0        0       0];  % 2
        relAmplitudes = [1 1 1 1 1]*relAmplitude;
        nuclei = {'1H','1H','1H','1H','1H'};
    case 'glu'
        %           3        3b       4       4b      2
        csVec   = [2.042     2.12   2.336    2.352   3.746];
        JMatrix = [    0   -14.85    6.41     8.41    7.33;   % 3
                  -14.85        0    8.48     6.88    4.65;   % 3b
                    6.41     8.48       0   -15.92       0;   % 4
                    8.41     6.88  -15.92        0       0;   % 4b
                    7.33     4.65       0        0       0];  % 2 
        relAmplitudes = [1 1 1 1 1]*relAmplitude;
        nuclei = {'1H','1H','1H','1H','1H'};
    case 'naa'
        %            1     2      3    3'
        csVec   = [2.008 4.382 2.6727 2.4863];
        JMatrix = [  0     0       0       0;    % 1
                     0     0    3.86    9.82;    % 2
                     0  3.86       0  -15.59;    % 3
                     0  9.82  -15.59       0];   % 3'
        relAmplitudes = [3 1 1 1]*relAmplitude;
        nuclei  = {'1H', '1H', '1H', '1H'};
    case 'gsh'
        %           10      7      7'    7''     2     3       3'     4      4'
        csVec   = [3.769  4.561  2.926  2.975  3.769  2.159  2.146  2.510  2.560]; 
        JMatrix = [    0      0      0      0      0      0      0      0      0;    % 10
                       0      0   7.09   4.71      0      0      0      0      0;    % 7
                       0   7.09      0 -14.06      0      0      0      0      0;    % 7'
                       0   4.71 -14.06      0      0      0      0      0      0;    % 7''
                       0      0      0      0      0   6.34   6.36      0      0;    % 2
                       0      0      0      0   6.34      0 -15.48    6.7    7.6;    % 3
                       0      0      0      0   6.36 -15.48      0    7.6    6.7;    % 3'
                       0      0      0      0      0    6.7    7.6      0 -15.92;    % 4
                       0      0      0      0      0    7.6    6.7 -15.92      0];   % 4'
        relAmplitudes = [2 1 1 1 1 1 1 1 1]*relAmplitude;           
        nuclei = {'1H','1H','1H','1H','1H','1H','1H','1H','1H'};
    case 'gsh-glu' % Just the glutamate moeity of GSH, for speed        
        %            2     3       3'     4      4'
        csVec   = [3.769  2.159  2.146  2.510  2.560]; 
        JMatrix = [    0   6.34   6.36      0      0;    % 2
                    6.34      0 -15.48    6.7    7.6;    % 3
                    6.36 -15.48      0    7.6    6.7;    % 3'
                       0    6.7    7.6      0 -15.92;    % 4
                       0    7.6    6.7 -15.92      0];   % 4'
        relAmplitudes = [1 1 1 1 1]*relAmplitude;
        nuclei = {'1H','1H','1H','1H', '1H'};
    case 'gsh-gly' % Just the glycine moeity of GSH, for speed
        csVec   = 3.769; 
        JMatrix = 0;
        relAmplitudes = relAmplitude*2; 
        nuclei = {'1H'};
    case 'gsh-cys'
        %              7     7'    7''                    
        csVec   = [4.561   2.926   2.975];
        JMatrix = [    0    7.09    4.71;
                    7.09       0  -14.06;
                    4.71  -14.06       0];
        relAmplitudes = [1 1 1]*relAmplitude;
        nuclei = {'1H', '1H', '1H'};
    case 'tau' % Taurine
        %            1      1'     2      2'
        csVec   = [3.420  3.420  3.246  3.246]; 
        JMatrix = [    0      0   6.74   6.46;    % 1
                       0      0   6.40   6.79;    % 1'
                    6.74   6.40      0      0;    % 2
                    6.46   6.79      0      0];   % 2'
        relAmplitudes = [1 1 1 1]*relAmplitude;
        nuclei = {'1H','1H','1H','1H'};
    case 'mi' % Myo-inositol
        %              1      2      3      4      5      6
        csVec   = [3.522  4.054  3.522  3.614  3.269  3.614];
        JMatrix = [    0   2.89      0      0      0   10.0;   % 1
                    2.89      0   3.01      0      0      0;   % 2
                       0   3.01      0   10.0      0      0;   % 3
                       0      0   10.0      0   9.49      0;   % 4
                       0      0      0   9.49      0   9.48;   % 5
                    10.0      0      0      0   9.48      0];  % 6
        relAmplitudes = [1 1 1 1 1 1]*relAmplitude;
        nuclei = {'1H','1H','1H','1H', '1H', '1H'};
        
    otherwise
        error('Unidentified molecule type %s in SpinsJAddSpin.', molecule);
end

numNuclei = numel(csVec);
gmRatio = zeros(1,numNuclei);
for idx=1:numNuclei
    gmRatio(idx) = GetGyromagneticRatio(nuclei{idx});
end

% Create equilibrium density matrix
rho0 = zeros(2^numNuclei, 2^numNuclei);
for idx=1:numNuclei
    rho0 = rho0 + relAmplitudes(idx)*IzN(idx, numNuclei)/2^numNuclei;
end

% Add the molecule's data
spins.molecule(numMolecules+1).name = moleculeName; 
spins.molecule(numMolecules+1).csVec = csVec;
spins.molecule(numMolecules+1).JMatrix = JMatrix;
spins.molecule(numMolecules+1).gmRatio = gmRatio;
spins.molecule(numMolecules+1).spin(1).r = r;
spins.molecule(numMolecules+1).spin(1).rho = rho0;
spins.molecule(numMolecules+1).spin(1).B1 = 1;
spins.molecule(numMolecules+1).spin(1).B0 = 0;
spins.molecule(numMolecules+1).spin(1).linewidth = 5;

