function [movieSpins, movieTimeAxis, fid]=MovieBlochIsochromatsSeq(seq, varargin)
% function [movieSpins, movieTimeAxis, fid]=MovieBlochIsochromatsSeq(seq,spins,numFrames,numPlottedSpins, isPlotRF, isCloseWindow, isInvertedColor, isPlotFID, isPlotSpinTrajectories, maxMovieDuration, spinColorVec)
% 
% [movieSpins, movieTimeAxis, fid]=MovieBlochIsochromatsSeq(seq, ...)
% 
% Creates a Matlab movie object with several isochromats, using the data
% in the input spin structure. Plots the Bloch sphere!
%
% Optional argument-value pairs:
%
%   spins                    Spin structure to be visualized
%   numFrames                Number of frames in movie. If > number of 
%                            steps in pulse, the function will be aborted.
%   numPlottedSpins          Number of spins to be plotted. If > number of 
%                            spins, the function will be aborted.
%   isPlotRF                 If set to true, the RF (red) will be plotted.
%   isCloseWindow            If set to true, the movie window will close 
%                            automatically when the movie ends.
%   isInvertedColor          If set to true, the background will be black
%   isPlotFID                If set to 1, ALL of the spins will be 
%                            simulated and the FID will be calculated 
%                            (Mx+1i*My) at each time point throughout the 
%                            evolution & plotted on the screen.
%   isPlotSpinTrajectories   If set to 1, trajectories of each isochromat 
%                            will be drawn on the surface of the Bloch
%                            sphere.
%   maxMovieDuration         The movie will be played back from t=0 to 
%                            t=maxMovieDuration (in ms).
%   spinColorVec             Cell array of 1x3 vectors of RGB color values 
%                            for each spin in the isochromat. If omitted, 
%                            an automatic matrix will be generated.
%
% Note: you should keep the number of spins under ~10, otherwise the movie
% will seem extremely cluttered.

% ========================================================================
% Parse Input
% ========================================================================

isBoolean = @(x)islogical(x) || isnumeric(x) && all(x(:)==0 | x(:)==1);

p = inputParser;
p.addRequired('seq', @(x) IsSequence(x));
p.addOptional('spins', InitSpinsRelax(0, 1, 1, [0;0;1], 1e6, 1e6, 1));
p.addOptional('numFrames', 0);
p.addOptional('numPlottedSpins', 1);
p.addOptional('isPlotRF', false);
p.addOptional('isCloseWindow', false);
p.addOptional('isInvertedColor', false, @(x) isBoolean(x));
p.addOptional('isPlotFID', false, @(x) isBoolean(x));
p.addOptional('isPlotSpinTrajectories', false, @(x) isBoolean(x));
p.addOptional('maxMovieDuration', CalcSeqTotalTime(seq), @(x) isnumeric(x));
p.addOptional('title', '', @(x) ischar(x));
p.addOptional('spinColorVec', []);

p.parse(seq, varargin{:});

spins = p.Results.spins;
numFrames = p.Results.numFrames;
numPlottedSpins = p.Results.numPlottedSpins;
isPlotRF = p.Results.isPlotRF;
isCloseWindow = p.Results.isCloseWindow;
isInvertedColor = p.Results.isInvertedColor;
isPlotFID = p.Results.isPlotFID;
isPlotSpinTrajectories = p.Results.isPlotSpinTrajectories;
maxMovieDuration = p.Results.maxMovieDuration;
spinColorVec = p.Results.spinColorVec; 
movieTitle = p.Results.title;

if numFrames==0, numFrames = 1; end

% ========================================================================
% Simulate time evolution under supplied sequence
% ========================================================================


% Vector containing the plotted spins' numberings. For example, if the same
% has 100 spins, and 10 are plotted, this vector will have the form:
%    1    12    23    34    45    56    67    78    89   100
% It may be non equi-spaced due to the constraint first and last spins 
% should be included.
numSpins = numel(spins);
if numPlottedSpins > numSpins
    disp('Too many plotted spins requested.');
    disp(['(',num2str(numPlottedSpins),' requested, ', num2str(numSpins),' in sample.)']);
    fprintf('Setting number of plotted spins to %d\n', numSpins);
    numPlottedSpins = numSpins;
end
plottedSpinsVec = floor(linspace(1,numSpins, numPlottedSpins));

% Compute the evolution of the magnetization under the application of the
% given sequence. Extract relevant evolution.
numReps = 1; % The system will not be driven into equilibrium by repeating the sequence multiple times
if isPlotFID % Simulate ALL spins
    for idxCurrentSpin=1:numSpins
        [Mx{idxCurrentSpin}, My{idxCurrentSpin}, Mz{idxCurrentSpin}, timeAxis, B1Amp, B1Phase] = ApplySequenceDiagnostics(seq, numReps, spins(idxCurrentSpin).cs, spins(idxCurrentSpin).r, spins(idxCurrentSpin).M, spins(idxCurrentSpin).T1, spins(idxCurrentSpin).T2, spins(idxCurrentSpin).M0);
    end;
else % Simulate only those spins that are going to be plotted
    for curPlottedSpin=1:numPlottedSpins
        idxCurrentSpin = plottedSpinsVec(curPlottedSpin);
        [Mx{curPlottedSpin}, My{curPlottedSpin}, Mz{curPlottedSpin}, timeAxis, B1Amp, B1Phase] = ApplySequenceDiagnostics(seq, numReps, spins(idxCurrentSpin).cs, spins(idxCurrentSpin).r, spins(idxCurrentSpin).M, spins(idxCurrentSpin).T1, spins(idxCurrentSpin).T2, spins(idxCurrentSpin).M0);
    end;
end

% Find the maximal frame, as dictated by maxMovieDuration
maxTimeIdx = find(timeAxis>=maxMovieDuration, 1, 'first');
if isempty(maxTimeIdx)
    maxTimeIdx = numel(timeAxis);
end
if numFrames > maxTimeIdx 
    disp('Too many frames requested. ');
    disp(['(',num2str(numFrames),' requested. There are only ', num2str(maxTimeIdx),' steps in requested time window.)']);
    fprintf('Setting number of frames to max.\n');
    numFrames = maxTimeIdx;
end

% Create vector 1 .. num of frames. 
frameVec = floor(linspace(1, maxTimeIdx, numFrames));


movieTimeAxis = timeAxis(frameVec);
fid = zeros(1, numFrames);
if isPlotFID
    for idxSpin=1:numSpins
        magTimeEvol{idxSpin}.Mx = Mx{idxSpin}(frameVec);
        magTimeEvol{idxSpin}.My = My{idxSpin}(frameVec);
        magTimeEvol{idxSpin}.Mz = Mz{idxSpin}(frameVec);
        fid = fid + magTimeEvol{idxSpin}.Mx + 1i*magTimeEvol{idxSpin}.My;
    end
else
    for curPlottedSpin=1:numPlottedSpins
        magTimeEvol{curPlottedSpin}.Mx = Mx{curPlottedSpin}(frameVec);
        magTimeEvol{curPlottedSpin}.My = My{curPlottedSpin}(frameVec);
        magTimeEvol{curPlottedSpin}.Mz = Mz{curPlottedSpin}(frameVec);
        fid = fid + magTimeEvol{curPlottedSpin}.Mx + 1i*magTimeEvol{curPlottedSpin}.My;
    end;
end

spinColor = [0 0 1];   % Blue
RFColor   = [1 0 0];   % Red
% Assign different colors to the different plotted spins.
if isempty(spinColorVec)
    for curPlottedSpin=1:numPlottedSpins
        spinColorVec{curPlottedSpin} = spinColor - spinColor*(numPlottedSpins - curPlottedSpin)/numPlottedSpins;
    end
end

if isInvertedColor
    for curPlottedSpin=1:numPlottedSpins
        spinColorVec{curPlottedSpin} = [1 1 1] - spinColorVec{curPlottedSpin};
    end
end



% Plot the 3D Bloch Sphere
hh=figure('Units', 'Pixels', 'Position', [300 300 800 600]);
[sphereCenter, scaleFactor] = PlotBloch(isInvertedColor, isPlotFID);
set(gca,'XTick',[]);
set(gca,'YTick',[]);
set(gca,'ZTick',[]);
if isInvertedColor
    set(gcf,'color','black');
    set(gca,'color','black');
else
    set(gcf,'color','white');
    set(gca,'color','white');
end
set(gca,'LineStyleOrder','--');
set(gca,'Visible','off');
set(gca,'Units','Pixels', 'Position',[0 0 800 600]); % Make the axes occupy the whole figure
QQ = get(hh,'Position');

hText = text(0, 0, 1.3, movieTitle, 'HorizontalAlignment','center');
fprintf('Position: \n');
hText.Position
fprintf('Extent: \n');
get(hText)

% Compute B1(t), in kHz - just the RF (without the z component)
rf    = [B1Amp.*cos(B1Phase); B1Amp.*sin(B1Phase); 0.*zeros(1,length(B1Phase))];
rfMax = max(max(abs(rf)));
rf    = rf./rfMax*scaleFactor; % Normalize to [0 scaleFactor]

scaledTimeAxis = (movieTimeAxis./max(movieTimeAxis) - 0.5)*2;
scaledFID = fid./max(abs(fid));
scaledRF = B1Amp(frameVec)./max(abs(B1Amp(frameVec)));



% Create movie
for curFrame = 1:numFrames
    % Plot spins
    for curPlottedSpin=1:numPlottedSpins
        if isPlotFID
            idxCurSpin = plottedSpinsVec(curPlottedSpin);
        else
            idxCurSpin = curPlottedSpin; 
        end
        M = [magTimeEvol{idxCurSpin}.Mx(curFrame), ... 
             magTimeEvol{idxCurSpin}.My(curFrame), ... 
             magTimeEvol{idxCurSpin}.Mz(curFrame)];
        magM = norm(M); 
        M = M.*scaleFactor; 
        spinArrowSize = 0.015/magM;
        q1(curPlottedSpin)=arrowPlot(sphereCenter+[0 0 0],sphereCenter+M,spinColorVec{curPlottedSpin},spinArrowSize);
        if isPlotSpinTrajectories
            q5(curPlottedSpin) = line(sphereCenter+magTimeEvol{curPlottedSpin}.Mx(1:curFrame)*scaleFactor, ...
                                      sphereCenter+magTimeEvol{curPlottedSpin}.My(1:curFrame)*scaleFactor, ...
                                      sphereCenter+magTimeEvol{curPlottedSpin}.Mz(1:curFrame)*scaleFactor, ...
                                      'Color', spinColorVec{curPlottedSpin}, 'LineStyle', '-');
        end
    end;
    % Plot RF
    if isPlotRF == true
        q2=arrowPlot(sphereCenter+[0 0 0],sphereCenter+rf(:,frameVec(curFrame))',RFColor,0.015);
    end;
    % Plot the FID
    if isPlotFID 
        q3RF = line(zeros(1,curFrame), scaledTimeAxis(1:curFrame), real(scaledRF(1:curFrame)).*0.09 - 0.55, 'LineStyle', '-', 'LineWidth', 2, 'Color', 'red'); 
        q3Re = line(zeros(1,curFrame), scaledTimeAxis(1:curFrame), real(scaledFID(1:curFrame)).*0.09 - 0.77, 'LineStyle', '-', 'LineWidth', 2, 'Color', 'blue'); 
        q3Im = line(zeros(1,curFrame), scaledTimeAxis(1:curFrame), imag(scaledFID(1:curFrame)).*0.09 - 1.0, 'LineStyle', '-', 'LineWidth', 2, 'Color', 'magenta'); 
    end
    % Plot isochromat trajectories
    movieSpins(curFrame) = getframe(hh,[1 1 QQ(3) QQ(4)]);
    % Delete plotted objects to free up memory
    if ((curFrame~=numFrames) || (isCloseWindow))
        for curPlottedSpin=1:numPlottedSpins
            delete(q1(curPlottedSpin));
            clear q1(curPlottedSpin)
        end;
        if isPlotSpinTrajectories
            for curPlottedSpin=1:numPlottedSpins
                delete(q5(curPlottedSpin));
                clear q5(curPlottedSpin)
            end
        end
        if isPlotRF == true
            delete(q2);
            clear q2
        end;
        if isPlotFID 
            delete(q3Re);
            delete(q3RF);
            delete(q3Im);
            clear q3;
            clear q3RF;
        end
    end
end
if isCloseWindow, delete(hh); end

