function pulseHeader = PulseExportSiemensInclude(pulse, filename, pulseBW, flipAngle, comment, minSlice, varName)
% SYNTAX: 
% 
%   pulseHeader = ExportPulseToIncludeFile(pulse, filename, ...
%                    pulseBW, flipAngle, comment, minSlice, varName)
% 
% Description: exports a given pulse structure (sans gradients) to
% a C++ include (.h) file, which can then be included in IDEA code
% and loaded into an arbitrary pulse structure. 
%
% Inputs:
%
% Variable Name   Units    Description
% pulse           -        Input RF pulse structure having the following
%                          fields:
%                          pulse.tp - duration of pulse, in ms
%                          pulse.RFamp - amplitude envelope (kHz)
%                          pulse.RFphase - phase of pulse as func. of time
%                          (radians)
%                          pulse.Gx - amplitude of x-gradient (in kHz/mm)
%                          pulse.Gy - amplitude of y-gradient (in kHz/mm)
%                          pulse.Gz - amplitude of z-gradient (in kHz/mm)
%                          Gx, Gy, Gz, RFamp, RFphase are all arrays with
%                          the same number of elements.
% filename        -        Output filename (including directory!) [1]
% pulseBW         kHz      Bandwidth of pulse, in kHz. Necessary for 
%                          calibrating reference gradient.
% flipAngle       deg.     Flip angle of pulse, in radians. Be sure to set
%                          the same flip angle in the Siemens sequence, using:
%                          myExternalPulse.setFlipAngle(flipAngle in degrees)
% [comment]       -        Comment to be added. Can be left blank (i.e.: ''),
%                          in which case 'Just a pulse' will be used. Same
%                          if omitted.
% [minSlice]      mm       Minimum excitable slice. If omitted, will be set
%                          to 1 mm [2]
% [varName]       -        String, used to define the variables in the
%                          include file. If omitted, the filename will be
%                          used (without the extension)[3] 
%
% [1] The filename will always have a .h extension (if omitted, or another
%     is provided, a .h will be placed automatically instead).
%
% [2] A seemingly "harmless" quantity, but one that should be set by the
%     designer for the following reason: the pulse has a certain stepsize
%     (dwell time), which determines the overall affected bandwidth:
%     FOVBW = 1/(dwell time)
%     Beyond that, the spectral pattern of the pulse will replicate
%     itself. Thus, a pulse designed to excite a 1 cm slice with a 10 cm
%     FOV will excite a 1 mm slice with a 1 cm FOV if the gradient is 
%     multiplied by 10. As a result, if the object is 10 cm long, 
%     artifacts will arise by going below a 1 cm slice.
%
% [3] The .h file will have the pulse's attributes defined as static
%     variables. For example, suppose varName = 'myPulse'. Then
%     static double myPulseAmp = {0.0 0.1 0.4 .... };
%     static double myPulsePhase = {0.0 180.0 90.0 ...};  // in deg.
%     static double myPulseAmpInt = 42.1; 
%     static double myPulseAbsInt = ...
%     and so forth.

inputPulse = pulse;

% Compute BW automatically
[BWExcBaseMz, BWExcFWHMMz]   = CalcPulseBW(pulse, 0.001, 'excitation', 'mz');
[BWRefBaseMz, BWRefFWHMMz]   = CalcPulseBW(pulse, 0.001, 'refocusing', 'mz');
[BWExcBaseMxy, BWExcFWHMMxy] = CalcPulseBW(pulse, 0.001, 'excitation', 'mxy');
[BWRefBaseMxy, BWRefFWHMMxy] = CalcPulseBW(pulse, 0.001, 'refocusing', 'mxy');

[BWExcBaseMz2, BWExcFWHMMz2]   = CalcPulseBW(pulse, 0.01, 'excitation', 'mz');
[BWRefBaseMz2, BWRefFWHMMz2]   = CalcPulseBW(pulse, 0.01, 'refocusing', 'mz');
[BWExcBaseMxy2, BWExcFWHMMxy2] = CalcPulseBW(pulse, 0.01, 'excitation', 'mxy');
[BWRefBaseMxy2, BWRefFWHMMxy2] = CalcPulseBW(pulse, 0.01, 'refocusing', 'mxy');


% Check filename extension. Remove if present. Append .h extension.
filenameNoExt = RemoveExtensionFromFilename(filename);
filename = [filenameNoExt,'.h'];

if nargin<6
    minSlice = 1;
end

if nargin<7
    varName = filenameNoExt;
end

% Check comment is not empty
if nargin<5
    comment = 'No user supplied comment.';
else
    if isempty(comment)
        comment = 'No user supplied comment.';
    end
end

% Convert deg. --> rad.
flipAngle = flipAngle/360*2*pi;

% Number of time steps of RF pulse
numSteps = length(pulse.RFamp);

% Rescale input pulse. 
% Magnitude is scaled to [0 1], Phase is scaled mod. 2 pi.
maxAmp = max(abs(pulse.RFamp));  % in kHz
if (maxAmp == 0)
    maxAmp = 1;
end
pulse.RFamp = pulse.RFamp./maxAmp;
% If amplitude is negative for any reason, make it positive and add pi to
% the phase
pulse.RFphase = pulse.RFphase + ((1-sign(pulse.RFamp))/2)*pi;
pulse.RFamp = abs(pulse.RFamp);
% Make sure phaes is between [0, 2*pi], as demanded by Siemens
pulse.RFphase = mod(pulse.RFphase, 2*pi);

% Compute necessary integrals for Siemens .PTA file
absIntegral = sum(pulse.RFamp);
powerIntegral = sum(pulse.RFamp.^2);

% Compute amplitude integral. See "External RF Pulses.doc" 
% in the "IDEA - Educational\My Notes" folder for more details.
dwellTime = pulse.tp/numSteps;
refAmp = 0.5;  % Amplitude of reference pulse is 0.5 kHz
ampIntegral = (flipAngle/pi) * (1/dwellTime) * (refAmp/maxAmp);

% Compute the reference gradient.
refGrad = ComputeReferenceGradientSiemens(pulse.tp, pulseBW);

% Open file
fid = fopen(filename,'w');

% Create external file header
fprintf(fid,'// =------------------=\n');
fprintf(fid,'// Arbitrary pulse file\n');
fprintf(fid,'// =------------------=\n');
fprintf(fid,'//\n');
fprintf(fid,'// General statistics (as exported): \n');
fprintf(fid,'//    Duration:     %.3f ms\n', inputPulse.tp);
fprintf(fid,'//    Max B1:       %.3f kHz\n', max(inputPulse.RFamp));
fprintf(fid,'//    Num. Steps:   %d\n', numSteps);
fprintf(fid,'//    SAR*:         %.3f\n', CalcSAR(inputPulse, 'ref'));
fprintf(fid,'//    Ref. Grad:    %.3f mT/m (based on %.3f kHz BW) \n', refGrad, pulseBW);
fprintf(fid,'//    Ref. Grad:    %.3f kHz/mm \n', refGrad*GetGyromagneticRatio('1h')/1000);
fprintf(fid,'//    Flip Angle:   %.1f deg. \n', flipAngle/pi*180);
fprintf(fid,'// Automatically computed BWs, assuming an excitation pulse, with a threshold of 0.001: \n');
fprintf(fid,'//    Mz, Base:     %.3f kHz (ref Grad: %.3f mT/m)\n', BWExcBaseMz, ComputeReferenceGradientSiemens(pulse.tp,  BWExcBaseMz)); 
fprintf(fid,'//    Mz, FWHM:     %.3f kHz (ref Grad: %.3f mT/m)\n', BWExcFWHMMz, ComputeReferenceGradientSiemens(pulse.tp,  BWExcFWHMMz)); 
fprintf(fid,'//    Mxy, Base:    %.3f kHz (ref Grad: %.3f mT/m)\n', BWExcBaseMxy, ComputeReferenceGradientSiemens(pulse.tp, BWExcBaseMxy)); 
fprintf(fid,'//    Mxy, FWHM:    %.3f kHz (ref Grad: %.3f mT/m)\n', BWExcFWHMMxy, ComputeReferenceGradientSiemens(pulse.tp, BWExcFWHMMxy)); 
fprintf(fid,'// Automatically computed BWs, assuming a refocusing pulse, with a threshold of 0.001: \n');
fprintf(fid,'//    Mz, Base:     %.3f kHz (ref Grad: %.3f mT/m)\n', BWRefBaseMz, ComputeReferenceGradientSiemens(pulse.tp,  BWRefBaseMz)); 
fprintf(fid,'//    Mz, FWHM:     %.3f kHz (ref Grad: %.3f mT/m)\n', BWRefFWHMMz, ComputeReferenceGradientSiemens(pulse.tp,  BWRefFWHMMz)); 
fprintf(fid,'//    Mxy, Base:    %.3f kHz (ref Grad: %.3f mT/m)\n', BWRefBaseMxy, ComputeReferenceGradientSiemens(pulse.tp, BWRefBaseMxy)); 
fprintf(fid,'//    Mxy, FWHM:    %.3f kHz (ref Grad: %.3f mT/m)\n', BWRefFWHMMxy, ComputeReferenceGradientSiemens(pulse.tp, BWRefFWHMMxy)); 
fprintf(fid,'// Automatically computed BWs, assuming an excitation pulse, with a threshold of 0.01: \n');
fprintf(fid,'//    Mz, Base:     %.3f kHz (ref Grad: %.3f mT/m)\n', BWExcBaseMz2, ComputeReferenceGradientSiemens(pulse.tp,  BWExcBaseMz2)); 
fprintf(fid,'//    Mz, FWHM:     %.3f kHz (ref Grad: %.3f mT/m)\n', BWExcFWHMMz2, ComputeReferenceGradientSiemens(pulse.tp,  BWExcFWHMMz2)); 
fprintf(fid,'//    Mxy, Base:    %.3f kHz (ref Grad: %.3f mT/m)\n', BWExcBaseMxy2, ComputeReferenceGradientSiemens(pulse.tp, BWExcBaseMxy2)); 
fprintf(fid,'//    Mxy, FWHM:    %.3f kHz (ref Grad: %.3f mT/m)\n', BWExcFWHMMxy2, ComputeReferenceGradientSiemens(pulse.tp, BWExcFWHMMxy2)); 
fprintf(fid,'// Automatically computed BWs, assuming a refocusing pulse, with a threshold of 0.01: \n');
fprintf(fid,'//    Mz, Base:     %.3f kHz (ref Grad: %.3f mT/m)\n', BWRefBaseMz2, ComputeReferenceGradientSiemens(pulse.tp,  BWRefBaseMz2)); 
fprintf(fid,'//    Mz, FWHM:     %.3f kHz (ref Grad: %.3f mT/m)\n', BWRefFWHMMz2, ComputeReferenceGradientSiemens(pulse.tp,  BWRefFWHMMz2)); 
fprintf(fid,'//    Mxy, Base:    %.3f kHz (ref Grad: %.3f mT/m)\n', BWRefBaseMxy2, ComputeReferenceGradientSiemens(pulse.tp, BWRefBaseMxy2)); 
fprintf(fid,'//    Mxy, FWHM:    %.3f kHz (ref Grad: %.3f mT/m)\n', BWRefFWHMMxy2, ComputeReferenceGradientSiemens(pulse.tp, BWRefFWHMMxy2)); 
fprintf(fid,'// * - Relative to a 1 ms pi-pulse.\n');
fprintf(fid,'//\n');
fprintf(fid,'// User comment: %s\n', comment);
fprintf(fid,'//\n');
fprintf(fid,'// Hint: do not forget to include these libraries and definitions: \n');
fprintf(fid,'//    #include "MrServers\\MrMeasSrv\\SeqIF\\libRT\\libRT.h"\n');
fprintf(fid,'//    #include "MrServers\\MrMeasSrv\\SeqFW\\libSSL\\SSL_local.h"\n');
fprintf(fid,'//    static sSample %sPulseArray[%d];\n', varName, numSteps);
fprintf(fid,'\n');
fprintf(fid,'float %sRefGrad = %0.4f;\n', varName, refGrad);
fprintf(fid,'float %sMinSlice = %.1f;\n', varName, minSlice);
fprintf(fid,'float %sMaxSlice = 200.0;\n', varName);
fprintf(fid,'float %sAmpInt = %.4f;\n', varName, ampIntegral);
fprintf(fid,'float %sPowerInt = %.4f;\n', varName, powerIntegral);
fprintf(fid,'float %sAbsInt = %.4f;\n', varName, absIntegral);
fprintf(fid,'\n');
fprintf(fid,'\n');
for idx=1:numSteps
    fprintf(fid,'%sPulseArray[%d].flAbs = float(%.6f);    %sPulseArray[%d].flPha = float(%.6f);\n', varName, idx-1, pulse.RFamp(idx), varName, idx-1, pulse.RFphase(idx));
end

% Save data to output variable
pulseHeader.comment = comment;
pulseHeader.refGrad = refGrad;
pulseHeader.refGradKHzmm = refGrad*GetGyromagneticRatio('1h')/1000;
pulseHeader.ampIntegral = ampIntegral;
pulseHeader.powerIntegral = powerIntegral;
pulseHeader.absIntegral = absIntegral;
pulseHeader.maxGradSlewRate = CalcMaxGradSlewRate(inputPulse);
pulseHeader.maxRFSlewRate = CalcMaxRFSlewRate(inputPulse);
pulseHeader.maxRFSlewRate = CalcSAR(inputPulse);


% Close file.
fclose(fid);