function [lambda,j,fm] = reconstructCurrents1D(mu_0,h,d,px,b,lambdaArr,imposeRefBC,plotF,method)
% Reconstruct one-dimensional current density from noisy measurements of
% its magnetic field using Tikhonov regularization.
%   Inputs: * mu_0 - permeability of free space. This value determines the
%           units assumed for all other quantities.
%           * h - measurement height above surface of sample.
%           * d - thickness of sample.
%           * px - length of a pixel on measurement grid.
%           * b - measured data vector.
%           * lambdaArr - array of values for Tikhonov regularization
%           parameter for initial sampling of a parameter-choice function.
%           This array should be dense enough to locate local minima in the
%           parameter-choice function but sparse enough to not slow down
%           the computation too much. If unsure, leave empty to choose the
%           default value, lambdaArr = logspace(-15,15,1e4).
%           * imposeRefBC - boolean option specifying whether to impose
%           reflexive boundary conditions on the solution. Set
%           imposeRefBC=1 if there is strong suspicion that currents
%           outside the measurement window significantly contribute to the
%           measured field. Set imposeRefBC=0 otherwise.
%           * plotF - boolean option specifying whether to plot the
%           parameter-choice function versus lambdaArr. Set plotF=1 to
%           plot, or plotF=0 otherwise.
%           * method - string specifying which parameter-choice method to
%           use. method can be one of four strings - 'SS', 'DF', 'GCV' or
%           'SURE'. If method is unspecified or left empty, method='SS'.
% Outputs:  * lambda - chosen value of the Tikhonov regularization
%           parameter.
%           * j - the reconstructed one-dimensional current density.
%           * fm - value of the parameter-choice function at its minimum.

if (exist('method','var')~=1) || isempty(method)
    method = 'SS';
end

A = generateTomoMatrix1D(mu_0,h,d,px,length(b),imposeRefBC); %generate coefficient matrix
% generate discrete Laplacian:
lapVec = [1,-2,1]; zeroVec1 = zeros(size(A,2),1); zeroVec1(1:3)=lapVec;
L = sparse(toeplitz([1;zeros(size(A,2)-3,1)],zeroVec1.'));

[V,U,X,C,S] = gsvd(full(L),full(A)); % A = U*S*X', L=V*C*X'
Y = inv(X');
n = size(A,2); p = size(L,1);
r = size(L,2) - rank(full(L));
q = rank(full(A))+rank(full(L)) - size(A,2);
sigma = diag(S(r+1:r+q,r+1:r+q)); % obtain {\sigma_k} from the diagonal of the appropriate submatrix of S
mu = diag(C(p+r-n+1:p+r-n+q,r+1:r+q)); % obtain {\mu_k} from the diagonal of the appropriate submatrix of C
gamma = sigma./mu; %obtain {\gamma_k} = {\sigma_k/\mu_k}
beta = U'*b;
[k0,s2] = estPicardParP(beta,r,q,1e-2);
x_lam = @(l) Y(:,1:r)*beta(1:r)+Y(:,r+1:r+q)*(gamma.^2./(gamma.^2+l^2).*beta(r+1:r+q)./sigma); %define Tikhonov solution of A*x=b

switch method %invert:
    case 'SS'
        [lambda,j,fm] = solveSSProblem(x_lam,gamma,beta,k0,s2,r,q,lambdaArr,plotF); %SS
    case 'DF'
        [lambda,j,fm] = solveDFAproblem(x_lam,beta,gamma,k0,r,q,lambdaArr,plotF); %DFA
    case 'GCV'
        [lambda,j,fm] = solveGCVproblem(x_lam,beta,gamma,r,q,lambdaArr,plotF); %GCV
    case 'SURE'
        [lambda,j,fm] = solveSUREproblem(x_lam,beta,gamma,s2,r,q,lambdaArr,plotF); %SURE
    otherwise
        error('Unrecognized method')
end