function [L1,L2] = generateSGolayDiffFilter(sz,N,F,imposeZeroBC)
% Generate Savitsky-Golay differentiation matrices.
% Inputs:   * sz - size of data to differentiate such that sz=size(data).
%           * N - order of polynomial to fit for smoothing differentiation
%           * F - window length for polynomial fit. Must be odd and larger 
%           than polynomial order N.
%           * imposeZeroBC - boolean flag for imposing zero boundary
%           conditions after differentiation. Set imposeZeroBC=1 to impose
%           zero b.c., and imposeZeroBC=0 to impose periodic b.c.
%           compatible with FFT.
% Outputs:  * L1, L2 - Savitsky-Golay differentiation matrices for rows and
%           columns, respectively. To apply these to a matrix M, use M' = L1*M*L2.'.

if floor(F/2)==F/2
    error('Window length F must be odd')
end
if N>=F
    error('Order of polynomial N must be smaller than window size F')
end

[~,y] = sgolay(N,F); %generate filter coefficients
HalfWin  = ((F+1)/2) -1; %half the window length

% form rows and columns of differentiation matrices:
vecC1 = [flipud(y(1:HalfWin+1,2));zeros(sz(1)-HalfWin-1,1)]; %for columns
vecC2 = [flipud(y(1:HalfWin+1,2));zeros(sz(2)-HalfWin-1,1)];

vecR1 = [y(HalfWin+1:end,2)' zeros(1,sz(1)-HalfWin-1)]; %for rows
vecR2 = [y(HalfWin+1:end,2)' zeros(1,sz(2)-HalfWin-1)];


if imposeZeroBC %impose zero boundary conditions on derivative
    L1 = toeplitz(vecC1,vecR1);
    L2 = toeplitz(vecC2,vecR2);
    vec1 = [zeros(HalfWin,1);ones(sz(1)-2*HalfWin,1);zeros(HalfWin,1)];
    vec2 = [zeros(HalfWin,1);ones(sz(2)-2*HalfWin,1);zeros(HalfWin,1)];
    Z1 = diag(vec1);Z2 = diag(vec2);
    L1 = Z1*L1;
    L2 = Z2*L2;
else %impose periodic boundary conditions compatible with FFT
    L1 = toeplitz(vecC1,vecR1) + toeplitz(zeros(size(vecC1)),[0,flip(vecC1(2:end).')]) ...
        + toeplitz([0;flip(vecR1(2:end).')],zeros(size(vecR1)));
    L2 = toeplitz(vecC2,vecR2) + toeplitz(zeros(size(vecC2)),[0,flip(vecC2(2:end).')]) ...
        + toeplitz([0;flip(vecR2(2:end).')],zeros(size(vecR2)));
end