function [jx,jy,Gmin,alphaG] = ...
    reconstructCurrentsFFT_direct(mu_0,h,d,px,data,lambdaArr,plotF,imposeRefBC,p)
% Use FFT to reconstruct currents directly from the magnetic field.
%   Inputs: * mu_0 - value of permeability of free space. This value
%           determines the units used throughout.
%           * z - z-coordinate of measurent plane. Note that z = h+d/2
%           where h is height above surface of sample and d is sample
%           thickness.
%           * d - sample thickness.
%           * px - vector representing size of each pixel in pre-defined units such that px(1)
%           is size along rows and px(2) is size along the columns.
%           * data - matrix representing the measured magnetic field data.
%           * lambdaArr - optional array of values for the Tikhonov
%           regularization parameter lambda on which the currents will be
%           initially evaluated to probe their behavior. Set to [] to use
%           the default array logspace(-15,15,1e4).
%           * plotF - a boolean variable (0/false or 1/true) that specifies
%           whether or not to plot the GCV function evaluated on lambdaArr
%           * imposeRefBC - a boolean variable specifying whether or not to 
%           impose reflexive boundary conditions. These should be imposed
%           if there is current flowing outside the measurement window. If
%           unsure, impose anyway.
%           * offs - width of reflection of the data for the reflexive
%           boundary conditions. If left empty, offs=size(data) (i.e., we
%           use the whole reflection of the data).
%           * p - percentage of the width and length of the boundary of the
%           image to cut when projecting to leave out edge effects.
%   Outputs: * jx,jy - matrices of the x- and y- components of the
%            reconstructed current distribution.
%            * Gmin - the minimum value of the GCV function.
%            * alphaG - the computed Tikhonov regularization parameter corresponding
%            to the minimum of the GCV function.

% tic
z = h+d/2;
s2 = estimateNoiseK0(data); %estimate the variance of the noise from the data.
szO = size(data);
if imposeRefBC
    offs=size(data);
    
    Dlr = fliplr(data);
    Dud = flipud(data);
    Dxx = flipud(Dlr);
    data = [Dxx,Dud;Dlr,data];
else
    offs = [0,0];
end
if isscalar(px) %assume square pixels
    px=[px,px];
end
if exist('p','var')~=1 || isempty(p)
    p=.1;
end
sz = size(data);
% generate kernels
[K1,K2] = TakeDFTOfK1K2(mu_0,z,d,sz,px); %generate kernels
KK = abs(K1).^2 + abs(K2).^2;

lf = calcLap(sz); %generate Laplacian convolution filter
lf = fft2(lf);
clf = abs(lf).^2;

%Get auxilliary quantity meant to avoid NaNs at (1,1) due to kernels. Analytically, this point should always be zero.
tmp = zeros(size(K1)); 
tmp(1,1)=1;

if ~isempty(lambdaArr)
    lambdaI = lambdaArr;
else
    lambdaI = logspace(-15,15,1e4); % default array of lambda
end

data = data-mean(data(:)); %clean zero frequency value
B = fft2(data);

%get reg. solutions:
jx = @(l) real(ifft2((conj(K1))./(KK + l*clf + tmp).*B));
jy = @(l) real(ifft2((conj(K2))./(KK + l*clf + tmp).*B));

data_rec = @(l) ifft2(K1.*fft2(jx(l))+K2.*fft2(jy(l))); %data reconstructed from the Tikhonov solution
indexat = @(matrix,indices)matrix(indices{1},indices{2});

secOffs = ceil(p*szO); %offset to remove p perents of edges
indsProj = {1+offs(1)+secOffs(1):szO(1)+offs(1)-secOffs(1),1+offs(2)+secOffs(2):szO(2)+offs(2)-secOffs(2)}; %indices for projection for GCV
rhoPP = @(l) norm(indexat(data_rec(0)-data_rec(l),indsProj),'fro')^2; % squared residual norm

noise = 2*randi(2,size(B))-3; %noise vector for monte-carlo estimation of T(lambda)
N = fft2(noise); N_n = indexat(noise,indsProj);
B_n = @(l) indexat(...
    real(ifft2(KK./(KK + l*clf + tmp).*N))...
,indsProj); 
TPP = @(l) sum(sum( abs(N_n).^2-conj(N_n).*B_n(l) ));

G = @(l) rhoPP(l) - 2*s2*TPP(l); %projected GCV function

[Gmin,alphaG] = findGlobalMinV1(lambdaI,G,plotF);

%save currents correspondng to found regularization parameter:
if length(alphaG)>1
    warning('Unique minimizer not found - using minimal')
    alphaG = min(alphaG);
end
jx = jx(alphaG);
jy = jy(alphaG);

if imposeRefBC
    jx = jx(1+offs(1):szO(1)+offs(1),1+offs(2):szO(2)+offs(2));
    jy = jy(1+offs(1):szO(1)+offs(1),1+offs(2):szO(2)+offs(2));
end
