function [Gmin,GPmin,Mmin,alphaG,alphaGP,alphaM] = ...
    reconstructCurrentsFFT_direct_MSD(mu_0,h,d,px,data,lambdaArr,plotF,imposeRefBC,p,JxN,JyN)

% Use FFT to reconstruct currents directly from magnetic field and compare the performances 
% of the standard GCV, the projected GCV and the optimal solution minimizing the MSD. 
%   Inputs: * mu_0 - value of permeability of free space. This value
%           determines the units used throughout.
%           * h - height above surface of sample
%           * d - sample thickness.
%           * px - vector representing size of each pixel in pre-defined units such that px(1)
%           is size along rows and px(2) is size along the columns.
%           * data - matrix representing the measured magnetic field data.
%           * lambdaArr - optional array of values for the Tikhonov
%           regularization parameter lambda on which the currents will be
%           initially evaluated to probe their behavior. Set to [] to use
%           the default array logspace(-15,15,1e4).
%           * plotF - a boolean variable (0/false or 1/true) that specifies
%           whether or not to plot the GCV function evaluated on lambdaArr
%           * imposeRefBC - a boolean variable specifying whether or not to 
%           impose reflexive boundary conditions. These should be imposed
%           if there is current flowing outside the measurement window. If
%           unsure, impose anyway.
%           * offs - width of reflection of the data for the reflexive
%           boundary conditions. If left empty, offs=size(data) (i.e., we
%           use the whole reflection of the data).
%           * p - percentage of the width and length of the boundary of the
%           image to cut when projecting to leave out edge effects.
%           * JxN,JyN - true x- and y- components of the current density
%   Outputs: * Gmin - MSD obtained by minimizing the standard GCV
%            function.
%            * GPmin - MSD obtained by minimizing the GCV function with
%            projections.
%            * Mmin - minimum value of the MSD representing the optimal
%            solution.
%            * alphaG - the computed Tikhonov regularization parameter corresponding
%            to the minimum of the standard GCV function.
%            * alphaGP - the computed Tikhonov regularization parameter corresponding
%            to the minimum of the projected GCV function.
%            * alphaM - the computed Tikhonov regularization parameter corresponding
%            to the minimum of the MSD function.

s2 = estimateNoiseK0(data); %estimate variance of noise perturbing the data
indexat = @(matrix,indices) matrix(indices{1},indices{2}); %auxilliary function to be used later for projections

if imposeRefBC %impose boundary conditions
    offs=size(data);
    szO = size(data);
    
    Dlr = fliplr(data);
    Dud = flipud(data);
    Dxx = flipud(Dlr);
    data = [Dxx,Dud;Dlr,data];
else
    offs = [0,0];
    szO = size(data);
end

if isscalar(px) %if pixel is square and px is a scalar
    px=[px,px];
end
sz = size(data);

% generate kernels
[K1,K2] = TakeDFTOfK1K2(mu_0,h,d,sz,px);
KK = abs(K1).^2 + abs(K2).^2;

lf = calcLap(sz); %generate Laplacian convolution filter
lf = fft2(lf);
clf = abs(lf).^2;

%Get auxilliary quantity meant to avoid NaNs at (1,1) due to kernels. Analytically, this point should always be zero.
tmp = zeros(size(K1));
tmp(1,1)=1;

if ~isempty(lambdaArr)
    lambdaI = lambdaArr;
else
    lambdaI = logspace(-15,15,1e4); % array of lambda
end

data = data-mean(data(:)); %clean zero frequency value
B = fft2(data);

%get reg. solutions:
jx = @(l) real(ifft2((conj(K1))./(KK + l*clf + tmp).*B));
jy = @(l) real(ifft2((conj(K2))./(KK + l*clf + tmp).*B));

auxRhoC = @(l) l*clf(:).*B(:)./(KK(:) + l*clf(:) + tmp(:)); %analytic formula for squared residual norm
rho = @(l) norm(auxRhoC(l))^2/numel(B); %divide by extra factor of numel(B) from FFT2

data_rec = @(l) ifft2(K1.*fft2(jx(l))+K2.*fft2(jy(l))); %reconstructed data

secOffs = ceil(p*szO); %width of boundary layer to remove from P-SURE estimate
indsProj = {1+offs(1)+secOffs(1):szO(1)+offs(1)-secOffs(1),1+offs(2)+secOffs(2):szO(2)+offs(2)-secOffs(2)}; %indices for projections for P-SURE
rhoP = @(l) norm(indexat(data_rec(0)-data_rec(l),indsProj),'fro')^2;

auxTl = @(l) l*clf./(KK + l*clf+ tmp); %analytic formula for trace of regularization operator.
T = @(l) sum(sum(auxTl(l))); %divide by extra factor of numel(B) from FFT2

noise = 2*randi(2,size(B))-3; %generate random matrix with entries +-1 with 50% probability 
N = fft2(noise); N_n = indexat(noise,indsProj);
B_n = @(l) indexat(...
    real(ifft2(KK./(KK + l*clf + tmp).*N))...
,indsProj);
TP = @(l) sum(sum( abs(N_n).^2-conj(N_n).*B_n(l) ));

G = @(l) rho(l)./((T(l)).^(2)); %Define the GCV function
GP = @(l) rhoP(l) - 2*s2*TP(l); %projected SURE function

if imposeRefBC %if data was reflected, discrad reflections in computing MSD
    indsProjMSD = {1+offs(1):szO(1)+offs(1),1+offs(2):szO(2)+offs(2)};
    MSD = @(l) (norm(indexat(jx(l),indsProjMSD)-JxN,'fro')^2+norm(indexat(jy(l),indsProjMSD)-JyN,'fro')^2)...
        /(norm(JxN,'fro')^2+norm(JyN,'fro')^2);
else
    MSD = @(l) (norm(jx(l)-JxN,'fro')^2+norm(jy(l)-JyN,'fro')^2)/(norm(JxN,'fro')^2+norm(JyN,'fro')^2);
end

%Find global minimum and corresponding lambda:
[~,alphaG] = findGlobalMinV1(lambdaI,G,plotF); 
[~,alphaGP] = findGlobalMinV1(lambdaI,GP,plotF); 
[Mmin,alphaM] = findGlobalMinV1(lambdaI,MSD,plotF);

Gmin = MSD(alphaG);
GPmin = MSD(alphaGP);
