function [jx,jy,Gmin,alphaG] = ...
    reconstructCurrentsFFT_stream(mu_0,h,d,px,data,lambdaArr,plotF,imposeRefBC,p)

% Use FFT to reconstruct currents using the auxilliary function g(x,y) such that j=curl(g \hat{z}). 
%   Inputs: * mu_0 - value of permeability of free space. This value
%           determines the units used throughout.
%           * z - z-coordinate of measurent plane. Note that z = h+d/2
%           where h is height above surface of sample and d is sample
%           thickness.
%           * d - sample thickness.
%           * px - vector representing size of each pixel in pre-defined units such that px(1)
%           is size along rows and px(2) is size along the columns.
%           * data - matrix representing the measured magnetic field data.
%           * lambdaArr - optional array of values for the Tikhonov
%           regularization parameter lambda on which the currents will be
%           initially evaluated to probe their behavior. Set to [] to use
%           the default array logspace(-15,15,1e4).
%           * plotF - a boolean variable (0/false or 1/true) that specifies
%           whether or not to plot the GCV function evaluated on lambdaArr
%           * imposeRefBC - a boolean variable specifying whether or not to 
%           impose reflexive boundary conditions. These should be imposed
%           if there is current flowing outside the measurement window. If
%           unsure, impose anyway.
%           * offs - width of reflection of the data for the reflexive
%           boundary conditions. If left empty, offs=size(data) (i.e., we
%           use the whole reflection of the data).
%           * p - percentage of the width and length of the boundary of the
%           image to cut when projecting to leave out edge effects.
%   Outputs: * jx,jy - matrices of the x- and y- components of the
%            reconstructed current distribution.
%            * Gmin - the minimum value of the GCV function.
%            * alphaG - the computed Tikhonov regularization parameter corresponding
%            to the minimum of the GCV function.
z = h+d/2;
s2 = estimateNoiseK0(data);
szO = size(data);
if imposeRefBC
    offs=size(data);
    
    Dlr = fliplr(data);
    Dud = flipud(data);
    Dxx = flipud(Dlr);
    data = [Dxx,Dud;Dlr,data];
else
    offs = [0,0];
end
if isscalar(px)
    px=[px,px];
end
if exist('p','var')~=1 || isempty(p)
    p=.1;
end
sz = size(data);
% generate kernels
K = TakeDFTOfKF(mu_0,z,d,sz,px);
KK = (abs(K)).^2;

lf = calcLap(sz);
lf = fft2(lf);
clf = (abs(lf).^2);
tmp = zeros(size(K));
tmp(1,1)=1;

if ~isempty(lambdaArr)
    lambdaI = lambdaArr;
else
    lambdaI = logspace(-15,15,1e4); % array of lambda
end

data = data-mean(data(:)); %clean zero frequency value
B = fft2(data);

%get reg. solutions:
g = @(l) real(ifft2((conj(K))./(KK + l*clf + tmp).*B));

secOffs = ceil(p*szO);
indsProj = {1+offs(1)+secOffs(1):szO(1)+offs(1)-secOffs(1),1+offs(2)+secOffs(2):szO(2)+offs(2)-secOffs(2)};

data_rec = @(l) ifft2(K.*fft2(g(l)));
indexat = @(matrix,indices)matrix(indices{1},indices{2});
rhoPP = @(l) norm(indexat(data_rec(0)-data_rec(l),indsProj),'fro')^2;

noise = 2*randi(2,size(B))-3;
N = fft2(noise); N_n = indexat(noise,indsProj);
B_n = @(l) indexat(...
    ifft2(KK./(KK + l*clf + tmp).*N)...
,indsProj);
TPP = @(l) sum(sum(abs(N_n).^2-conj(N_n).*B_n(l)));

G = @(l) rhoPP(l) - 2*s2*TPP(l);

[L1,L2] = generateSGolayDiffFilter(size(data),2,5,0);

jx = @(l) L1*g(l)/px(1);
jy = @(l) -g(l)*L2'/px(2);

[Gmin,alphaG] = findGlobalMinV1(lambdaI,G,plotF); %Find global minimum of GCV and save only the corresponding lambda

%save currents correspondng to found regularization parameter:
if length(alphaG)>1
    warning('Unique minimizer not found - using minimal')
    alphaG = min(alphaG);
end
jx = jx(alphaG);
jy = jy(alphaG);

if imposeRefBC
    jx = jx(1+offs(1):szO(1)+offs(1),1+offs(2):szO(2)+offs(2));
    jy = jy(1+offs(1):szO(1)+offs(1),1+offs(2):szO(2)+offs(2));
end
% toc