function [Gmin,GPmin,Mmin,alphaG,alphaGP,alphaM] = ...
    reconstructCurrentsFFT_stream_MSD(mu_0,h,d,px,data,lambdaArr,plotF,imposeRefBC,p,JxN,JyN)

% Use FFT to reconstruct currents using the auxilliary function g(x,y) such that j=curl(g \hat{z}) and compare the performances 
% of the standard GCV, the projected GCV and the optimal solution minimizing the MSD. 
%   Inputs: * mu_0 - value of permeability of free space. This value
%           determines the units used throughout.
%           * z - z-coordinate of measurent plane. Note that z = h+d/2
%           where h is height above surface of sample and d is sample
%           thickness.
%           * d - sample thickness.
%           * px - vector representing size of each pixel in pre-defined units such that px(1)
%           is size along rows and px(2) is size along the columns.
%           * data - matrix representing the measured magnetic field data.
%           * lambdaArr - optional array of values for the Tikhonov
%           regularization parameter lambda on which the currents will be
%           initially evaluated to probe their behavior. Set to [] to use
%           the default array logspace(-15,15,1e4).
%           * plotF - a boolean variable (0/false or 1/true) that specifies
%           whether or not to plot the GCV function evaluated on lambdaArr
%           * imposeRefBC - a boolean variable specifying whether or not to 
%           impose reflexive boundary conditions. These should be imposed
%           if there is current flowing outside the measurement window. If
%           unsure, impose anyway.
%           * offs - width of reflection of the data for the reflexive
%           boundary conditions. If left empty, offs=size(data) (i.e., we
%           use the whole reflection of the data).
%           * p - percentage of the width and length of the boundary of the
%           image to cut when projecting to leave out edge effects.
%   Outputs: * Gmin - MSD obtained by minimizing the standard GCV
%            function.
%            * GPmin - MSD obtained by minimizing the GCV function with
%            projections.
%            * Mmin - minimum value of the MSD representing the optimal
%            solution.
%            * alphaG - the computed Tikhonov regularization parameter corresponding
%            to the minimum of the standard GCV function.
%            * alphaGP - the computed Tikhonov regularization parameter corresponding
%            to the minimum of the projected GCV function.
%            * alphaM - the computed Tikhonov regularization parameter corresponding
%            to the minimum of the MSD function.

s2 = estimateNoiseK0(data); %estimate variance of noise perturbing the data
indexat = @(matrix,indices) matrix(indices{1},indices{2}); %auxilliary function to be used later for projections

if imposeRefBC %impose boundary conditions
    offs=size(data);
    szO = size(data);
    
    Dlr = fliplr(data);
    Dud = flipud(data);
    Dxx = flipud(Dlr);
    data = [Dxx,Dud;Dlr,data];
else
    offs = [0,0];
    szO = size(data);
end

if isscalar(px) %if pixel is square and px is a scalar
    px=[px,px];
end
sz = size(data);

% generate kernels
K = TakeDFTOfKF(mu_0,h,d,sz,px);
KK = abs(K).^2;

lf = calcLap(sz); %generate Laplacian convolution filter
lf = fft2(lf);
clf = abs(lf).^2;

%Get auxilliary quantity meant to avoid NaNs at (1,1) due to kernels. Analytically, this point should always be zero.
tmp = zeros(size(K));
tmp(1,1)=1;

if ~isempty(lambdaArr)
    lambdaI = lambdaArr;
else
    lambdaI = logspace(-15,15,1e4); % array of lambda
end

data = data-mean(data(:)); %clean zero frequency value
B = fft2(data);

%get reg. solutions:
g = @(l) real(ifft2((conj(K))./(KK + l*clf + tmp).*B));

auxRho = @(l) l*clf(:).*B(:)./(KK(:) + l*clf(:) + tmp(:));
rho = @(l) norm(auxRho(l))^2/numel(B);

secOffs = ceil(p*szO); %width of boundary layer to remove from P-SURE estimate
indsProj = {1+offs(1)+secOffs(1):szO(1)+offs(1)-secOffs(1),1+offs(2)+secOffs(2):szO(2)+offs(2)-secOffs(2)}; %indices for projections for P-SURE

data_rec = @(l) ifft2(K.*fft2(g(l))); %reconstructed data
rhoP = @(l) norm(indexat(data_rec(0)-data_rec(l),indsProj),'fro')^2;

noise = 2*randi(2,size(B))-3; %generate random matrix with entries +-1 with 50% probability 
N = fft2(noise); N_n = indexat(noise,indsProj);
B_n = @(l) indexat(...
    ifft2(KK./(KK + l*clf + tmp).*N)...
,indsProj);
TP = @(l) sum(sum(abs(N_n).^2-conj(N_n).*B_n(l)));

auxTl = @(l) l*clf(:)./(KK(:) + l*clf(:)+ tmp(:));
T = @(l) sum(auxTl(l))/numel(B);

G = @(l) rho(l)./((T(l)).^2); %GCV function
GP = @(l) rhoP(l) - 2*s2*TP(l); %projected SURE function

[L1,L2] = generateSGolayDiffFilter(size(data),2,5,0); %Savitsky-Golay differentiation filter with polynomial order 2 and window length 5

jx = @(l) L1*g(l)/px(1); %differentiate g(x,y) to get currents 
jy = @(l) -g(l)*L2'/px(2);

if imposeRefBC %if data was reflected, discrad reflections in computing MSD
    indsProjMSD = {1+offs(1):szO(1)+offs(1),1+offs(2):szO(2)+offs(2)};
    MSD = @(l) (norm(indexat(jx(l),indsProjMSD)-JxN,'fro')^2+norm(indexat(jy(l),indsProjMSD)-JyN,'fro')^2)...
        /(norm(JxN,'fro')^2+norm(JyN,'fro')^2);
else
    MSD = @(l) (norm(jx(l)-JxN,'fro')^2+norm(jy(l)-JyN,'fro')^2)/(norm(JxN,'fro')^2+norm(JyN,'fro')^2);
end

%Find global minimum and corresponding lambda
[~,alphaG] = findGlobalMinV1(lambdaI,G,plotF); 
[~,alphaGP] = findGlobalMinV1(lambdaI,GP,plotF);
[Mmin,alphaM] = findGlobalMinV1(lambdaI,MSD,plotF);

Gmin = MSD(alphaG);
GPmin = MSD(alphaGP);