function MSDC = CompareMethods(LotNum)
% Generates the cell array MSDC containing the MSD data of 10 methods
% including optimal one on 3 test problems and 3 noise levels.
%   Inputs:     * LotNum - number of noise realizations to test the methods
%               on for each noise level.
%   Outputs:    * MSDC - a 9-by-10 cell array containing MSD values for 10
%               methods on nine test probblems such that:
%               MSDC(:,1) - optimal method minimizing the MSD
%               MSDC(:,2) - SS method minimizing the PMSE with Picard
%               parameter estimated using the sequence of averages {V(k)}.
%               MSDC(:,3) - SS method minimizing the PMSE with Picard
%               parameter estimated using the Lilliefors test at 99.9%
%               confidence level starting from the beginning of the
%               sequence.
%               MSDC(:,4) - SS method minimizing the PMSE with Picard
%               parameter estimated using Lilliefors test at 95% starting
%               from the end of the sequence.
%               MSDC(:,5) - same as MSDC(:,2) but with the SS method
%               minimizing the MSE.
%               MSDC(:,6) - same as MSDC(:,3) but with the SS method
%               minimizing the MSE.
%               MSDC(:,7) - same as MSDC(:,4) but with the SS method
%               minimizing the MSE.
%               MSDC(:,8) - DF method with Picard parameter estimated as in MSDC(:,2) above.
%               MSDC(:,9) - the GCV method.
%               MSDC(:,10) - the SURE method with the variance estimated
%               with the same algorithm used for the Picard parameter in MSDC(:,2).
%               The output MSDC can be used to plot boxplots using the
%               'generateBoxPlots' function.
% 
% Eitan Levin, Alexander Meltzer,
% Department of Condensed Matter Physics,
% Weizmann Institute of Science,
% 2015-16.

MSDC = cell(9,7);kji=1;

lambdaArr = logspace(-15,5,1e3); %array of lambda values for minimization of functions
bnd=5e-2; %bound for variance function V(k) for estimation of Picard parameter
plotF=0;

for flag=1:3
    switch flag
        case 1
            [A,b_t,x_t] = gravity(1e3); %generate problem
            [L,~] = get_l(1e3,1); %generate 2nd derivative
        case 2
            [A,b_t,x_t] = phillips(1e3);
            [L,~] = get_l(1e3,2); %generate 2nd derivative
        case 3
            [A,b_t,x_t] = heat(1e3); %generate problem
            L = eye(1e3); %generate 2nd derivative
    end
    [V,U,X,C,S] = gsvd(full(L),full(A)); % A = U*S*X', L=V*C*X'
    Y = inv(X');
    n = size(A,2); p = size(L,1);
    r = size(L,2) - rank(full(L));
    q = rank(full(A))+rank(full(L)) - size(A,2);
    sigma = diag(S(r+1:r+q,r+1:r+q)); % obtain {\sigma_k} from the diagonal of the appropriate submatrix of S
    mu = diag(C(p+r-n+1:p+r-n+q,r+1:r+q)); % obtain {\mu_k} from the diagonal of the appropriate submatrix of C
    gamma = sigma./mu; %obtain {\gamma_k} = {\sigma_k/\mu_k}
    
    MSD = @(x_l) norm(x_l-x_t)^2/norm(x_t)^2;
    for noise = [1e-2 1e-3 1e-4]

        ns = sqrt(noise)*max(abs(b_t)); %noise variance

        for jj=1:LotNum %for each noise realization
            %add noise:
            e = ns*randn(size(b_t)); b = b_t+e;
            beta = U'*b;
            [k0_P,s2_P] = estPicardParP(beta,r,q,bnd);
            [k0_ML,s2_ML] = estPicardParML(beta,r,q);
            [k0_OL,~,s_OL] = estimate_Picard_Param(beta); s2_OL = s_OL^2;
            if k0_OL<r+1
                k0_OL = r+1;
            elseif k0_OL>r+q
                k0_OL = r+q;
            end
            if isempty(s2_OL) || isnan(s2_OL)
                s2_OL=0;
            end
            x_lam = @(l) Y(:,1:r)*beta(1:r)+Y(:,r+1:r+q)*(gamma.^2./(gamma.^2+l^2).*beta(r+1:r+q)./sigma); %define Tikhonov solution of A*x=b
            [~,x_SS_P,~] = solveSSProblem(x_lam,gamma,beta,k0_P,s2_P,r,q,lambdaArr,plotF); %SS+P
            [~,x_SS_ML,~] = solveSSProblem(x_lam,gamma,beta,k0_ML,s2_ML,r,q,lambdaArr,plotF); %SS+ML
            [~,x_SS_OL,~] = solveSSProblem(x_lam,gamma,beta,k0_OL,s2_OL,r,q,lambdaArr,plotF); %SS+OL
            
            [~,x_SSM_P,~] = solveSS_MSEProblem(Y,x_lam,beta,gamma,sigma,k0_P,s2_P,r,lambdaArr,plotF); %SS+P on MSE
            [~,x_SSM_ML,~] = solveSS_MSEProblem(Y,x_lam,beta,gamma,sigma,k0_ML,s2_ML,r,lambdaArr,plotF); %SS+ML on MSE
            [~,x_SSM_OL,~] = solveSS_MSEProblem(Y,x_lam,beta,gamma,sigma,k0_OL,s2_OL,r,lambdaArr,plotF); %SS+OL on MSE
            
            [~,x_DFA,~] = solveDFAproblem(x_lam,beta,gamma,k0_P,r,q,lambdaArr,plotF); %DFA
            [~,x_GCV,~] = solveGCVproblem(x_lam,beta,gamma,r,q,lambdaArr,plotF); %GCV
            [~,x_SURE,~] = solveSUREproblem(x_lam,beta,gamma,s2_P,r,q,lambdaArr,plotF); %SURE
            
            
            
            [~,~,MSDC{kji,1}(jj)] = solveMSDProblem(x_lam,x_t,lambdaArr,plotF); %calculate optimal solution using MSD
            
            %Calculate the MSD produced by each method:
            MSDC{kji,2}(jj) = MSD(x_SS_P);
            MSDC{kji,3}(jj) = MSD(x_SS_ML);
            MSDC{kji,4}(jj) = MSD(x_SS_OL);
            MSDC{kji,5}(jj) = MSD(x_SSM_P);
            MSDC{kji,6}(jj) = MSD(x_SSM_ML);
            MSDC{kji,7}(jj) = MSD(x_SSM_OL);
            MSDC{kji,8}(jj) = MSD(x_DFA);
            MSDC{kji,9}(jj) = MSD(x_GCV);
            MSDC{kji,10}(jj) = MSD(x_SURE);
            
        end
        kji = kji+1;
    end
end
MSDC = cellfun(@(a)reshape(a,[],1),MSDC,'UniformOutput',0);
% generateBoxPlots(MSDC) %generate boxplots