function MSDC = CompareOnStandardForm(LotNum)
% Generates the cell array MSDC containing the MSD data of 10 methods
% including optimal one on a large scale image deblurring problem with 
% seperable blur and 3 noise levels.
%   Inputs:     * LotNum - number of noise realizations to test the methods
%               on for each noise level.
%   Outputs:    * MSDC - a 3-by-10 cell array containing MSD values for 10
%               methods on three test probblems such that:
%               MSDC(:,1) - optimal method minimizing the MSD
%               MSDC(:,2) - SS method minimizing the PMSE with Picard
%               parameter estimated using the sequence of averages {V(k)}.
%               MSDC(:,3) - SS method minimizing the PMSE with Picard
%               parameter estimated using the Lilliefors test at 99.9%
%               confidence level starting from the beginning of the
%               sequence.
%               MSDC(:,4) - SS method minimizing the PMSE with Picard
%               parameter estimated using Lilliefors test at 95% starting
%               from the end of the sequence.
%               MSDC(:,5) - same as MSDC(:,2) but with the SS method
%               minimizing the MSE.
%               MSDC(:,6) - same as MSDC(:,3) but with the SS method
%               minimizing the MSE.
%               MSDC(:,7) - same as MSDC(:,4) but with the SS method
%               minimizing the MSE.
%               MSDC(:,8) - DF method with Picard parameter estimated as in MSDC(:,2) above.
%               MSDC(:,9) - the GCV method.
%               MSDC(:,10) - the SURE method with the variance estimated
%               with the same algorithm used for the Picard parameter in MSDC(:,2).
%               The output MSDC can be used to plot boxplots using the
%               'generateBoxPlots' function.
%
% Eitan Levin, Alexander Meltzer,
% Department of Condensed Matter Physics,
% Weizmann Institute of Science,
% 2015-16.

FullProb=tic;
% Load 128 x 128 MRI image:
x_true = double(imread('mri.tif'));
x_t = x_true(:);
% Create 128^2 x 128^2 blurring matrix:
band = 16; sigma = 2;
z1 = [exp(-([0:band-1].^2)/(2*sigma^2)),zeros(1,size(x_true,2)-band)];
Z1 = toeplitz(z1,z1)./sqrt(2*pi*sigma^2);
z2 = [exp(-([0:band-1].^2)/(2*sigma^2)),zeros(1,size(x_true,1)-band)];
Z2 = toeplitz(z2,z2)./sqrt(2*pi*sigma^2);
A = kron(Z1, Z2);
b_t = A*x_t;
r=0; %L=I and is full-rank
q = rank(Z1)*rank(Z2);
% Compute SVD of A
[U1,S1,V1] = svd(Z1);
[U2,S2,V2] = svd(Z2);
U = kron(U1,U2); S = kron(S1,S2); V = kron(V1,V2);
sigma = diag(S); [~,Is] = sort(sigma); Is = flip(Is); sigma = sigma(Is);
U = U(:,Is); V = V(:,Is);

noiseArr = [1e-2,1e-4,1e-6];
lambdaArr = logspace(-15,5,1e3);
MSDC = cell(3,10);
e=5e-2; plotF = 0;
MSD = @(x_l) norm(x_l - x_t)^2/norm(x_t)^2;
for jj=1:length(noiseArr)
    noise = noiseArr(jj);
    for kk=1:LotNum
        b = b_t + sqrt(noise)*max(abs(b_t))*randn(size(b_t));
        beta = U'*b;
        x_lam = @(l) V(:,1:r)*beta(1:r)+V(:,r+1:r+q)*(sigma./(sigma.^2+l^2).*beta(r+1:r+q));
        
%         Estimate the Picard parameter with three methods:
        [k0_P,s2_P] = estPicardParP(beta,r,q,e);
        [k0_ML,s2_ML] = estPicardParML(beta,r,q);
        [k0_OL,~,s_OL] = estimate_Picard_Param(beta); s2_OL = s_OL^2;
        if isempty(s2_OL) || isnan(s2_OL)
            s2_OL=0;
        end
        [~,~,MSDC{jj,1}(kk)] = solveMSDProblem(x_lam,x_t,lambdaArr,plotF);

        [~,x_l,~] = solveSSProblem(x_lam,sigma,beta,k0_P,s2_P,r,q,lambdaArr,plotF); MSDC{jj,2}(kk) = MSD(x_l);
        [~,x_l,~] = solveSSProblem(x_lam,sigma,beta,k0_ML,s2_ML,r,q,lambdaArr,plotF); MSDC{jj,3}(kk) = MSD(x_l);
        [~,x_l,~] = solveSSProblem(x_lam,sigma,beta,k0_OL,s2_OL,r,q,lambdaArr,plotF); MSDC{jj,4}(kk) = MSD(x_l);
        
        [~,x_l,~] = solveSS_MSEProblem(V,x_lam,beta,sigma,sigma,k0_P,s2_P,r,lambdaArr,plotF);MSDC{jj,5}(kk) = MSD(x_l);
        [~,x_l,~] = solveSS_MSEProblem(V,x_lam,beta,sigma,sigma,k0_ML,s2_ML,r,lambdaArr,plotF);MSDC{jj,6}(kk) = MSD(x_l);
        [~,x_l,~] = solveSS_MSEProblem(V,x_lam,beta,sigma,sigma,k0_OL,s2_OL,r,lambdaArr,plotF);MSDC{jj,7}(kk) = MSD(x_l);
        
        [~,x_l,~] = solveDFAproblem(x_lam,beta,sigma,k0_P,r,q,lambdaArr,plotF);MSDC{jj,8}(kk)=MSD(x_l);
        [~,x_l,~] = solveGCVproblem(x_lam,beta,sigma,r,q,lambdaArr,plotF);MSDC{jj,9}(kk)=MSD(x_l);
        [~,x_l,~] = solveSUREproblem(x_lam,beta,sigma,s2_P,r,q,lambdaArr,plotF);MSDC{jj,10}(kk) = MSD(x_l);
    end
end
toc(FullProb)