function [lambda,x_l,fm] = solveSS_MSEProblem(Y,x_lam,beta,gamma,sigma,k0,s2,r,lambdaArr,plotF)
%Applied the SS method for the MSE to a given problem with one of three possible
%algorithms specified by flag for determining the Picard index.
%   Inputs: * Y - the matrix from the GSVD of (A,L) such that A=U*S*inv(Y)
%           and L=V*C*inv(Y).
%           * x_lam - anonymous function returning the regularized solution
%           vector x_lam(lambda) for a value of the regularization
%           parameter lambda
%           * beta - the Fourier coefficients beta=U'*b of the perturbed
%           data with respect to the basis of the matrix U from the GSVD of
%           (A,L) such that A=U*S*inv(Y).
%           * gamma - a vector of the generalized singular values of (A,L)
%           * sigma - a vector of the values {sigma_j} from the diagonal of
%           the matrix S from the GSVD of (A,L) such that A=U*S*inv(Y).
%           * k0 - an estimate of the Picard parameter.
%           * s2 - an estimate of the variance of the noise.
%           * r = size(L,2) - rank(L).
%           * lambdaArr - array of lambda valuesfor initial sampling of
%           function for minimization.
%           * plotF - set plotF=1 to plot the SS function versus the
%           lambda values in lambdaArr.
%   Outputs: * lambda - the chosen Tikhonov regularization parameter.
%            * x_l - the computed solution vecotr.
%            * fm - the value of the SS function at its minimum.
% 
% Eitan Levin, Alexander Meltzer,
% Department of Condensed Matter Physics, 
% Weizmann Institute of Science,
% 2015-16.
tic

%define Tikhonov solution truncated at k0:
x_lam_t = @(l) Y(:,1:r)*beta(1:r) + Y(:,r+1:k0-1)*((gamma(1:k0-1-r).^2./(gamma(1:k0-1-r).^2+l^2)).*beta(r+1:k0-1)./sigma(1:k0-1-r)); 
x_LS = Y(:,1:r)*beta(1:r) + Y(:,r+1:k0-1)*(beta(r+1:k0-1)./sigma(1:k0-1-r)); %truncated least-squares solution
Ymag = sum(Y(:,r+1:k0-1).^2,1).'; %vector of the norms of the columns of Y up to k0.
rho = @(l) norm(x_LS-x_lam_t(l))^2; %residual norm truncated at k0
R = @(l) s2*sum(Ymag.*(l^2./(gamma(1:k0-1-r).^2+l^2))./(sigma(1:k0-1-r).^2));
f = @(l) rho(l) - 2*R(l); %define the SS function

[fm,lambda]=findGlobalMinV1(lambdaArr,f,plotF); %minimize the SS function
if length(lambda)~=1 %if above failed to produce a unique minimum - very rare
    [fm,lambda] = findGlobalMinV2(lambdaArr,f);
end
x_l = x_lam(lambda); %save Tikhonov solution corresponding to found lambda
toc