function [X_MAP,S_MAP,k_MAP] = BM_MAP_THR(Y,var_e,model_params)
%BM_MAP_THR computes a thresholding-like aproximation for the BM-based MAP estimator 
%of the representation coefficients. 
% BM - Boltzmann Machine
% BM-based model - BM prior for the sparsity pattern and Gaussian nonzero coefficients.
% =====================================================================================
% Input:
% Y - an n-by-N matrix consisting of N noisy signals. 
% var_e - the variance of the gaussian distribution of the additive noise.
% model_params - parameters for the stochastic model. Required fields:
% model_params.dictionary - a matrix of size n-by-m consisting of the dictionary atoms.
% model_params.variances - a vector of size m-by-1 consisting of the variances of the 
% gaussian distributions of the nonzero representation coefficients.
% model_params.W, model_params.b - the Boltzmann parameters: an interaction matrix
% of size m-by-m and a bias vector of size m-by-1.
% =====================================================================================
% Output:
% X_MAP - an m-by-N matrix consisting of the recovered representation vectors. 
% S_MAP - an m-by-N matrix consisting of the sparsity patterns.
% k_MAP - a vector of size N-by-1 consisting of the lengths of the recovered supports.
% =====================================================================================
% Michael Elad and Tomer Faktor
% Computer Science and Electrical Engineering Departments
% Technion, Haifa 32000 Israel
% elad@cs.technion.ac.il, tomerfa@tx.technion.ac.il
%
% August 2011
% =====================================================================================
N=size(Y,2);
A=model_params.dictionary;
[n,m]=size(A);
var_x=model_params.variances;
W=model_params.W;
b=model_params.b;
maxNumCoef=n/2;
X_MAP=zeros(m,N);
S_MAP=-1*ones(m,N);
k_MAP=zeros(1,N);
hh = waitbar(0,'Approximating MAP via thresholding-like approach');
for l=1:N
    if ~rem(l,100)
        waitbar(l/N,hh)
    end
    y=Y(:,l);
    % Initialization and setting the parameters  for the stopping rule
    Sempty=-ones(m,1);
    MAPValRef=0.5*Sempty'*W*Sempty+(b'-0.25*log(var_x'/var_e))*Sempty;
    MAPValOne=zeros(1,m);
    for i=1:m
        Stemp=Sempty;
        Stemp(i)=1;
        supp=find(Stemp==1);
        As=A(:,supp);
        Qs=As'*As+diag(var_e./var_x(supp));
        hs=As'*y;
        MAPValOne(i)=hs'*(Qs\hs)/2/var_e-log(det(Qs))/2+(b'-0.25*log(var_x'/var_e))*Stemp+0.5*Stemp'*W*Stemp;
    end
    [~,order]=sort(MAPValOne,'descend');
    S=Sempty;
    MAPVal=zeros(1,maxNumCoef);
    for i=1:maxNumCoef
        S(order(i))=1;
        supp=find(S==1);
        As=A(:,supp);
        Qs=As'*As+diag(var_e./var_x(supp));
        hs=As'*y;
        MAPVal(i)=hs'*(Qs\hs)/2/var_e-log(det(Qs))/2+(b'-0.25*log(var_x'/var_e))*S+0.5*S'*W*S;
    end
    [max_MAPVal,pos]=max(MAPVal);
    s_recov=[];
    x_recov=zeros(m,1);
    if max_MAPVal>MAPValRef
        s_recov=order(1:pos);
        As=A(:,s_recov);
        Qs=As'*As+diag(var_e./var_x(s_recov));
        hs=As'*y;
        x_recov(s_recov)=Qs\hs;
    end
    X_MAP(:,l)=x_recov(:);
    S_MAP(s_recov,l)=1;
    k_MAP(l)=numel(s_recov);
end
close(hh) 