%test_BM_learning - synthetic learning experiments for a banded interaction matrix
%
% test_BM_learning tests MPL learning of the Boltzmann parameters
%
% Generates Fig. 6 in:
% T. Faktor, Y. C. Eldar, and M. Elad, "Exploiting statistical dependencies in sparse 
% representations for signal recovery", submitted to IEEE Trans. Signal Processing 
% =====================================================================================
% Tomer Faktor
% Department of Electrical Engineering
% Technion, Haifa 32000 Israel
% tomerfa@tx.technion.ac.il
%
% August 2011
% =====================================================================================
clear
close all
% Set the true Boltzmann parameters
m=64;
W=zeros(m);
L=9;
[V1,V2]=meshgrid(1:m,1:m);
mask=abs(V1-V2)<=L;
delta_W=0.5;
W=spdiags(-delta_W+2*delta_W*rand(m,m-1),1:L,W);
W=W+W';
W=full(W);
figure,imagesc(W,[-0.5,0.5]),axis off,colormap(gray),colorbar
title('true W')
mean_b=-1.5;
b=mean_b+randn(m,1);
% Generate samples from the Boltzmann machine distribution
N=16000;
num_rounds=10;
S=BM_generate_samples(N,W,b,num_rounds);
k=sum(S==1);
figure,imagesc(S),colormap(gray)
title(['Average cardinality for set of supports=',num2str(round(10*mean(k))/10)])
% Set initial values for the Boltzmann parameters
W0=zeros(m);
mean_S=mean(S,2);
b0=atanh(mean_S); %The MPL estimator for W=0
b0(mean_S==-1)=-5;
b0(mean_S==1)=5;
% Compute the value of the log-PL function for the true Boltzmann parameters
jump_size=1000;
Np=ceil(N/jump_size);
PL_true=0;
for p=1:Np
    S_partial=S(:,1+(p-1)*jump_size:min(p*jump_size,N));
    V=W*S_partial+repmat(b,1,size(S_partial,2));
    PL_true=PL_true+sum(sum(S_partial.*V-log(cosh(V))));
end
% Find which entries in the interaction matrix are more likely to be revealed using these sammples
Pr_vec=mean(0.5*(S+1),2);
Pr_rows=repmat(Pr_vec,1,m);
cond=(Pr_rows<=50/N); % a binary matrix indicating the locations of these entries
W_revealed=W;
W_revealed(cond | cond')=0; % a matrix consisting of these interactions
figure,imagesc(W_revealed,[-0.5,0.5]),axis off,colormap(gray),colorbar
title('true W - only interactions that can be revealed')
% MPL via gradient ascent (GA)
deg=0;
[W_hat_GA,b_hat_GA,PL_hat_GA,err_W_hat_GA,err_b_hat_GA] = MPL_Boltzmann_SESOP(S,W0,b0,deg,0,W,b);
avg_err_W_GA=err_W_hat_GA/m;
avg_err_b_GA=err_b_hat_GA/sqrt(m^2-m);
figure,imagesc(W_hat_GA,[-0.5,0.5]),axis off,colormap(gray),colorbar
title('W recovery via GA')
W_hat_GA_banded=zeros(m);
W_hat_GA_banded(mask)=W_hat_GA(mask);
figure,imagesc(W_hat_GA_banded,[-0.5,0.5]),axis off,colormap(gray),colorbar
title('W recovery via GA - banded')
% MPL via SESOP-2 speed-up
deg=2;
[W_hat_SESOP,b_hat_SESOP,PL_hat_SESOP,err_W_hat_SESOP,err_b_hat_SESOP] = MPL_Boltzmann_SESOP(S,W0,b0,deg,0,W,b);
avg_err_W_SESOP=err_W_hat_SESOP/m;
avg_err_b_SESOP=err_b_hat_SESOP/sqrt(m^2-m);
figure,imagesc(W_hat_SESOP,[-0.5,0.5]),axis off,colormap(gray),colorbar
title('W recovery via SESOP-2')
W_hat_SESOP_banded=zeros(m);
W_hat_SESOP_banded(mask)=W_hat_SESOP(mask);
figure,imagesc(W_hat_SESOP_banded,[-0.5,0.5]),axis off,colormap(gray),colorbar
title('W recovery via SESOP-2 - banded')
% Show results
num_iter=max(length(PL_hat_GA),length(PL_hat_SESOP))-1;
figure,plot(0:num_iter,PL_hat_GA,'-.k','LineWidth',1)
hold on
plot(0:num_iter,PL_hat_SESOP,'-k','LineWidth',1)
plot(0:num_iter,PL_true*ones(1,length(PL_hat_GA)),'--k','LineWidth',1)
hold off
xlabel('Iteration number')
ylabel('Value of log pseudo-likelihood function')
legend('GA','SESOP-2','True')
xlim([-1,num_iter+1])
figure,plot(0:num_iter,avg_err_W_GA,'-.k','LineWidth',1)
hold on
plot(0:num_iter,avg_err_W_SESOP,'-k','LineWidth',1)
hold off
xlabel('Iteration number')
ylabel('Average recovery error for the interaction matrix')
legend('GA','SESOP-2')
xlim([-1,num_iter+1])
figure,plot(0:num_iter,avg_err_b_GA,'-.k','LineWidth',1)
hold on
plot(0:num_iter,avg_err_b_SESOP,'-k','LineWidth',1)
hold off
xlabel('Iteration number')
ylabel('Average recovery for the bias vector')
legend('GA','SESOP-2')
xlim([-1,num_iter+1])