function [X_MMSE] = BM_MMSE_rand_OMP(Y,var_e,model_params,num_runs)
%BM_MMSE_rand_OMP computes a random OMP-like aproximation for the BM-based MMSE  
%estimator of the representation coefficients.
% BM - Boltzmann Machine
% BM-based model - BM prior for the sparsity pattern and Gaussian nonzero coefficients.
% =====================================================================================
% Input:
% Y - an n-by-N matrix consisting of N noisy signals. 
% var_e - the variance of the gaussian distribution of the additive noise.
% model_params - parameters for the stochastic model. Required fields:
% model_params.dictionary - a matrix of size n-by-m consisting of the dictionary atoms.
% model_params.variances - a vector of size m-by-1 consisting of the variances of the 
% gaussian distributions of the nonzero representation coefficients.
% model_params.W, model_params.b - the Boltzmann parameters: an interaction matrix 
% of size m-by-m and a bias vector of size m-by-1.
% num_runs - number of runs of the randomized greedy algorithms. 
% =====================================================================================
% Output:
% X_MMSE - an m-by-N matrix consisting the recovered representation vectors 
% =====================================================================================
% Michael Elad and Tomer Faktor
% Computer Science and Electrical Engineering Departments
% Technion, Haifa 32000 Israel
% elad@cs.technion.ac.il, tomerfa@tx.technion.ac.il
%
% August 2011
% =====================================================================================
N=size(Y,2);
A=model_params.dictionary;
[n,m]=size(A);
var_x=model_params.variances;
W=model_params.W;
b=model_params.b;
err=1e-3;
maxNumCoef=n/2;
X_MMSE=zeros(m,N);
hh = waitbar(0,'Approximating MMSE via random OMP-like approach');
for l=1:N
    if ~rem(l,100)
        waitbar(l/N,hh)
    end
    y=Y(:,l);
    % Initialization and setting the parameters  for the stopping rule
    norm_r=norm(y);   
    Sempty=-ones(m,1);
    x_recov=zeros(m,num_runs);
    for t=1:num_runs % randomly generating num_runs solutions
        S=Sempty;
        ValRef=0.5*Sempty'*W*Sempty+(b'-0.25*log(var_x'/var_e))*Sempty;
        k=0;
        while norm_r>err && k < maxNumCoef
            Val=ValRef*ones(m,1)-1;
            for i=1:m
                Stemp=S;
                if Stemp(i)==-1
                    Stemp(i)=1;
                else
                    continue;
                end
                supp=find(Stemp==1);
                As=A(:,supp);
                Qs=As'*As+diag(var_e./var_x(supp));
                hs=As'*y;
                Val(i)=hs'*(Qs\hs)/2/var_e-log(det(Qs))/2+(b'-0.25*log(var_x'/var_e))*Stemp+0.5*Stemp'*W*Stemp;
            end
            diff_Val=Val-ValRef;
            max_diff=max(diff_Val);
            if max_diff<=200
                Pr_vec=exp(diff_Val);
            else
                Pr_vec=exp(diff_Val+200-max_diff);
            end
            Pr_vec(S==1)=0; % no double choice of atoms
            Pr_vec(Val<=ValRef)=0; % do not choose bad atoms
            if sum(Pr_vec)<1e-6
                break;
            else
                pos=random_choice(Pr_vec/sum(Pr_vec));
                S(pos)=1;
                ValRef=Val(pos);
                k=k+1;
            end
        end
        s_recov=find(S==1);
        As=A(:,s_recov);
        Qs=As'*As+diag(var_e./var_x(s_recov));
        hs=As'*y;
        x_recov(s_recov,t)=Qs\hs;
    end
    X_MMSE(:,l)=mean(x_recov,2);
end
close(hh)

%================================================

function m=random_choice(prob)

Ref=cumsum(prob);
x=rand(1);
m=find(x-Ref<0,1);

return;