function [X_MAP,S_MAP,k_MAP] = BM_MAP_Gibbs_sampler(Y,var_e,model_params,num_rounds,beta0)
%BM_MAP_Gibbs_sampler aproximates the BM-based MAP estimator of the representation
%coefficients via Gibbs sampling and simulated annealing.
% BM - Boltzmann Machine
% BM-based model - BM prior for the sparsity pattern and Gaussian nonzero coefficients.
% =====================================================================================
% Input:
% Y is an n-by-N matrix consisting of N noisy signals. 
% var_e is the variance of the gaussian distribution of the additive noise.
% model_params - parameters for the stochastic model. Required fields:
% model_params.dictionary - a matrix of size n-by-m consisting of the dictionary atoms.
% The dictionary should be square and unitary.
% model_params.variances - a vector of size m-by-1 consisting of the variances of the 
% gaussian distributions of the nonzero representation coefficients.
% model_params.W, model_params.b - the Boltzmann parameters: an interaction matrix
% of size m-by-m and a bias vector of size m-by-1.
% num_rounds - number of rounds for the Gibbs sampler.
% beta0 - parameter for the annealing schedule.
% =====================================================================================
% Output:
% X_MAP is an m-by-N matrix consisting of the restored representation vectors. 
% S_MAP is an m-by-N matrix consisting of the sparsity patterns.
% k_MAP is a vector of size N-by-1 consisting of the lengths of the recovered supports.
% =====================================================================================
% Tomer Faktor
% Department of Electrical Engineering
% Technion, Haifa 32000 Israel
% tomerfa@tx.technion.ac.il
%
% August 2011
% =====================================================================================
if nargin<5
    beta0=1;
end
N=size(Y,2);
A=model_params.dictionary;
m=size(A,2);
var_x=model_params.variances;
W=model_params.W;
b=model_params.b;
T0=beta0^(m*num_rounds-1);
X_MAP=zeros(m,N);
S_MAP=-1*ones(m,N);
k_MAP=zeros(1,N);
hh = waitbar(0,'Approximating MAP via Gibbs sampling and simulated annealing');
for l=1:N
    if ~rem(l,100)
        waitbar(l/N,hh)
    end
    y=Y(:,l);
    S=2*(rand(m,1)>0.5)-1; % initialize randomly using uniform distribution 
    T=T0;
    supp=find(S==1);
    As=A(:,supp);
    Qs=As'*As+diag(var_e./var_x(supp));
    hs=As'*y;
    MAPValRef=hs'*(Qs\hs)/2/var_e-log(det(Qs))/2+(b'-0.25*log(var_x'/var_e))*S+0.5*S'*W*S;
    for k=1:num_rounds
        seq_inds=randperm(m);
        for i=1:m
            S_temp=S;
            curr_ind=seq_inds(i);
            S_temp(curr_ind)=-S_temp(curr_ind);
            supp=find(S_temp==1);
            As=A(:,supp);
            Qs=As'*As+diag(var_e./var_x(supp));
            hs=As'*y;
            MAPVal_temp=hs'*(Qs\hs)/2/var_e-log(det(Qs))/2+(b'-0.25*log(var_x'/var_e))*S_temp+0.5*S_temp'*W*S_temp;
            Pr=1./(1+exp(MAPValRef-MAPVal_temp));
            cond=rand(1)<Pr;
            if cond
                S=S_temp;
                MAPValRef=MAPVal_temp;
            end
            T=T/beta0;
        end
    end
    s_recov=find(S==1);
    As=A(:,s_recov);
    Qs=As'*As+diag(var_e./var_x(s_recov));
    hs=As'*y;
    x_recov=zeros(m,1);
    x_recov(s_recov)=Qs\hs;
    X_MAP(:,l)=x_recov(:);
    S_MAP(s_recov,l)=1;
    k_MAP(l)=numel(s_recov);
end
close(hh)