function [X_MAP,S_MAP,k_MAP,model_params] = BM_denoising(Y,var_e,A,k0,L)
%BM_denoising - joint MAP pursuit and model estimation for the BM-based model.
% BM - Boltzmann Machine
% BM-based model - BM prior for the sparsity pattern and Gaussian nonzero coefficients.
%
% [X_MAP,S_MAP,k_MAP,model_params] = BM_denoising(Y,var_e,A) recovers both the
% representation coefficients and the model parameters for a set of noisy signals. 
% If A is square and unitary, the algorithm learns a banded interaction matrix and 
% performs exact MAP pursuit of the representations via message passisng.
% Otherwise, the algorithm learns a general interaction matrix and performs approximate
% MAP pursuit of the representations via a greedy OMP-like approach.
%
% [...] = BM_denoising(Y,var_e,A,k0) specifies the prior belief on the sparsity level.
% The first denoising stage in the algorithm assumes an i.i.d. prior on the
% representation coefficients, This prior is determined by k0.
%
% [...] = BM_denoising(Y,var_e,A,k0,L) also specifies the number of nonzero diagonals
% in the upper triangle part of the learned interaction matrix.
% =====================================================================================
% Input:
% Y - an n-by-N matrix consisting of N noisy signals. 
% var_e - the variance of the gaussian distribution of the additive noise.
% A - an n-by-m matrix consisting of dictionary atoms. The dictionary remains 
% fixed (up to atom permutations in the case of a banded interaction matrix).
% k0 - the prior belief on the average cardinality of the supports.
% L - number of nonzero diagonals in the upper triangle part of the interaction matrix.
% =====================================================================================
% Output:
% X_MAP - an m-by-N matrix consisting of the recovered representation vectors. 
% S_MAP - an m-by-N matrix consisting of the the sparsity patterns.
% k_MAP - a vector of size N-by-1 consisting of the lengths of the recovered supports.
% model_params - estimated parameters for the stochastic model. Fields:
% model_params.dictionary - a matrix of size n-by-m consisting of the dictionary atoms.
% model_params.variances - a vector of size m-by-1 consisting of the variances of the 
% gaussian distributions of the nonzero representation coefficients.
% model_params.W, model_params.b - the Boltzmann parameters: an interaction matrix
% of size m-by-m and a bias vector of size m-by-1.
% =====================================================================================
% Tomer Faktor
% Department of Electrical Engineering
% Technion, Haifa 32000 Israel
% tomerfa@tx.technion.ac.il
%
% August 2011
% =====================================================================================
[n,m]=size(A);
G=A'*A;
unitary_cond=(m==n && sum(sum(abs(G-eye(m))>1e-10))==0);
if unitary_cond
    model_type='unitary_banded'; % assume a unitary dictionary and a banded interaction matrix
else
    model_type='general'; % make no assumputions on the dictionary and interaction matrix
end
if nargin<4
   k0=round(m/10);
end
if nargin<5
    if strcmp(model_type,'unitary_banded')
        L=round(1.5*log2(m));
    elseif strcmp(model_type,'general')
        L=m-1;
    end
end
Pr_marginal_hat=(k0/m)*ones(m,1); % prior belief for the marginals
% Initialize model parameters
model_params.dictionary=A;
model_params.variances=50^2*ones(m,1);
model_params.W=zeros(m); % i.i.d. prior
model_params.b=0.5*log(Pr_marginal_hat./(1-Pr_marginal_hat)); % i.i.d. prior
% pursuit - recover X  via MAP using i.i.d. prior
if strcmp(model_type,'unitary_banded')
    [X_MAP,S_MAP]=BM_unitary_MAP_message_passing(Y,var_e,model_params);
elseif strcmp(model_type,'general')
    [X_MAP,S_MAP]=BM_MAP_OMP(Y,var_e,model_params);
end
% update signal variances - use  ML estimators
var_mat=(0.5*(S_MAP+1))*(X_MAP.^2)';
var_x_hat=diag(var_mat)./sum(0.5*(S_MAP+1),2);
sigma_x_hat=sqrt(var_x_hat);
sigma_x_hat(sigma_x_hat==0)=50;
sigma_x_hat(isnan(sigma_x_hat))=50;
sigma_x_hat(sigma_x_hat<2 & sigma_x_hat>0)=2;
A_hat=A;
% update W,b - use MPL via SESOP
if L>0
    [W_hat,b_hat] = MPL_Boltzmann_SESOP(S_MAP);
    if L<m-1
        [W_hat,inds_bnd]=estimate_W_banded(W_hat,L);
        b_hat=b_hat(inds_bnd);
        sigma_x_hat=sigma_x_hat(inds_bnd);
        A_hat=A_hat(:,inds_bnd);
        model_params.inds_bnd=inds_bnd;
    end
elseif L==0 % Independent prior
    Pr_marginal_hat=mean(0.5*(S_hat+1),2);
    b_hat=0.5*log(Pr_marginal_hat./(1-Pr_marginal_hat)); % The ML estimator
    b_hat(Pr_marginal_hat==0)=-5;
    b_hat(Pr_marginal_hat==1)=5;
end
model_params.dictionary=A_hat;
model_params.variances=sigma_x_hat.^2;
model_params.W=W_hat;
model_params.b=b_hat;
% pursuit - recover X  via MAP using BM prior
if strcmp(model_type,'unitary_banded')
    [X_MAP,S_MAP,k_MAP]=BM_unitary_MAP_message_passing(Y,var_e,model_params);
elseif strcmp(model_type,'general')
    [X_MAP,S_MAP,k_MAP]=BM_MAP_OMP(Y,var_e,model_params);
end