%test_BM_denoising_patches - denoising experiments on image patches.
%
% test_BM_denoising_patches compares BM-based denoising with OMP denoising on image
% patches for the unitary and redundant DCT dictionaries and different noise levels.
%
% Generates results of Table I in:
% T. Faktor, Y. C. Eldar, and M. Elad, "Exploiting statistical dependencies in sparse 
% representations for signal recovery", submitted to IEEE Trans. Signal Processing 
% =====================================================================================
% Tomer Faktor
% Department of Electrical Engineering
% Technion, Haifa 32000 Israel
% tomerfa@tx.technion.ac.il
%
% August 2011
% =====================================================================================
clear
close all
% Load noise-free patches extracted from natural images
load('noise_free_patches.mat'); % 100,000 patches
N=1e5; % number of patches on which we want to perform the denoising tests
noise_free_patches=noise_free_patches(:,1:N);
% Set dictionaries for the denoising experiments
n=size(noise_free_patches,1); % each patch is of size 8-by-8 ==> n=64
m=4*n;
A=zeros(n);
blk_size=round(sqrt(n));
[j_grid,i_grid]=meshgrid(1:blk_size,1:blk_size);
alpha=[1,sqrt(2)*ones(1,blk_size-1)];
for p=1:blk_size
    for q=1:blk_size
        A_one_blk=alpha(p)*alpha(q)*cos(pi*(2*i_grid-1).*(p-1)/(2*blk_size))*...
            cos(pi*(2*j_grid-1).*(q-1)/(2*blk_size)); % unitary DCT dictionary
        A(:,(p-1)*blk_size+q)=A_one_blk(:);
    end
end
unitary_DCT=A*diag(1./sqrt(diag(A'*A))); % normalize dictionary atoms (columns in A)
A=zeros(n,m);
blk_size1=round(sqrt(n));
blk_size2=round(sqrt(m));
[j_grid,i_grid]=meshgrid(1:blk_size1,1:blk_size1);
alpha=[1,sqrt(2)*ones(1,blk_size2-1)];
for p=1:blk_size2
    for q=1:blk_size2
        A_one_blk=alpha(p)*alpha(q)*cos(pi*(2*i_grid-1).*(p-1)/(2*blk_size2))*...
            cos(pi*(2*j_grid-1).*(q-1)/(2*blk_size2)); % redundant DCT dictionary
        A(:,(p-1)*blk_size2+q)=A_one_blk(:);
    end
end
redundant_DCT=A*diag(1./sqrt(diag(A'*A))); % normalize dictionary atoms (columns in A)
% Perform denoising experiments
k0=10;
L=9;
sigma_e_vec=[2,5,10,15,20,25];
hh = waitbar(0,'Testing patch denoising methods');
for i=1:length(sigma_e_vec)
    waitbar(i/numel(sigma_e_vec),hh)
    sigma_e=sigma_e_vec(i);
    % Add noise and subtract DC from each patch
    noisy_patches=noise_free_patches+sigma_e*randn(n,N);
    sub_Y=mean(noisy_patches);
    Y=noisy_patches-repmat(sub_Y,n,1);
    Y0=noise_free_patches-repmat(sub_Y,n,1);
    % Classical sparsity model, unitary DCT dictionary, exact l_0 norm pursuit
    [X_OMP_unitary,S_OMP_unitary,k_OMP_unitary]=unitary_OMP(Y,sigma_e,unitary_DCT);
    RMSE_per_pixel_OMP_unitary(i)=norm(Y0-unitary_DCT*X_OMP_unitary,'fro')/sqrt(n*N);
    mean_k_OMP_unitary(i)=mean(k_OMP_unitary);
    % BM model with banded interaction matrix, unitary DCT dictionary, exact MAP pursuit
    [X_MAP_unitary,S_MAP_unitary,k_MAP_unitary,model_params_unitary] = BM_denoising(Y,sigma_e^2,...
        unitary_DCT,k0,L);
    RMSE_per_pixel_MAP_unitary(i)=norm(Y0-model_params_unitary.dictionary*X_MAP_unitary,'fro')/sqrt(n*N);
    mean_k_MAP_unitary(i)=mean(k_MAP_unitary);
    % Classical sparsity model + unitary DCT dictionary, approximate l_0 norm pursuit (OMP)
    stop_method='err_thr';
    stop_params.err_noise=1.15*sqrt(n)*sigma_e;
    [X_OMP_redundant,S_OMP_redundant,k_OMP_redundant]=OMP(Y,redundant_DCT,stop_method,stop_params);
    RMSE_per_pixel_OMP_redundant(i)=norm(Y0-redundant_DCT*X_OMP_redundant,'fro')/sqrt(n*N);
    mean_k_OMP_redundant(i)=mean(k_OMP_redundant);
    % BM model, redundant DCT dictionary, approximate MAP pursuit (OMP-like)
    [X_MAP_redundant,S_MAP_redundant,k_MAP_redundant,model_params_redundant] = BM_denoising(Y,sigma_e^2,...
        redundant_DCT,k0);
    RMSE_per_pixel_MAP_redundant(i)=norm(Y0-redundant_DCT*X_MAP_redundant,'fro')/sqrt(n*N);
    mean_k_MAP_redundant(i)=mean(k_MAP_redundant);
end
close(hh)