%test_BM_unitary_pursuit - synthetic pursuit experiments for a unitary dictionary and a 
%banded interaction matrix.
%
% test_BM_unitary_pursuit compares various pursuit methods for different noise levels
%
% Generates Fig. 4 in:
% T. Faktor, Y. C. Eldar, and M. Elad, "Exploiting statistical dependencies in sparse 
% representations for signal recovery", submitted to IEEE Trans. Signal Processing
% =====================================================================================
% Tomer Faktor
% Department of Electrical Engineering
% Technion, Haifa 32000 Israel
% tomerfa@tx.technion.ac.il
%
% August 2011
% =====================================================================================
clear
close all
% Set model parameters
n=64;
del_W=0.5;
del_b=0.5;
b0=-2.5;
L=9;
W=zeros(n);
v1=rand(n,L);
W=spdiags(-2*del_W+del_W*(v1>0.15)+del_W*(v1>0.35)+del_W*(v1>0.65)+del_W*(v1>0.85),1:L,W);
W=W+W';
W=full(W);
v2=rand(n,1);
b=b0-del_b+del_b*(v2>1/3)+del_b*(v2>2/3);
sigma_x=15+45*rand(n,1);
A=zeros(n);
blk_size=round(sqrt(n));
[j_grid,i_grid]=meshgrid(1:blk_size,1:blk_size);
alpha=[1,sqrt(2)*ones(1,blk_size-1)];
for p=1:blk_size
    for q=1:blk_size
        A_one_blk=alpha(p)*alpha(q)*cos(pi*(2*i_grid-1).*(p-1)/(2*blk_size))*...
            cos(pi*(2*j_grid-1).*(q-1)/(2*blk_size)); % Unitary DCT dictionary
        A(:,(p-1)*blk_size+q)=A_one_blk(:);
    end
end
A=A*diag(1./sqrt(diag(A'*A))); % normalize dictionary atoms (columns in A)
model_params.W=W;
model_params.b=b;
model_params.variances=sigma_x.^2;
model_params.dictionary=A;
% Generate synthetic signals
N=10000;
all_inds=1:N;
num_rounds=10;
S=BM_generate_samples(N,W,b,num_rounds);
k=sum(S==1);
figure,imagesc(S),colormap(gray)
title(['Average cardinality for set of supports=',num2str(round(10*mean(k))/10)])
X=zeros(n,N);
sigma_x_mat=repmat(sigma_x,1,N);
X(S==1)=sigma_x_mat(S==1).*randn(sum(k),1);
% Test various pursuit methods
sigma_e_vec=[2,5,10:5:30];
hh = waitbar(0,'Testing pursuit methods');
for i=1:numel(sigma_e_vec)
    waitbar(i/numel(sigma_e_vec),hh)
    sigma_e=sigma_e_vec(i);
    Y=A*X+sigma_e*randn(n,N);
    % Exact MAP - BM prior
    [X_MAP_exact,S_MAP_exact,k_MAP_exact] = BM_unitary_MAP_message_passing(Y,sigma_e^2,model_params);
    dist_support_MAP_exact(i) = support_dist(S_MAP_exact,S);
    err_x_MAP_exact(i)=norm(X_MAP_exact-X,'fro')/norm(X,'fro');
    % Approximate MAP - OMP-like
    [X_MAP_OMP,S_MAP_OMP,k_MAP_OMP] = BM_MAP_OMP(Y,sigma_e^2,model_params);
    dist_support_MAP_OMP(i) = support_dist(S_MAP_OMP,S);
    err_x_MAP_OMP(i)=norm(X_MAP_OMP-X,'fro')/norm(X,'fro');
    % Approximate MAP - Gibbs sampling + simulated annealing
    num_rounds=10;
    beta0=1.01;
    [X_MAP_Gibbs,S_MAP_Gibbs,k_MAP_Gibbs] = BM_unitary_MAP_Gibbs_sampler(Y,sigma_e^2,model_params,...
        num_rounds,beta0);
    dist_support_MAP_Gibbs(i) = support_dist(S_MAP_Gibbs,S);
    err_x_MAP_Gibbs(i)=norm(X_MAP_Gibbs-X,'fro')/norm(X,'fro');
    % Approximate MMSE - random OMP-like
    X_MMSE_rand_OMP = BM_MMSE_rand_OMP(Y,sigma_e^2,model_params,10);
    X_MMSE_rand_OMP=full(X_MMSE_rand_OMP);
    X_abs_sorted=sort(abs(X_MMSE_rand_OMP),'descend');
    inds=find(k_MAP_OMP>0);
    thr_vec=X_abs_sorted(k_MAP_OMP(inds)+n*(all_inds(inds)-1));
    S_MMSE_rand_OMP=-ones(n,N);
    S_MMSE_rand_OMP(:,inds)=2*(abs(X_MMSE_rand_OMP(:,inds))>=repmat(thr_vec,n,1))-1;
    dist_support_MMSE_rand_OMP(i) = support_dist(S_MMSE_rand_OMP,S);
    err_x_MMSE_rand_OMP(i)=norm(X_MMSE_rand_OMP-X,'fro')/norm(X,'fro');
    % OMP
    [X_OMP,S_OMP,k_OMP] = unitary_OMP(Y,sigma_e,A);
    X_OMP_corrected = oracle_formula(Y,sigma_e^2,model_params,S_OMP);
    dist_support_OMP(i) = support_dist(S_OMP,S);
    err_x_OMP(i)=norm(X_OMP_corrected-X,'fro')/norm(X,'fro');
    % Bayesian oracle
    X_oracle = oracle_formula(Y,sigma_e^2,model_params,S);
    err_x_oracle(i)=norm(X_oracle-X,'fro')/norm(X,'fro');
end
close(hh)
% Show results
figure,plot(sigma_e_vec,dist_support_MAP_exact,'-.k','LineWidth',1)
hold on
plot(sigma_e_vec,dist_support_MAP_OMP,'-k','LineWidth',1)
plot(sigma_e_vec,dist_support_MAP_Gibbs,'ok','MarkerSize',5,'MarkerFaceColor','k')
plot(sigma_e_vec,dist_support_MMSE_rand_OMP,'sk','MarkerSize',5,'MarkerFaceColor','k')
plot(sigma_e_vec,dist_support_OMP,':k','LineWidth',1);
hold off
xlim([sigma_e_vec(1),sigma_e_vec(end)])
ylim([0,1.05])
xlabel('\sigma_e')
ylabel('Normalized error in the support')
legend('MAP - exact','MAP - OMP-like','MAP - Gibbs sampler','MMSE - random OMP-like','OMP');
figure,plot(sigma_e_vec,err_x_MAP_exact,'-.k','LineWidth',1)
hold on
plot(sigma_e_vec,err_x_MAP_OMP,'-k','LineWidth',1)
plot(sigma_e_vec,err_x_MAP_Gibbs,'ok','MarkerSize',5,'MarkerFaceColor','k')
plot(sigma_e_vec,err_x_MMSE_rand_OMP,'sk','MarkerSize',5,'MarkerFaceColor','k')
plot(sigma_e_vec,err_x_OMP,':k','LineWidth',1);
plot(sigma_e_vec,err_x_oracle,'--k','LineWidth',1);
hold off
xlim([sigma_e_vec(1),sigma_e_vec(end)])
ylim([0,0.5*ceil(max(err_x_OMP)/0.5)])
xlabel('\sigma_e')
ylabel('Relative recovery error')
legend('MAP - exact','MAP - OMP-like','MAP - Gibbs sampler','MMSE - random OMP-like','OMP','oracle');