function [isolated_band, filter_coefs, fused_channel, fused_channel_sample_freq, fused_channel_offset] = ...
    isolate_channel(channels, channel_time_space, channels_negative, channels_positive, channel_sample_freq, channel_freq_offsets, ...
                    band_start_freq, band_end_freq, left_tolerance, right_tolerance, ...
                    do_plots)

%% Fuse channels

single_channel_index = find(channel_freq_offsets <= band_start_freq + eps(band_start_freq) & ...
                            channel_freq_offsets+channel_sample_freq/2 >= band_end_freq - eps(band_end_freq));
if (~isempty(single_channel_index))
    fused_channel = channels(:,single_channel_index);
    fused_channel_sample_freq = channel_sample_freq;
    fused_channel_offset = channel_freq_offsets(single_channel_index);
else
    double_channel_index = find(channel_freq_offsets <= band_start_freq + eps(band_start_freq) & ...
                                channel_freq_offsets+channel_sample_freq >= band_end_freq - eps(band_end_freq));
    if (~isempty(double_channel_index))
        fused_channel = fuse_two_channels(channel_sample_freq, channel_time_space, ...
                                          channels_negative(:,double_channel_index), channels_positive(:,double_channel_index), ...
                                          channels_negative(:,double_channel_index+1), channels_positive(:,double_channel_index+1));
        fused_channel_sample_freq = 2*channel_sample_freq;
        fused_channel_offset = channel_freq_offsets(double_channel_index);
    else
        error('FREQ_RECOV:CHANNEL_NOT_FOUND', 'Containing channel(s) not found.');
    end;
end;

%% Design filter

shifted_band_start_freq = band_start_freq-fused_channel_offset;
shifted_band_end_freq = band_end_freq-fused_channel_offset;
shifted_left_tolerance = min(left_tolerance, shifted_band_start_freq);
shifted_right_tolerance = min(right_tolerance, fused_channel_sample_freq/2-shifted_band_end_freq);

minimal_edge_spacing = 0.05*fused_channel_sample_freq; % [Hz]

passband_ripple = 1e-6; % linear
stopband_ripple = 1e-2; % linear
max_filter_order = 200;

left_edge_ok = shifted_band_start_freq-shifted_left_tolerance > minimal_edge_spacing;
right_edge_ok = fused_channel_sample_freq/2-shifted_band_end_freq+shifted_right_tolerance > minimal_edge_spacing;

if (left_edge_ok && right_edge_ok)
    % design band-pass filter
    filter_spec = firpmord([shifted_band_start_freq-shifted_left_tolerance ...
                            shifted_band_start_freq shifted_band_end_freq ...
                            shifted_band_end_freq+shifted_right_tolerance], ...
                           [0 1 0], [stopband_ripple passband_ripple stopband_ripple], fused_channel_sample_freq, 'cell');
	assert(filter_spec{1} <= max_filter_order);
	filter_coefs = firpm(filter_spec{:});
elseif (~left_edge_ok && right_edge_ok)
    % design low-pass filter
    filter_spec = firpmord([shifted_band_end_freq ...
                            shifted_band_end_freq+shifted_right_tolerance], ...
                           [1 0], [passband_ripple stopband_ripple], fused_channel_sample_freq, 'cell');
	assert(filter_spec{1} <= max_filter_order);
	filter_coefs = firpm(filter_spec{:});
elseif (left_edge_ok && ~right_edge_ok)
    % design high-pass filter
    filter_spec = firpmord([shifted_band_start_freq-shifted_left_tolerance ...
                            shifted_band_start_freq], ...
                           [0 1], [stopband_ripple passband_ripple], fused_channel_sample_freq, 'cell');
	assert(filter_spec{1} <= max_filter_order);
	filter_coefs = firpm(filter_spec{:});
else
    % design identity filter
	filter_coefs = [1 0];
end;

isolated_band = filtfilt(filter_coefs, 1, fused_channel);

[isolated_band_psd, isolated_band_psd_freq_space] = ...
    pwelch(isolated_band, [], [], [], fused_channel_sample_freq);

[filter_freq_response, filter_freq_space] = freqz(filter_coefs, 1, 1000, fused_channel_sample_freq);

if (exist('do_plots', 'var') && do_plots == true)
    figure;
    plot(filter_freq_space+fused_channel_offset, db(abs(filter_freq_response).^2), ...
         isolated_band_psd_freq_space+fused_channel_offset, db(isolated_band_psd, 'power'), 'm');
    title('Isolated Band');
    xlabel('Frequency [Hz]');
    ylabel('Magnitude [dB]');
    xlim([isolated_band_psd_freq_space(1) isolated_band_psd_freq_space(end)]+fused_channel_offset);
end;

end