function [S SNum] = HaarSeg(I,W,raw,chromPos,...
                            breaksFdrQ,haarStartLevel,...
                            haarEndLevel,postProcessingFlag)                       
 % [S SNum] = HaarSeg(I,W,raw,chromPos,breaksFdrQ,haarStartLevel,haarEndLevel,postProcessingFlag)
 %
 % HaarSeg performs segmentation according to the HaarSeg algorithm. 
 % This is the main segmentation script.
 % This script includes several optional extentions, supporting weights 
 % (also known as quality of measurments), raw measurments, and linear time 
 % post processing steps which improve the segmentation result,
 % at a slight run time penalty.
 % 
 % Input:
 % I:  matrix of log(R/G) measurments. Columns are array. Rows are 
 %     measurments sorted according to their genomic location.
 % W:  weight matrix, corresponding to quality of measurment. 
 %     Insert an empty matrix "[]" if your platform does not support
 %     this feature. Insert 1/(sigma^2) as weights if your platform
 %     output sigma as the quality of measurment. W must have the same
 %     size as I matrix.
 % raw: Mininum of the raw red and raw green measurment, before the log.
 %      raw is used for the non-stationary variance compensation. Insert 
 %      an empty matrix "[]" if you do not wish to use this feature. 
 % chromPos: A matrix of two columns. The first column is the start index 
 %      of each chromosome. The second column is the end index of each chromosome.
 % breaksFdrQ: The FDR q parameter. Common used values are 0.05, 0.01, 0.001.
 %      Default value is 0.001.
 % haarStartLevel: The detail subband from which we start to detect peaks. The higher
 %      this value is, the less sensitive we are to short segments. 
 %      The default is value is 1, corresponding to segments of 2 probes.
 % haarEndLevel: The detail subband until which we use to detect peaks. The higher
 %      this value is, the more sensitive we are to large trends in the data. This value
 %      DOES NOT indicate the largest possible segment that can be detected.
 %      The default is value is 5, corresponding to step of 32 probes in each direction.
 % postProcessingFlag: If this flag is non-zero, HaarSeg apply two post-processing steps:
 %      (a) t-test : to reduce false positive breakpoints.
 %      (b) 1-probe localization: to improve localization of breakpoints.
 %      Both post-processing steps are sub-optimal, linear time.
 %      Current version does not take into account weights in
 %      the post-processing steps. 
 %      The default value of postProcessingFlag is 0 (not-active).
 %      To active the post processing set this flag to 1.
 % 
 % Output:
 % S: The segmentation result. Same size as I.
 % SNum: Number of detected segments. 
 %
 % Usage Examples:
 % The basic haarseg on a single chromosome, or any other 1-D signal:
 % result = HaarSeg(log_ratio_data,[],[],[1 size(log_ratio_data,1)],0.001,1,5,0);
 %
 % OR just use default values:
 % result = HaarSeg(log_ratio_data);
 %
 % HaarSeg with quality of measurment and non-stationary variance (single chromosome):
 % result = HaarSeg(log_ratio_data,1./variance.^2,min(raw_red,raw_green),[1 size(log_ratio_data,1)],0.001,1,5);
 %
 %
 %    Copyright (C) 2008  Erez Ben-Yaacov
 %
 %    This program is free software: you can redistribute it and/or modify
 %    it under the terms of the GNU General Public License as published by
 %    the Free Software Foundation, either version 3 of the License, or
 %    (at your option) any later version.
 %
 %    This program is distributed in the hope that it will be useful,
 %    but WITHOUT ANY WARRANTY; without even the implied warranty of
 %    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 %    GNU General Public License for more details.
 %    <http://www.gnu.org/licenses/>

if (nargin < 2)
    W = [];
end
if (nargin < 3)
    raw = [];
end
if (nargin < 4)
   chromPos = [1 size(I,1)];
end
if (nargin < 5)
   breaksFdrQ = 1e-3;
end
if (nargin < 6)
   haarStartLevel = 1;
end
if (nargin < 7)
   haarEndLevel = 5;
end
if (nargin < 8)
    postProcessingFlag = 0;
end

valueMethod = 'mean';
ProbeNum = size(I,1);
ArrayNum = size(I,2);
weightsFlag = numel(W);
nsvFlag = numel(raw);

if (nsvFlag)
    % non stationary variance empirical threshold set to 50
    NSV_TH = 50;
    varMask = raw < NSV_TH;
    clear raw
end

% mex functions expect double percision, safety cast
I = double(I);
W = double(W);

% the segmentation
S = zeros(size(I));
SNum = zeros(size(chromPos,1),ArrayNum);
if (postProcessingFlag)
    % post-processing t-test p-value is set to 0.05
    TTEST_P_VALUE = 0.05;
    ttable = tinv(TTEST_P_VALUE/2,1:size(I,1));
end

for arr = 1:ArrayNum
    % variance estimate is done on the entrie array data
    [diffI peakLoc] = mexConvAndPeak(I(:,arr),1);
    matPeakLoc = peakLoc(1:find(peakLoc == -1,1,'first')-1) + 1;
    if (nsvFlag)
        diffMask = (stepConv(varMask(:,arr), 2) >= 0.5);
        peakSigmaEst = median(abs(diffI(~diffMask))) / 0.6745;
        noisySigmaEst = median(abs(diffI(diffMask))) / 0.6745;
        if (isnan(noisySigmaEst))
            noisySigmaEst = 0;
        end
        if (isnan(peakSigmaEst))
            peakSigmaEst = 0;
        end
    else
        peakSigmaEst = median(abs(diffI)) / 0.6745;
    end
    
    % segmentation is done on each chromosome seperatly
    for chr = 1:size(chromPos,1)
        y = I(chromPos(chr,1):chromPos(chr,2),arr);
	if (nsvFlag)
	    yVarMask = varMask(chromPos(chr,1):chromPos(chr,2),arr);
	end
        if (weightsFlag)
            wei = W(chromPos(chr,1):chromPos(chr,2),arr);
        end
        
        % find breakpoints
        uniPeakLoc = int32(-1);
        for level = haarStartLevel:haarEndLevel
            stepHalfSize = 2^(level);
            if (weightsFlag)
                [convRes peakLoc] = mexConvAndPeak(y,stepHalfSize,wei); 
            else
                [convRes peakLoc] = mexConvAndPeak(y,stepHalfSize);
            end
            matPeakLoc = peakLoc(1:find(peakLoc == -1,1,'first')-1) + 1;
	    
	    if (nsvFlag)
	        convMask = (stepConv(yVarMask, 2*stepHalfSize) >= 0.5);
	        sigmaEst = (1-convMask)*peakSigmaEst + convMask*noisySigmaEst;
	        T = fdrThres(convRes(matPeakLoc)./sigmaEst(matPeakLoc), breaksFdrQ, 1);
	    else
	        T = fdrThres(convRes(matPeakLoc), breaksFdrQ, peakSigmaEst);
	    end
            
            unifyWin = int32(2^(level - 1));
            
            tmpPeakLoc = uniPeakLoc;
	    if (nsvFlag)
	        uniPeakLoc = mexThresAndUnify(convRes./sigmaEst, peakLoc, tmpPeakLoc, T, unifyWin);  		
	    else
	        uniPeakLoc = mexThresAndUnify(convRes, peakLoc, tmpPeakLoc, T, unifyWin);
	    end

        end %for level
        breakpoints = uniPeakLoc(1:find(uniPeakLoc == -1,1,'first')-1) + 1;

        % apply post-processing steps, if required to:        
        if (postProcessingFlag)
            % apply t-test, to reduce false-positive breakpoints
            breakpoints = ttestUnify(breakpoints,y,peakSigmaEst,ttable);

            % apply localization test, to improve breakpoint location
            % current implementation try to move each breakpoint
            % 1 sample left/right.
            peaks = int32(-ones(size(y)));
            peaks(1:numel(breakpoints)) = int32(breakpoints - 1);
            npeaks = mexAdjustBreaks(y,peaks);
            breakpoints = double(npeaks(1:find(npeaks == -1,1,'first')-1) + 1);
        end        
        
        % set values between breakpoints
        segNum = numel(breakpoints);
        if (weightsFlag)
            segs = segmentByPeaks(y,breakpoints,valueMethod,wei);
        else
            segs = segmentByPeaks(y,breakpoints,valueMethod);            
        end
       
        S(chromPos(chr,1):chromPos(chr,2),arr) = segs;
        SNum(chr, arr) = segNum;
    end%for chr
end%for arr