function [Cn,C]=AnalogSampling_GUI(tk,p,h,SNR,SamplingKernel,order,w_bar)
% This function perform the analog sampling process for a given sampling
% kernel and diracs as pulses
% Written by Kfir Gedalyahu and Rami Cohen

% tk - column vector of paths delays in the range [0,1]
% p - number of sampling channels.
% h - gain sequences, the k'th row correspond to the gain sequences of the k'th path

% SamplingKernel - 'BPF': complex ideal band-pass filters
%                                        -'LPF': ideal low pass filter
%                                        -'BLPF': butterworth low pass filter
% order -  butterworth low pass filter order. Required only when using this kernel
%
% Cn- noisy sampling sequences
% C - non-noisy sampling sequences
%
% GUI
% density - the density of the analog grid

density=10;
T=1; % normalized period
Nsym=size(h,2); % number of symbols
K=length(tk);
if K~=size(h,1)
    error('Number of gain sequences must be equal to the number of paths.');
end

switch SamplingKernel
    case 'BPF'
        if p<=K 
            error('p must be greater than number of paths.');
        end
            
        ISI=1000; % analog sampling kernel length (one side)
        ak=[h zeros(K,2*ISI)];

        % samples
        C=zeros(p,size(ak,2));

        for i=1:p
            for k=1:K
                DiscFilter=Phi(-ISI:ISI,tk(k),i-1,T,1/T);
                C(i,:)=C(i,:)+fftfilt(DiscFilter,ak(k,:));
            end
        end

        Ec=sum(sum((abs(C(:,ISI+(1:Nsym))).^2)))/p/Nsym; % signal energy
        n=10^(-SNR/20)*sqrt(Ec/2)*(randn(size(C))+1i*randn(size(C))); % random noise with variance Ec
        Cn=C+n;

    case 'LPF'
        if p<=K  || round(p/2)~=(p/2)
            error('p must be greater than number of paths and even.');
        end
        
        ISI=1000; % analog sampling kernel length (one side)
        ak=[h zeros(K,2*ISI)];
        
        % samples
        C=zeros(p*density,size(ak,2));
        n=-ISI:ISI;
        for i=1:p*density
            for k=1:K
                delta=(i-1)*T/p/density;
                DiscFilter=sinc(p/T*(n*T+delta-tk(k)));
                C(i,:)=C(i,:)+fftfilt(DiscFilter,ak(k,:));
            end
            waitbar((i/p/density)/3,w_bar,sprintf('Analog Sampling'));
        end

        C_nd=C(1:density:end,:);
        Ec=1/p/Nsym*sum(sum(abs(C_nd(:,ISI+(1:Nsym))).^2));
        n=10^(-SNR/20)*sqrt(Ec/2)*(randn(size(C))+1i*randn(size(C))); % random noise with variance Ec
        Cn=C+n;
        
    case 'BLPF'
        if p<=2*K  || round(p/2)~=(p/2)
            error('p must be greater than twice the number of paths and even.');
        end
        
        ISI=100;
        a=2;
        w0=(pi*p)/a/T; 
        ak=[h zeros(K,ISI)];
        
        % samples
        C=zeros(p*density,size(ak,2));
        n=0:ISI;
        for i=1:p*density
            for k=1:K
                delta=(i-1)*T/p/density;
                DiscFilter=AnalogButter(n*T+delta-tk(k),w0,order);
                C(i,:)=C(i,:)+fftfilt(DiscFilter,ak(k,:));
            end
            waitbar((i/p/density)/4,w_bar,sprintf('Analog Sampling'));
        end
        
        C_nd=C(1:density:p*density,:); % p*density/T
        Ec=1/p/Nsym*sum(sum(abs(C_nd(:,(1:Nsym))).^2));
        n=10^(-SNR/20)*sqrt(Ec/2)*(randn(size(C))+1i*randn(size(C))); % random noise with variance Ec
        Cn=C+n;
        
    otherwise
        error('Unknown sampling kernel.')
end
