function [Aout,tk_est]=ChanRecover_GUI(Cn,K,N,DigDelay,SamplingKernel,order,w_bar)
% This function recover the channel's parameters from noisy samples
% according to "Time Delay Estimation from Low Rate Samples: A Union of Subspaces Approach"
% by Kfir Gedalyahu and Yonina C. Eldar
% Written by Kfir Gedalyahu and Rami Cohen
%
% Cn - noisy samples
% K - number of paths
% N - number of pulses
% DigDelay - delays of the digital correction filters
% SamplingKernel - 'BPF': complex ideal band-pass filters
%                                        -'LPF': ideal low pass filter
%                                        -'BLPF': butterworth low pass filter
% order -  Butterworth low pass filter order. Required only when using this kernel
%
% Aout - recovered gain sequences
% tk_est - recovered delays

T=1; % normalized period
p=size(Cn,1);

switch SamplingKernel
    case 'BPF'
        ISI=1000; % analog sampling kernel length (one side)

        % TLS ESPRIT
        tk_est=sort(TLS_ESPRIT(Cn(:,ISI+(1:N)),K,T));

        % sequences recovery
        [Tk,M]=meshgrid(tk_est,0:p-1);
        Ntau=exp(-1i*2*pi/T*(M).*Tk);
        A1=pinv(Ntau)*Cn;
        Aout=zeros(size(A1));

        n=-DigDelay:DigDelay;
        for i=1:K
            DiscFilter=sinc(tk_est(i)+n*T).*exp(1i*pi/T*(tk_est(i)+n*T));
            Aout(i,:)=fftfilt(DiscFilter,A1(i,:));
        end;

        Aout=(Aout(:,ISI+DigDelay+(1:N)));

    case 'LPF'
        ISI=1000; % analog sampling kernel length (one side)

        % first digital correction
        D=[Cn zeros(p,2*DigDelay)];
        n=-DigDelay:DigDelay;
        for i=1:p
            delta=(i-1)*T/p;
            DiscFilter=sinc(1/T*(+n*T-delta)).*exp(1i*pi*n).*exp(1i*pi/T*(p-1)*delta);
            D(i,:)=fftfilt(DiscFilter,D(i,:));
            waitbar(1/3+i/p/3,w_bar,sprintf('Digital Correction - Stage 1'));
        end

        Df=fft(D,[],1);
        % time recovery
        tk_est=TLS_ESPRIT(Df(:,ISI+DigDelay+(1:N)),K,T);

        % sequences recovery
        [Tk,M]=meshgrid(tk_est,(0:p-1)-p/2);
        Ntau=exp(-1i*2*pi/T*(M).*Tk);
        A1=pinv(Ntau)*Df;
        Aout=zeros(size(A1));

        n=-DigDelay:DigDelay;
        for i=1:K
            DiscFilter=sinc(tk_est(i)+n*T).*exp(1i*pi/T*(tk_est(i)+n*T));
            Aout(i,:)=fftfilt(DiscFilter,A1(i,:));
            waitbar(2/3+(i)/K/3,w_bar,sprintf('Digital Correction - Stage 2'));

        end;
        Aout=(Aout(:,ISI+2*DigDelay+(1:N)));
        
        waitbar(1,w_bar,sprintf('Done'));
        pause(0.1);
        
    case 'BLPF'
        a=2;
        w0=(pi*p)/a/T;
        n=-DigDelay:DigDelay;

        % create correction filters
        ButtCorFilter=zeros(p,length(n));
        for i=1:p
            ButtCorFilter(i,:)=1/(2*pi*p)*ButterCorrect(w0,order,-p/2+i,n);
            waitbar(1/4+i/p/8,w_bar,sprintf('Digital Correction - Stage 1'));
        end

        % first digital correction
        D=[Cn zeros(p,2*DigDelay)];
        n=-DigDelay:DigDelay;
        for i=1:p
            delta=(i-1)*T/p;
            DiscFilter=sinc(1/T*(+n*T-delta)).*exp(1i*pi*n).*exp(1i*pi/T*(p-1)*delta);
            D(i,:)=fftfilt(DiscFilter,D(i,:));
            waitbar(0.375+i/p/8,w_bar,sprintf('Digital Correction - Stage 1'));
        end

        Df=fft(D,[],1);

        % second digital correction
        chanind=(1:p/2)+round(p/4); % indexes of used channels - use half of channels
        if round(p/4)~=p/4
            chanind=[chanind(1)-1 chanind];
        end
        
        Df2=[Df zeros(p,2*DigDelay)];
        for i=chanind
            Df2(i,:)=filter(ButtCorFilter(i,:),1,Df2(i,:));
            waitbar(1/2+i/length(chanind)/4,w_bar,sprintf('Digital Correction - Stage 2'));
        end

        % time recovery
        tk_est=TLS_ESPRIT(Df2(chanind,2*DigDelay+(1:N)),K,T);

        % sequences recovery
        [Tk,M]=meshgrid(tk_est,(0:p-1)-p/2);
        Ntau=exp(-1i*2*pi/T*(M).*Tk);
        A1=pinv(Ntau(chanind,:))*Df2(chanind,:);
        Aout=zeros(size(A1));

        n=-DigDelay:DigDelay;
        for i=1:K
            DiscFilter=sinc(tk_est(i)+n*T).*exp(1i*pi/T*(tk_est(i)+n*T));
            Aout(i,:)=fftfilt(DiscFilter,A1(i,:));
            waitbar(3/4+i/K/4,w_bar,sprintf('Digital Correction - Stage 3'));
        end;
        Aout=(Aout(:,3*DigDelay+(1:N)));
        waitbar(1,w_bar,sprintf('Done'));
        pause(0.1);

    otherwise
        error('Unknown sampling kernel.')
end

