#!/usr/bin/python
#
# Copyright (C) 2010 Orr Srour
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>
# 

"""
\n FIA - Fluxomers Iterative Algorithm

SYNOPSIS:
    fia_analysis {-i[=--fia] PROJFILE | -t[=--ftbl] PROJFILE} [-o[=--output_file] OUTPUTFILE]

USAGE:
    fia_analysis performs the following two main steps:
    
        1) Metabolic pathway analysis.
        
            Here FIA transforms the text file representing the metabolic network into
            mathematical matrices required later for the MFA optimization process.            
            
        2) Metabolic fluxes evaluation.

            Here FIA tries to find the fluxes that best suit the given metabolic system.
            The output log of this stage is a file called    <PROJFILE>.out,
            or else if specified by the "-o" flag.

    fia_analysis supports two types of input file formats: FIA and 13CFlux FTBL.
    The two are very much alike, except for the fact that the FIA format supports only declaration of
    uni-directional fluxes, and does not contain the FLUXES and INEQUALITIES sections.
    For bi-directional fluxes, FIA simply defines two seperated fluxes.
    When analyzing 13CFlux FTBL files, fia_analysis looks for C=0 constraints in the FLUXES XCH section
    in order to determine directionality of fluxes.

    Copyright (C) 2010, Orr Srour.
    This program comes with ABSOLUTELY NO WARRANTY;

OPTIONS:
    -i --fia
        Specifies that the metabolic pathway input file <PROJFILE> is a FIA formated file. 
        All fluxes are assumed unidirectional (for bi-directional specify 2 seperated fluxes).        

    -t --ftbl
        Specifies that the metabolic pathway input file <PROJFILE> is a 13CFlux FTBL formated file.         
        Flux is assumed unidirectional unless C=0 constraint is applied to it in the FLUXES XCH section.
        The FIA formated file will be saved under <PROJFILE>.fia .

   -o --output_file
        optional. 
        Specifies the name of the result log output file. The default file name is PROJFILE.out
        
    at least one of the above must be supplied.
        
USAGE EXAMPLE:

fia_analysis -t temp.ftbl
    Runs anaylsis & evaluation of the network defined in the FTBL formated input file temp.ftbl .
    same as:
        fia_analysis --ftbl temp.ftbl 

fia_analysis --fia temp.fia
    Runs anaylsis & evaluation on the given FIA formated input file.
    Same as:
        fia_analysis -i temp.fia

"""

import sys
import getopt
import pickle
import time
#import getopt

#import time
#from  cvxmod import *
#import cvxmod
#import cvxopt
from numpy import matrix
from scipy.linalg import inv, det, eig
from scipy import sparse 
#from scipy import zeros
from scipy import *
import scipy
from scipy import optimize
#import progress_bar
import warnings
#import progressbar

warnings.simplefilter('ignore',scipy.sparse.SparseEfficiencyWarning)
#warnings.simplefilter('ignore')

from scipy.sparse.linalg import umfpack as um

#from cvx_opt_matapp_solve import *
       
umfpack = um.UmfpackContext()

#from solve_orr import *
#import mlabwrap

prog_width = 40
display_prog = False
#!/usr/bin/env python

import sys

class MyWriter:
    """
    A file-handler replacement that duplicates the actions applied to
    the hanlder to a log file handler as well.        
    """
    def __init__(self, stdout, filename):
        """
        Initialization of the file-handler duplicator.
        @type stdout: file
        @param stdout: Originial file handler to be forked
        @type filename: string
        @param filename: Name of the log file for which actions should be replicated into
        """
        self.stdout = stdout
        self.logfile = file(filename, 'a')

    def write(self, text):
        """
        Replacement for the file handler's write method.
        """
        self.stdout.write(text)
        self.logfile.write(text)

    def flush(self):
        """
        Replacement for the file handler's flush method.
        """
        self.stdout.flush()
        self.logfile.flush()
        
    def close(self):
        """
        Replacement for the file handler's close method.
        """
        self.stdout.close()
        self.logfile.close()

class FtblFile:
    """
    This class is the main object of the FIA algorithm.
    Its main purposes are:
        1. Conversion of FIA and FTBL files into python mathematical objects.
        2. Solution of the MFA optimization problem
    """
    filename = '' # filename
    ftbl_file = '' # text file object
    
    def __init__ (self, projname, inpfiletype):
        """
        @type projname: string
        @param projname: Name of the project to be analyzed.
        @type inpfiletype: string
        @param inpfiletype: Type of the input file to be analyzed. Can be either 'fia' (for fia input files) or 'ftbl' (for ftbl input files)        
        """
        self.ProjName = projname #: Project's name        
        self.labelinp_var = 100 #: Label input variance (it is not supplied with FTBL files). Currently we don't really use it (since the initial value is generated by demanding exactly the inputs)        
        self.use_labelinp_var = True
        self.flux_upper_bound = upper_bound

        self.start_time = time.time()
        
        self.meas_flux_const_value = 1e-7 # The value of the virtual flux used to measure labels
        self.ftbl_fluxsec_eq_cons_var = virt_flux_value ### TEMPORARY VALUE FOR EQUALITY CONSTRAINTS ARISED FROM THE FTBL(ONLY!) FLUXES SECTION
        self.InputFileType = inpfiletype #: input file type (fia or ftbl)
        self.excell_pools = [] #: Dictionary of the external (global input or output) metabolic pools 
        self.ftbl_file = self.load_file(projname) #: Parsed input file object (TAB spliited lists for the various lines of the input file)
        if display_prog:
            prog_bar = progress_bar.ProgressBar(prog_width)
        self.create_pools_dict()
        if display_prog:
            prog_bar.update(50)
        self.create_flux_dict()
##        try:
##            self.create_flux_dict()
##        except:
##            print "\n ERROR while building the fluxes dictionary. Please check your fluxes definitions."
##            sys.exit(-1)
        if display_prog:
            prog_bar.update(100)
        print ' done.'
        sys.stdout.flush()
        if debug_level>1:
            print 'Found the following input/output pools:', self.excell_pools
        #self.create_flux_stoi_eq()
        # If we want only to analyze the network:
        
        self.create_U_matrix() # Generate flux transformation (isotopomeric->real) matrix
        try:
            self.create_label_meas_eq()
        except:
            print "\nERROR while parsing the label measurements section."
            sys.exit(-1)

        try:
            self.create_ms_meas_eq() # Generate the mass spectrometry equations
        except:
            print "\nERROR while parsing the MS measurements section."
            sys.exit(-1)
            
        self.create_inp_eq_matrix()
        self.create_flux_meas_mat()
        self.create_global_meas_mat()
        
        if not (Evaluate_Only):
            self.create_label_transition_mat() 
        
        #if not (Evaluate_Only):        
        self.create_lu_trans_matrix()
            
        #self.save_data()        
##        self.save_ftblfile(ProjName +".dat")

#        if not (Analyze_Only):
        print "\nTime for file analysis: ", time.time()-self.start_time
        print "Ready to start the optimization process."
        self.evaluate()
            
    def save_ftblfile(self, filename):
        """
        This function saves the entire FtblFile class to a file using python's pickle packge.
        This enables the user to use the data analyzed later without the need of recalculating the
        transition or LU matrices (which can take a bit of a time).
        """
        if debug_level > 1:
            print "Saving data in file: ", filename," ...",
            f = open(filename,'w')
            pickle.dump(self,f,-1)
            print " done."
            sys.stdout.flush()
        
    def save_data(self):
        """
        MATLAB debugging function.
        This function saves the mathematical objects used for the optimizaiton process
        in files in order to be analyzed by U{MATLAB<http://www.mathworks.com>}.
        """        
        matlab_save_sparsed_matrix(ProjPathName + ProjName + "_Q"+ext,self.Q)
        print "saving H..."
        matlab_save_sparsed_matrix(ProjPathName + ProjName + "_H"+ext,self.H)
        print "saving S..."        
        matlab_save_sparsed_matrix(ProjPathName + ProjName + "_S"+ext,self.S)
        print "saving U..."
        matlab_save_sparsed_matrix(ProjPathName + ProjName + "_U"+ext,self.U)


        #print "saving alpha_M..."
        #matlab_save_sparsed_matrix(ProjPathName + ProjName + "_alpha_M"+ext,self.alpha_M)
        #print "saving alpha_N..."
        #matlab_save_sparsed_matrix(ProjPathName + ProjName + "_alpha_N"+ext,self.alpha_N)
        #print "saving alpha_trans matrix..."
        #matlab_save_sparsed_matrix(ProjPathName + ProjName + "_alpha_mat"+ext,self.alpha_mat)
        #print "saving alpha_Q matrix..."
        #matlab_save_sparsed_matrix(ProjPathName + ProjName + "_alpha_Q_mat"+ext,self.alpha_Q_mat)


        #print "saving UV_alpha_M..."
        #matlab_save_sparsed_matrix(ProjPathName + ProjName + "_UV_alpha_M"+ext,self.UV_alpha_M)
        #print "saving UV_alpha_N..."
        #matlab_save_sparsed_matrix(ProjPathName + ProjName + "_UV_alpha_N"+ext,self.UV_alpha_N)
        #print "saving UV_alpha_trans matrix..."
        #matlab_save_sparsed_matrix(ProjPathName + ProjName + "_UV_alpha_mat_M"+ext,self.UV_alpha_mat_M)
        #matlab_save_sparsed_matrix(ProjPathName + ProjName + "_UV_alpha_mat_N"+ext,self.UV_alpha_mat_N)
        #print "saving alpha_Q matrix..."
        #matlab_save_sparsed_matrix(ProjPathName + ProjName + "UV_alpha_Q_mat"+ext,self.UV_alpha_Q_mat)

        
        print "saving labelinp_num..."
        matlab_save_sparsed_matrix(ProjPathName + ProjName + '_labelinp_num'+ext, self.labelinp_num )
        print "saving labelinp_denom..."
        matlab_save_sparsed_matrix(ProjPathName + ProjName + '_labelinp_denom'+ext, self.labelinp_denom )
        print "saving labelinp_value..."
        matlab_save_sparsed_matrix(ProjPathName + ProjName + '_labelinp_value'+ext, self.labelinp_value )

        print "saving labelmeas_num..."
        matlab_save_sparsed_matrix(ProjPathName + ProjName + '_labelmeas_num'+ext, self.labelmeas_num )
        print "saving labelmeas_denom..."
        matlab_save_sparsed_matrix(ProjPathName + ProjName + '_labelmeas_denom'+ext, self.labelmeas_denom )
        print "saving labelmeas_value..."
        matlab_save_sparsed_matrix(ProjPathName + ProjName + '_labelmeas_value'+ext, self.labelmeas_value )
        print "saving labelmeas_var..."
        matlab_save_sparsed_matrix(ProjPathName + ProjName + '_labelmeas_var'+ext, self.labelmeas_var)        

        if gen_trans_mat:        
            print "saving trans_mat_a..."
            matlab_save_sparsed_matrix(ProjPathName + ProjName + '_trans_mat_a'+ext, self.trans_mat_a )
            print "saving trans_mat_a_beta..."
            matlab_save_sparsed_matrix(ProjPathName + ProjName + '_trans_mat_a_beta'+ext, self.trans_mat_a_beta )
            print "saving trans_mat_a_gamma..."
            matlab_save_sparsed_matrix(ProjPathName + ProjName + '_trans_mat_a_gamma'+ext, self.trans_mat_a_gamma )
            print "saving trans_mat_b..."
            matlab_save_sparsed_matrix(ProjPathName + ProjName + '_trans_mat_b'+ext, self.trans_mat_b )
            print "saving trans_mat_b_beta..."
            matlab_save_sparsed_matrix(ProjPathName + ProjName + '_trans_mat_b_beta'+ext, self.trans_mat_b_beta )        
            print "saving beta_vec_M..."
            matlab_save_sparsed_matrix(ProjPathName + ProjName + '_beta_vec_M'+ext, self.beta_vec_M )        
            print "saving beta_vec_N..."
            matlab_save_sparsed_matrix(ProjPathName + ProjName + '_beta_vec_N'+ext, self.beta_vec_N )        
            save_text_vec(ProjPathName + ProjName + "_beta_vec_names.txt",self.beta_vec)
        print "saving text vectors..."
#        save_text_vec(ProjPathName + ProjName + "_label_names.txt",[self.fluxes.items()[self.fluxes.values().index(i)][0] for i in range(len(self.fluxes))])
        save_text_vec(ProjPathName + ProjName + "_label_names.txt",[a[0] for a in sort_dict_by_values(self.fluxes)])

        save_text_vec(ProjPathName + ProjName + "_flux_names.txt",self.flux_nw_vec.keys())

        

    def get_flux_name(self, flux_num):
        '''
            returns the name of fluxer whose index is flux_num.
            @type flux_num: number
            @param flux_num: The number of the fluxomer we are looking for.
            @rtype: string
            @return: The name of the fluxomer.
        '''
        return self.fluxes.items()[self.fluxes.values().index(flux_num)][0]
    
    def load_file(self, filename):
        """
        This function loads the input file into the FtblClass object by:
            1. Dropping comments (lines starting with "//")
            2. Replacing EOL "\\n" or "\\r" with ""
            3. Erasing spaces
        Eventually we are left with a file containing only TABs. 
        This object is then transformed into an array of lists - every array element is a line of the parsed file, and splitted by TABs into a list object.

        @type filename: string
        @param filename: Name of the input file to be examined.
        @rtype: list
        @return: List of lists containing the TAB splitted parsed version of the input file.
        """
        
        if debug_level > 1:     
            print "Opening the ftbl file... ",
            sys.stdout.flush()
        f = open(filename,"r")
        r = []
        for line in f:
            line = line.split("//")[0].replace("\n","").replace("\r","").replace(" ","")
            if line.replace("\t","") != "" and line != "\t":
                r.append(line)
        f.close()
        return r        

    def find_segment(self, seg_name):
        '''
        This function finds the block "seg_name" block segment in the parsed input file L{self.ftbl_file} object. 
        New segments begin with every non-idented line in the input file. Their names are the first string in these lines.
        
        @type seg_name: string
        @param seg_name: Name of the segment to be analyzed
        @rtype: list
        @return: A list containing the segment's rows, tab seperated.
        '''
        r = []
        
        first_line = -1
        last_line = -1
        counting = False
        for e,i in enumerate(self.ftbl_file):
            if i[0] != "\t" and counting == True:
                last_line = e-1
                counting = False
            if i.lower().startswith(seg_name.lower()):
                first_line = e+1
                counting = True

        if first_line == -1:
            return "" # if we didn't find the segment, return ""

        remove_leading = False
        if self.ftbl_file[first_line][0] == "\t":
            remove_leading = True
        for i in self.ftbl_file[first_line:last_line+1]:
            t = i.split("\t")
            if remove_leading == True:
                t=t[1:]
            r.append(t) # we always drop the first tab (since everything starts with tab)

        return r

    def create_pools_dict(self):
        """
        This function creates the initial metabolic pools dictionary object by analyzing the NETWORK section of the input file. The function creates the L{self.pools} dictrionary object - Metabolic pools dictionary for which the keys are pools names, and the values are the number of carbon atoms for each pool.        
        """
        netseg = self.find_segment('NETWORK')[1:]
        pools = {}
        for linenum, line in enumerate(netseg):            
            if line[0] != "":
                for k,j in enumerate(line[1:]):
                    if j != "":
                        try:
                            pools[j] = len(netseg[linenum+1][k+1])-1
                        except:
                            "\n ERROR while parsing the NETWORK section.", line
        self.pools = pools #: Metabolic pools dictionary (Keys are pools names, values are the number of carbon atoms for each pool)
        
    def create_flux_dict(self):
        """
        This function creates and updates the main building blocks objects used for the metabolic pathway analysis.

        Classification of fluxes
        ========================        
            Classifiction of fluxes into: uni-directional fluxes, bi-directional fluxes (FTBL files only) and input/output fluxes.
            The uni and bi direictional analysis is only needed for FTBL files (FIA assumes all the fluxes are uni-directional). 
            The decision is made by looking for 0 constrained exchange fluxes in the FLUXES section of the FTBL file.

        Re-generation of the NETWORK section - adding necessary virtual fluxes
        ======================================================================
            There are 2 sorts of virtual fluxes.
            
            1. In order to equalize the variances of the measured metabolic labels, we add virtual fluxes
            with constant (very small) strength, coming out of the measured pools, and optimize over them.
            This way, the measurement quantity is not influeced by the value of the flux itself.

            2. In addition, when we are given with a chemical reaction that transforms two elements of the same metabolic pool into others, we translate
            it into a regular multi-input chmecial reaction by adding a virtual pool and flux to our system:

            As an example::
                f: A + A -> B + C     (flux "f" transforms A+A into B+C)
            is transformed into::
                f: A + A_s -> B + C   (flux "f" transofmrs A+A_s ito B+C)
                f_A: A -> A_s         (flux "f_A" transforms A into A_s)
        
            This transformation is being done on the input file object itself (hence by changing the parsed NETWORK segment of the input file).
            The same applies for reversed fluxes of bi-diredctional FTBL files fluxes.

        Analysis of extra-cellular pools
        ================================
            Finding pools which are global input or output to the system, and save them in the L{self.excell_pools} object.
            These pools are recognized as pools with no input / output fluxes going into / exiting them.      

        Re-generation of the NETWORK section - adding necessary reversed fluxes
        =======================================================================        
            The mathematical analysis of the problem consists of flux vector representing uni-directional fluxes.
            Due to this reason, we need to transform the NETWORK section to contain only uni-directional fluxes by adding to it necessary reversed
            fluxes. This only applies when FTBL input files are supplied (since again, FIA files are assumed to contain only uni-diretional fluxes in anyway).
        
        Generation of the global fluxes vector
        ======================================
            Generation of the global fluxes (as opposed to fluxomers) vector dictionary object. 
            The dictrionary L{self.flux_nw_vec} keys are the names of the global fluxes in our system, 
            and the values are the number of carbon atoms each of them transforms.

        Generation of the equality matrix
        =================================
            The equality matrix L{self.S} and its associated equality vector L{self.S_b} are the mathematical constraints of the global optimization problem
            That should apply on the global fluxes vector (hence self.S * x == self.S_b).
            The equality matrix is constructed out of 2 main elements:
                1. The stoichiometric equations - as analyzed from the parsed NETWORK section of the input file
                2. The eqaulities constraints from the EQUALITIES section of the input file.

        Generation of the equality measurements
        =======================================
            Equalities arised from the constraints section of the FLUXES section (only applies to FTBL files) are added 
            to the system as measurements and not as constraints (as opposed to equalities from the EQUALITIES section).
            Note that this only relevant for FTBL files. The python relevant objects are L{self.FCM} and L{self.FCv}, 
            and we have M{FCM*x = FCv} .

        Generation of the global equation list
        ======================================
            The entire metabolic network is defined by a set of fluxomeric equations defined in L{self.eq_list}.
            The structure of this list is::
                [ [ poolname : ([input fluxes], [output fluxes])], ... ]
            where the input and output fluxes are list of the names of the incoming and outgoing fluxes.
        """
        # First line is the header:
        netseg = self.find_segment('NETWORK') 
        #: IOvec analyzes the header of the NETWORK section in order to decide which lines are inputs to the metabolic pools, and which are outputs. 
        IOvec = [] 
        for i  in netseg[0][1:]:
            if i != "":
                IOvec.append( i[0] ) # educt, #product

        FluxHeader = netseg[0]
        head_range = range(1,5)
        netseg = netseg[1:]

        
        fluxes_data = []
        unidir_fluxes = []        
        
        # If we are dealing with a FTLB file, we need to find the unidirectional fluxes (for FIA, all the fluxes are unidirectional). We do that by finding the 0 constrained XCH fluxes.
        #unidir_fluxes = []
        if self.InputFileType == 'ftbl':        
            fluxseg = self.find_segment('FLUXES')            
            in_xch_sec = False            
            for l in fluxseg:                
                if l[0] == 'XCH':
                    in_xch_sec = True
                    in_net_sec = False                    
                if in_xch_sec and l[2]=='C':
                    try:
                        if float(l[3]) != 0:
                            print "ERROR while parsing the FLUXES XCH section. Only 0 constraints are allowed."
                            sys.exit(-1)
                    except:                        
                        print "\nERROR while parsing the FLUXES XCH section. Only 0 constraints are allowed."
                        print "line:", l
                        sys.exit(-1)    
                    unidir_fluxes.append(l[1].replace("'",""))
                    
        # end of uni-dir analysis

        # Re-generation of the NETWORK section, looking for fluxes that take 2 elements of their input pool. See the function documentation for more details.
        # Generation of the fluxes_data object, representing the updated NETWORK section.
        ##        fluxes_data = [] # we already initialized this array before
        special_fluxes = []
        for linenum, line in enumerate(netseg):
            if line[0] != "":
                if line[1] == line[2]: # if we have flux that requires 2 of the same input pool,
                                       # we add a virtual pool for ease of implementation
                    line[1] = line[1] + "_" + line[0] # add the virutal pool                              
                    tmpline = [line[0]+"_s", line[2],"",line[1],""]
                    tmpline2 = ["",netseg[linenum+1][1],"",netseg[linenum+1][1],""]
                    self.pools[line[1]]=len(netseg[linenum+1][1])-1
                    unidir_fluxes.append(tmpline[0])                    
                    fluxes_data.append( [line,netseg[linenum+1]] )
                    fluxes_data.append( [tmpline,tmpline2] )
                else:
                    fluxes_data.append( [line,netseg[linenum+1]] )
                    
                    # if someone forgot to put "TAB" in the end of the line, we'll add it for him.
                    if len(line) == 4:
                        line+=[""]                    
                    # (if that's a reversed flux we'll add it by hand)
                    # if that's a bidir flux and it becomes 2 instances of the pool                        
                    if self.InputFileType == 'ftbl' and (line[3]==line[4]) and (line[0] not in unidir_fluxes):                        
                        # problem if it's a 1->2 flux that goes into an output ...
                        special_fluxes.append(line[0])
                        special_fluxes.append(line[0] + "_r") 
                        # add the reversed line:
                        tmpline_2 = [line[0]+"_r",line[3],line[4]+"_"+line[0],line[1],line[2]]
                        tmpline2_2 = ["",netseg[linenum+1][3],netseg[linenum+1][4],netseg[linenum+1][1],netseg[linenum+1][2]]
                        fluxes_data.append( [tmpline_2,tmpline2_2] )
##                        unidir_fluxes.append(tmpline_2[0])
                        # add the line that goes to line[0]_s:
                        tmpline = [line[0]+"_s",line[3],"",line[4]+"_"+line[0]]
                        unidir_fluxes.append(tmpline[0])
                        tmpline2 = ["",netseg[linenum+1][3],"",netseg[linenum+1][3],""]
                        self.pools[line[4]+"_"+line[0]]=len(netseg[linenum+1][3])-1                        
                        fluxes_data.append( [tmpline,tmpline2] )
        
        # find excell_pool (find pools which are only inputs or outputs - "extra-cellular pools")
        for p in self.pools:
            inp = 0
            out = 0
            for f in fluxes_data:                
                for ind,a in enumerate(IOvec):                    
                    if len(f[0])>ind+1: # in case someone forgot to put TABS after his last entered flux
                        if a[0] == "P" and f[0][ind+1] == p: #if it's a product (flux f[0][0] goes into p)
                            inp+=1
                        if a[0] == "E" and f[0][ind+1] == p: #if it's an educt (flux f[0][0] goes out of p)
                            out+=1
                    else:
                        f[0].append('')
            
            if (inp==0 or out==0): #or ((p in pools_virtualized) and out==1):
                self.excell_pools.append(p)
            #if ((p in pools_virtualized) and out==1):
                # we need to add additional flux because we're in an output pool...

        # Regenerate the NETWORK section - add virtual fluxes for measured metabolic pools
        # (in order to make their variance function independent of their strength)
        pools_virtualized = [] # to these pools we'll add a virtual pool, so later we need to check if they were extracellular before our addition.
        
        labseg = self.find_segment("LABEL_MEASUREMENTS")
        self.labseg = labseg
        letters = 'abcdefghijklmnopqrstuvwxyz'
        self.measured_fluxes = []
        
        # For the LABEL_MEASUREMENTS section:
        for k in labseg[1:]:
            if k[0]!='':
                fluxes_data.append([["MEAS_FLUX_"+k[0],k[0],"",k[0]+"_TEMPPOOL",""],["","#"+letters[:len(k[4])-1],'',"#"+letters[:len(k[4])-1],'']])
                unidir_fluxes.append("MEAS_FLUX_"+k[0])
                if k[0] in self.excell_pools:
                    fluxes_data.append([["tmp_MEAS_FLUX_"+k[0],k[0],"",k[0]+"_TEMPPOOL",""],["","#"+letters[:len(k[4])-1],'',"#"+letters[:len(k[4])-1],'']])
                    unidir_fluxes.append("tmp_MEAS_FLUX_"+k[0])
                    self.excell_pools.remove(k[0])
##                self.excell_pools.append(k[0]+"_TEMPPOOL")
                self.pools[k[0]+"_TEMPPOOL"] = len(k[4])-1
                pools_virtualized.append(k[0])
                self.measured_fluxes.append("MEAS_FLUX_"+k[0]) 
        # For the MASS_SPECTROMETRY section:
        msseg = self.find_segment("MASS_SPECTROMETRY")
        self.msseg = msseg
        visited_fluxes = []
        for k in msseg[1:]:
            if k[0]!='':
                if k[0] not in visited_fluxes:
                    fluxes_data.append([["MEAS_FLUX_"+k[0],k[0],"",k[0]+"_TEMPPOOL",""],["","#"+letters[:self.pools[k[0]]],'',"#"+letters[:self.pools[k[0]]],'']])
                    unidir_fluxes.append("MEAS_FLUX_"+k[0])
                    if k[0] in self.excell_pools:
                        fluxes_data.append([["tmp_MEAS_FLUX_"+k[0],k[0],"",k[0]+"_TEMPPOOL",""],["","#"+letters[:self.pools[k[0]]],'',"#"+letters[:self.pools[k[0]]],'']])
                        unidir_fluxes.append("tmp_MEAS_FLUX_"+k[0])
                        self.excell_pools.remove(k[0])
                    self.pools[k[0]+"_TEMPPOOL"] = self.pools[k[0]]
##                    self.excell_pools.append(k[0]+"_TEMPPOOL")
                    pools_virtualized.append(k[0])
                    self.measured_fluxes.append("MEAS_FLUX_"+k[0])
                    visited_fluxes.append(k[0])

        for k in set(pools_virtualized):
            self.excell_pools.append(k+"_TEMPPOOL")
            
        # add bidirectional fluxes to fluxes_data for FTBL input files:
        
        if self.InputFileType == 'ftbl':        
            fluxseg = self.find_segment('FLUXES')
                
            print "Found the following unidirectional fluxes:", unidir_fluxes
            #print "All the rest of the fluxes (except for global input/output) are assumed to be bi-directional."
            fluxes_data_copy=[list([list(z) for z in k]) for k in fluxes_data]
            bidir_fluxes = []
            io_fluxes_inp = []
            io_fluxes_out = []
            for l in fluxes_data_copy:
                excell_flux = False
                for p in self.excell_pools:                    
                    if p in l[0][1:3]:                        
                        excell_flux = True
                        io_fluxes_inp.append(l[0][0])
                    if p in l[0][3:5]:
                        excell_flux = True
                        io_fluxes_out.append(l[0][0])
                if (excell_flux==False) and (l[0][0] not in unidir_fluxes) and (l[0][0] not in special_fluxes): # if that's a bidirectional flux                                        
                    k=[list(q) for q in l]
                    # Let's make the reverse flux out of the current given flux:
                    k[0][0]=l[0][0]+'_r'
                    k[0][1]=l[0][3]
                    k[0][2]=l[0][4]
                    k[0][3]=l[0][1]
                    k[0][4]=l[0][2]
                    k[1][1]=l[1][3]
                    k[1][2]=l[1][4]
                    k[1][3]=l[1][1]
                    k[1][4]=l[1][2]                    
                    fluxes_data.append(k) # let's add its reverse flux to the network as well
                    bidir_fluxes.append(l[0][0])
                if (l[0][0] in special_fluxes) and (l[0][0][-2:] == '_r'):  # Could rise error if the flux name string has only one char
                    pass
                if (l[0][0] in special_fluxes) and (l[0][0][-2:] != '_r'):  # Could rise error if the flux name string has only one char
                    bidir_fluxes.append(l[0][0])
                
            print "Found the following bidirectional fluxes:", bidir_fluxes
            print "Found the following global input fluxes:", io_fluxes_inp
            print "Found the following global output fluxes:", io_fluxes_out        
            self.io_fluxes_inp=io_fluxes_inp
            self.io_fluxes_out=io_fluxes_out
            io_fluxes = io_fluxes_inp+io_fluxes_out
        self.fluxes_data = fluxes_data
                    
        # Create the global fluxes dictionary object. A dictionary with flux width for every flux_name
        flux_nw_vec = {} # fluxes name_width dict
        
        for f in fluxes_data:
            cur_width = 0            
            for ind,a in enumerate(IOvec):
                if a[0] == "E" and f[0][ind+1] != "": #if it's an educt (reactant) (coming out of)
                    cur_width += len(f[1][ind+1]) - 1 # len of (#ABCD) - 1
            flux_nw_vec [ f[0][0] ] =  cur_width  # append [fluxname, flux_width]            
        self.flux_nw_vec = flux_nw_vec        

        # Create the equality constraints S matrix.
        # First we add to it the stoichiometric equations:
        S = sparse.lil_matrix((len(self.pools)-len(self.excell_pools), len(self.flux_nw_vec)))        
        sr = 0
        for p in self.pools:
            if p not in self.excell_pools:                
                # for every pool that needs to stay in equilibrium
                inp = []
                out = []
                for f in fluxes_data:                    
                    for ind,a in enumerate(IOvec):
                        if a[0] == "P" and f[0][ind+1] == p: #if it's a product (flux f[0][0] goes into p)
                            inp.append(self.get_metabolic_flux_index(f[0][0]))
                        if a[0] == "E" and f[0][ind+1] == p: #if it's an educt (flux f[0][0] goes out of p)
                            out.append(self.get_metabolic_flux_index(f[0][0]))
                for k in inp:
                    S[sr, k] = S[sr, k] + 1
                for k in out:
                    S[sr, k] = S[sr, k] - 1
                sr += 1                

        # Add to S the equality constraints from the EQUALITIES section of the input file
        S_b = [0]*S.shape[0]
        eqseg = self.find_segment('EQUALITIES')
        # Let's add the constraints upon the measured fluxes:
        eqseg.append(['NET',''])        
        for k in self.measured_fluxes:
            eqseg.append(['',self.meas_flux_const_value,k])
        
        # and continue with the other equalities constraints:
        in_net=False
        try:
            for line in eqseg:
                if line[0] == 'NET':
                    in_net=True
                elif line[0] == 'XCH':
                    in_net=False
                elif in_net==True and line[1] != 'VALUE' and line[1] != '': 
                    S_b.append(float(line[1]))
                    cur_formula=line[2]
                    cur_formula.split("+")
                    pos_args=[k.split("-")[0] for k in cur_formula.split("+") if k.split("-")[0] != '']
                    neg_args=[k.split("+")[0] for k in cur_formula.split("-")[1:] if k.split("+")[0] != '']
                    for ind,q in enumerate(neg_args):
                        if len(q.split("*")) == 1:
                            q="1*"+q
                            neg_args[ind] = q
                        neg_args[ind]=neg_args[ind].split("*")
                    for ind,q in enumerate(pos_args):
                        if len(q.split("*")) == 1:
                            q="1*"+q
                            pos_args[ind] = q
                        pos_args[ind]=pos_args[ind].split("*")
                    cur_row = S.shape[0] # let's add a new row to S...
                    S = S.reshape( (S.shape[0]+1,S.shape[1]) )
                    for k in pos_args:
                        if self.InputFileType == 'ftbl':
                            if k[1] in unidir_fluxes or k[1] in io_fluxes:
                                S[cur_row,self.get_metabolic_flux_index(k[1])] += float(k[0])
                            elif k[1] in bidir_fluxes:                            
                                S[cur_row,self.get_metabolic_flux_index(k[1])] += float(k[0])                            
                                S[cur_row,self.get_metabolic_flux_index(k[1]+"_r")] -= float(k[0])
                        else:
                            S[cur_row,self.get_metabolic_flux_index(k[1])] += float(k[0])
                    for k in neg_args:
                        if self.InputFileType == 'ftbl':        
                            if k[1] in unidir_fluxes or k[1] in io_fluxes:
                                S[cur_row,self.get_metabolic_flux_index(k[1])] -= float(k[0])
                            elif k[1] in bidir_fluxes:                            
                                S[cur_row,self.get_metabolic_flux_index(k[1])] -= float(k[0])
                                S[cur_row,self.get_metabolic_flux_index(k[1]+"_r")] += float(k[0])
                        else:
                            S[cur_row,self.get_metabolic_flux_index(k[1])] -= float(k[0])
        except:
            print "\nERROR while analyzing the EQUALITIES section"
            print "The problematic line was:", line
            sys.exit(-1)
        #: Stoichiometric equality vector
        self.S_b = matrix(S_b).T
        #: Stoichiometric matrix (with equality constraints)
        self.S = sparse.csr_matrix(S)
        
        FCM = [] #sparse.lil_matrix( (1,S.shape[1]) )
        
        FCv = [] #sparse.lil_matrix( (1,1) ) # The (FCM*x = FCv) vector
        
        # Add the C constraints from the FLUXES section to the measurements matrices (as opposed to the EQUALITIES constraints):
        if self.InputFileType == 'ftbl':        
            fluxseg = self.find_segment('FLUXES')
            unidir_fluxes = []
            in_xch_sec = False            
            for l in fluxseg:
                if l[0] == 'NET':
                    in_xch_sec = False
                    in_net_sec = True                    
                if l[0] == 'XCH':
                    in_xch_sec = True
                    in_net_sec = False                    
                if in_xch_sec and l[2]=='C':
                    if float(l[3]) != 0:
                        print "Error while parsing the FLUXES XCH section. Only 0 constraints are allowed."
                        sys.exit(-1)
                if in_net_sec and l[2]=='C':
                    if (l[1] not in unidir_fluxes) and (l[1] not in io_fluxes):
##                        FCM.append([l[1],l[1]+'_r'])
                        print "\nERROR! Constraints over bi-directional fluxes are currently not supported.\n"
                        print "Problem with", l[1]                        
                        sys.exit(-1)
                    else:                        
                        FCM.append(l[1])
                    FCv.append(l[3])
        #: Fluxes constraints matrix from the FLUXES section - added as measurements (FCM*x = FCv)
        self.FCM = FCM
        #: Fluxes constraints values from the FLUXES section - added as measurements (FCM*x = FCv) 
        self.FCv = FCv
        
        # Create full fluxes vector (with isotope labels)
        fluxes = [] # full flux name list
        for f in flux_nw_vec:
            for i in range(2**flux_nw_vec[f]): # 2^flux_width
                fluxes.append(f+"/"+num2bin(i,flux_nw_vec[f]))
        flux_dict = {}
        for k in fluxes:
            flux_dict[k] = fluxes.index(k)
        self.fluxes = flux_dict # dictionary with flux names and indeces

        # Create the equation list. For each flux: [ poolname : ([input], [output])]
        eq_list = []
        for p in self.pools: # for every pool P
            #if p not in self.excell_pools: # if that's not an extra-cellular pools
            if 1:
                for iso in range(2**self.pools[p]): # for each isotope of pool P
                    iso_bin = num2bin(iso,self.pools[p])
                    # find all incoming fluxes:
                    tmp_flux = ""
                    for f in fluxes_data:
                        # Create an isotope template for this flux:                        
                        tmp_flux_carbons = "" # temp carbons string
                        for i in head_range:
                            if FluxHeader[i][0] == 'E':                                    
                                tmp_flux_carbons = tmp_flux_carbons + f[1][i][1:]
                        # If necessary, replace the letters with our selected isotope
                        saved_tmp_flux_carbons = tmp_flux_carbons # let's make sure that if the same flux comes twice, we count it twice.
                        for k in head_range:                            
                            tmp_flux_carbons = saved_tmp_flux_carbons
                            if f[0][k] == p:
                                try:
                                    # let's go over all the carbons of our educt metabolite  and replace its carbons with iso_bin
                                    for ind,i in enumerate(f[1][k][1:]):                                    
                                        tmp_flux_carbons = tmp_flux_carbons.replace(i, iso_bin[ind])                                    
                                    # let's replace the carbons left with \x00 (x's)
                                    a = FluxHeader[k][0];
                                    if a == "E": #if it's an educt (reactant) we add it to the equation
                                        tmp_flux += "*+"
                                    else: #if it's a product, we take it off our equation
                                        tmp_flux += "*-"
                                    tmp_flux += f[0][0] + "/" # add the flux name and "/"
                                    for i in tmp_flux_carbons:
                                        if  i == "0" or i == "1":
                                            tmp_flux +=i
                                        else:
                                            tmp_flux +="\x00"
                                except:                                    
                                    print "\nERROR while parsing the file. Probably has something to do with:", f[0][k],p
                                    sys.exit(-1)
                    in_flux = []                                        
                    out_flux = []
                    for k in tmp_flux.split("*")[1:]:
                        if k[0] == "+":
                            for z in expand_x_values(k[1:]):
                                #in_flux.append(self.fluxes[z])
                                in_flux.append(z)
                        else:
                            for z in expand_x_values(k[1:]):
                                #out_flux.append(self.fluxes[z])
                                out_flux.append(z)
                    eq_list.append([p, in_flux, out_flux, iso_bin])
        #: Global equation list. Each element is built as: [ poolname : ([input fluxomers], [output fluxomers])]
        self.eq_list = eq_list
        
    def get_metabolic_flux_index(self, flux_name):
        """
        This function returns the index of the global metabolic flux "flux_name".
        
        @type flux_name: string
        @param flux_name: The name of the flux we want to find.
        @rtype: number
        @return: The index of the global metabolic flux "flux_name".
        """
#        print flux_name        
        return self.flux_nw_vec.keys().index(flux_name)

    def create_flux_meas_mat(self):
        '''
        This function parses the FLUX_MEASUREMENTS segment of the ftbl file.
        It creates the following matrices:
            1. L{self.flux_meas_mat} - The matrix choosing the measured metabolic fluxes out of the fluxomer vector. 
            2. L{self.flux_meas_values} - The measured values.
            3. L{self.flux_meas_variance} - The measurement variance vector.
        Eventually we will search for::
            M{min [ (self.flux_meas_mat * x - self.flux_meas_values_).T * diag(self.flux_meas_variance) * (self.flux_meas_mat * x - self.flux_meas_values_)] }
        '''
        if debug_level > 1:
            print "Creating the input flux measurement matrices (flux_meas_*)... ",
            sys.stdout.flush()        
        # Find out what labels we have
        # labels is a list with the names of the measured labels
        fluxseg = self.find_segment("FLUX_MEASUREMENTS")    
        self.fluxseg = fluxseg

        flux_meas_mat = zeros( (len(fluxseg)-1+len(self.FCM),len(self.fluxes)) )
        flux_meas_values = zeros( (len(fluxseg)-1+len(self.FCM),1) )
        flux_meas_var = zeros( (len(fluxseg)-1+len(self.FCM),1) )
        
        tU = matrix(self.U.toarray())
        try:
            for linenum, line in enumerate(fluxseg[1:]):
                flux_meas_mat[linenum,:] = tU[self.get_metabolic_flux_index(line[0])]            
                flux_meas_values[linenum] = float(line[1])
                flux_meas_var[linenum] = float(line[2])
        except:
            print "\nError while parsing the FLUX_MEASUREMENTS section."
            print "The problematic line was", line
            sys.exit(-1)
        cur_line = linenum+1
        for (flxname,val) in zip(self.FCM, self.FCv): # Add the equlities equations with very high variance
            #for ind,k in enumerate(flxname):
            #    print k
            #    print self.nw_flux_vec            
            flux_meas_mat[cur_line,:] = tU[self.get_metabolic_flux_index(flxname)]
            flux_meas_values[cur_line] = float(val)
            flux_meas_var[cur_line] = self.ftbl_fluxsec_eq_cons_var ### TEMPORARY VALUE FOR EQUALITY CONSTRAINTS ARISED FROM THE FTBL(ONLY!) FLUXES SECTION
            cur_line+=1            
        #: Flux values measurement matrix
        self.flux_meas_mat = flux_meas_mat
        #: Flux values measurement vector
        self.flux_meas_values = flux_meas_values
        #: Flux values measurement variance vector
        self.flux_meas_var = flux_meas_var
        print ' done.'
        sys.stdout.flush()
        
    def create_global_meas_mat(self):
        """
        This function creates the objective function for the MFA optmization problem
        (the global "G" measurement matrix).

        The objective of the MFA problem is to minimize::
            M{|| tG * x - tB ||^2}

        The matrix L{self.tG} is created by concatanation of the 3 possible measurments matrices:
            1. Label measurements (L{self.labelinp_num} and L{self.labelinp_denom}).
            2. Fluxes measurement (L{self.labelmeas_num} and L{self.labelmeas_denom}).
            3. Mass-Spectrometry measurements (L{self.ms_meas_num} and L{self.ms_meas_denom}).

        The vector L{self.tB} is created by concatanation of appropriate values of the above.
        """

        print "Creating the MFA optimization objective matrix... ",
        sys.stdout.flush()
        # The variance of the measured input labels is assumed to be the constant: self.labelinp_var
        label_inp_sigma = (self.labelinp_var**(-1))*ones( (self.labelinp_value.shape[0],1) )
        new_LS = True
        if self.use_labelinp_var == True:
            # Fisrt we add the label input measurements:
            LS_num_mat = (sdiag(label_inp_sigma)*self.labelinp_num).toarray()
            LS_value = self.labelinp_value.toarray()
            LS_denom_mat = (sdiag(label_inp_sigma)*self.labelinp_denom).toarray()
            new_LS = False
            
        # now, if needed, we add the label measurements:
        if len(self.labelmeas_value) > 1:
            if new_LS:
                LS_num_mat =  sdiag(one_div_vec(self.labelmeas_var))*self.labelmeas_num.toarray()
                LS_value = matrix(self.labelmeas_value).T
                LS_denom_mat = (sdiag(one_div_vec(self.labelmeas_var))*self.labelmeas_denom).toarray()
                new_LS = False
            LS_num_mat = concatenate( (LS_num_mat, sdiag(one_div_vec(self.labelmeas_var))*self.labelmeas_num.toarray()) )
            LS_value = concatenate( (LS_value, matrix(self.labelmeas_value).T))
            LS_denom_mat = concatenate( (LS_denom_mat, (sdiag(one_div_vec(self.labelmeas_var))*self.labelmeas_denom).toarray()) )

        # Last but not least, the mass-spectrometry measurements:
        if len(self.ms_meas_value) > 1:
            if new_LS:
                LS_num_mat = sdiag(one_div_vec(self.ms_meas_var))*self.ms_meas_num.toarray()
                LS_value = matrix(self.ms_meas_value).T
                LS_denom_mat = (sdiag(one_div_vec(self.ms_meas_var))*self.ms_meas_denom).toarray()
                new_LS = False
            LS_num_mat = concatenate( (LS_num_mat, sdiag(one_div_vec(self.ms_meas_var))*self.ms_meas_num.toarray()) )
            LS_value = concatenate( (LS_value, matrix(self.ms_meas_value).T) )
            LS_denom_mat = concatenate( (LS_denom_mat, (sdiag(one_div_vec(self.ms_meas_var))*self.ms_meas_denom).toarray()) )

        G = LS_num_mat - sdiag(LS_value)*LS_denom_mat        
        self.G = G
        #: Global measurement matrix. The optimization problem seeks to minimize || tG*x - tB ||
##        self.tG = sparse.csr_matrix(vstack( (self.G,sparse.spdiags(1/self.flux_meas_var.T,[0],len(self.flux_meas_var),len(self.flux_meas_var))* self.flux_meas_mat) )  )
##        self.tG = sparse.csr_matrix(vstack( (self.G,sparse.spdiags(self.meas_flux_const_value/self.flux_meas_var.T,[0],len(self.flux_meas_var),len(self.flux_meas_var))* self.flux_meas_mat) )  )
        self.tG = sparse.csr_matrix(vstack( (self.G,
                                             sparse.spdiags(1/self.flux_meas_var.T,
                                                            [0],
                                                            len(self.flux_meas_var),
                                                            len(self.flux_meas_var))
                                             * self.flux_meas_mat) )  )
        
        #: Global measurement vector. The optimization problem seeks to minimize || tG*x - tB ||
##        self.tB = vstack( (zeros( (shape(self.G)[0],1) ) ,sparse.spdiags(1/self.flux_meas_var.T,[0],len(self.flux_meas_var),len(self.flux_meas_var))*self.flux_meas_values) )
##        self.tB = vstack( (zeros( (shape(self.G)[0],1) ) ,sparse.spdiags(self.meas_flux_const_value/self.flux_meas_var.T,[0],len(self.flux_meas_var),len(self.flux_meas_var))*self.flux_meas_values) )        
        self.tB = vstack( (zeros( (shape(self.G)[0],1) ) ,
                           sparse.spdiags(1/self.flux_meas_var.T,
                                          [0],
                                          len(self.flux_meas_var),
                                          len(self.flux_meas_var))
                           *self.flux_meas_values) )
       
        # vector that indicates the number of input flux the measurement was taken from. used for speeding up things.
        self.labelinp_num_fluxes = nonzero(self.U*(self.labelinp_denom).T)[0]
        print " done."
        sys.stdout.flush()
                
    def trnasform_eq_into_matrix(self, eq):
        """
        This function transforms string equation of fluxes into a fluxomers stoichiometric matrix.
        For example::
            "*+flux1/0*-flux2/1" -> [1 0 0 0; 0 0 0 -1]) 
        
        @type eq: string
        @param eq: An string equation. Every equation element is seperated by either "*+" or "*-" with the fluxomer name following it. 
            No preciding numbers are allowed, only plain fluxomer names. 
            For multiple instances of the same fluxomer, simply add/substract it more than once to the equation.
        @rtype: CSR matrix
        @return: The fluxomer stoichiometric matrix represented by eq.
        """
        S = sparse.lil_matrix((len(eq), len(self.fluxes)) )
        for (ind, k) in enumerate(eq):
            t = k.split("*")[1:]
            for h in t:                
                m = self.fluxes[h[1:]]                
                if h[0] == "+":
                    S[ind, m] = S[ind, m] + 1
                else:
                    S[ind, m] = S[ind, m] - 1
        return S.tocsr()

    
    def create_label_meas_eq(self):
        '''
        This function parses the LABEL_MEASUREMENTS segment of the ftbl file.
        The output of the function is the matrices and vectors:
            1. L{self.labelmeas_num} - The label measurements numerator matrix
            2. L{self.labelmeas_denom} - The label measurements denominator matrix
            3. L{self.labelmeas_value} - The label measurements values vector
            4. L{self.labelmeas_var} - The label measurements values vector

        The objective function will be to minimize::
            M{ ||diag(self.labelmeas_var) * [self.labelmeas_num - diag(self.labelmeas_value) * self.labelmeas_denom]*x ||^2 }
        '''
        if debug_level > 1:
            print "Creating the label measurement matrices (label_meas_*)... ",
            sys.stdout.flush()            
        # Find out what labels we have
        # labels is a list with the names of the measured labels
        labseg = self.find_segment("LABEL_MEASUREMENTS")
        self.labseg = labseg
        labels = []
        meas_numerator = []
        meas_value = []
        meas_var = []
        cur_label = ""
        for linenum, line in enumerate(labseg):
            if line[0] != "":
                cur_label = line[0]
                labels.append(cur_label)
            if len(line) >= 4:
                meas_numerator.append(cur_label +"/"+line[4][1:].replace("x","\x00"))
                meas_value.append(line[2])
                meas_var.append(line[3])
        
        # now meas_numerator is a list constructed as follows: [ 'EDUCT_NAME1/ISOTOPE1',  'EDUCT_NAME2/ISOTOPE2',...]
        # meas_value are the measurement values for the numerators of meas_numerator
        # meas_var is the variance for the above vectors.
        res_num = []
        res_denom = []
        # now we need to translate the pool and isotope to our flux variables.    
        
        for (mn,mv, mvar) in zip(meas_numerator[1:], meas_value[1:], meas_var[1:]):
            (fname, iso_bin) = mn.split("/")
            iso_bin_group = expand_x_values(iso_bin)
            iso_dem_group = expand_x_values(len(iso_bin)*'\x00')                                        
            tmp_num_eq = ""
            tmp_dem_eq = ""
            #Found=False
            for k in iso_bin_group:
                tmp_num_eq += "*+MEAS_FLUX_"+fname+"/"+k
            for k in iso_dem_group:
                tmp_dem_eq += "*+MEAS_FLUX_"+fname+"/"+k
##            print tmp_num_eq
##            for k in self.eq_list:
##                #if k[0] == fname and (Found==False) and k[3] in iso_bin_group:
##                if k[0] == fname and k[3] in iso_bin_group:
##                    #Found=True
##                    tmp_num_eq += "*+".join(k[1]) + "*+" #+q #q.split("/")[0]+"/"+iso_bin           
##            #for k in self.eq_list:
##                if k[0] == fname and k[3] in iso_dem_group:
##                    tmp_dem_eq += "*+".join(k[1]) + "*+" #+q #q.split("/")[0]+"/"+iso_bin
##            
##            res_num.append(tmp_num_eq[:-2])
##            res_denom.append(tmp_dem_eq[:-2])
            res_num.append(tmp_num_eq)
            res_denom.append(tmp_dem_eq)
        
        if len(res_num)>0:
            #: label measurement numerator matrix            
            self.labelmeas_num = self.trnasform_eq_into_matrix(res_num)
            #: label measurement denominator matrix
            self.labelmeas_denom = self.trnasform_eq_into_matrix(res_denom)
            #: label measurement values vector
            self.labelmeas_value = sparse.lil_matrix((len(meas_value[1:]),1))
            #self.labelmeas_value = zeros((len(meas_value[1:]),1))
            for (x,y) in enumerate(meas_value[1:]):
                if float(y) == 0:
                    y='1e-7'
                self.labelmeas_value[x,0] = float(y)
            self.labelmeas_value.tocsr()
            #self.labelmeas_value = sparse.dok_matrix(self.labelmeas_value)
            #: label measurement variance vector
            self.labelmeas_var = sparse.lil_matrix((len(meas_var[1:]),1) )
            for (x,y) in enumerate(meas_var[1:]):
                self.labelmeas_var[(x,0)] = float(y)
            self.labelmeas_var.tocsr()
            self.labelmeas_var = self.labelmeas_var * self.meas_flux_const_value
            self.labelmeas_value = self.labelmeas_value.T.toarray()[0]
        else:
            self.labelmeas_num = []
            self.labelmeas_denom = []
            self.labelmeas_var = []
            self.labelmeas_value = []
        print " done."
        sys.stdout.flush()
          
    def create_ms_meas_eq(self):
        '''
        This function parses the MASS_SPECTROMETRY segment of the ftbl file.
        The output of the function is the matrices and vectors:
            1. L{self.ms_meas_num} - The MS measurements numerator matrix
            2. L{self.ms_meas_denom} - The MS measurements denominator matrix
            3. L{self.ms_meas_value} - The MS measurements values vector
            4. L{self.ms_meas_var} - The MS measurements values vector

        We always treat these measurement as a ratio measurement between the MS values, relative to the sum of the MS measurements
        (hence we add to the objective the ratios: mass1/(mass0+mass1+mass2+...), mass2/(mass0+mass1+mass2+...) etc').
        
        The objective function will be to minimize::
            M{ ||diag(self.ms_meas_var) * [self.ms_meas_num - diag(self.ms_meas_value) * self.ms_meas_denom]*x ||^2 }

        '''
        if debug_level > 1:
            print "Creating the mass spectrometry measurement matrices (ms_meas_*)... ",
            sys.stdout.flush()            
        # Find out what labels we have
        # labels is a list with the names of the measured labels
        msseg = self.find_segment("MASS_SPECTROMETRY")
        self.msseg = msseg
        
        labels = []
        ms_numerator = []
        ms_value = []
        ms_var = []
        ms_denom = []
        self.ms_normalization_fact = []
        cur_label = ""
        denom_items_counter = 0
        for linenum, line in enumerate(msseg[1:]):            
            if line[0] != "":
                cur_label = line[0]
                for k in range(denom_items_counter): # let's make the denom of the last item we've visited
                    t_save_denom = [save_denom[0][0]]
                    for q in save_denom:
                        t_save_denom+=q[1:]
                    ms_denom.append(t_save_denom)
                    ms_value.append(save_value_vec[k]/save_value)
                    self.ms_normalization_fact.append(save_value)
                save_denom = []
                save_value_vec = []
                save_value = 0
                denom_items_counter = 0
            if line[1] != "":
                cur_indices = [int(k)-1 for k in line[1].split(",")]
            if len(line) >= 4:
                denom_items_counter+=1
                ms_is_vec = create_ms_isotopomers_dict(self.pools[cur_label], cur_indices)
                nume = [cur_label]+[q.replace("x","\x00") for q in ms_is_vec[int(line[2])]]
                ms_numerator.append(nume)
                save_denom.append(nume)
                save_value+=float(line[3])
                save_value_vec.append(float(line[3]))      
                ms_var.append(line[4])

        for k in range(denom_items_counter): # let's make the denom of the last item we've visited
            t_save_denom = [save_denom[0][0]]
            for q in save_denom:
                t_save_denom+=q[1:]
            ms_denom.append(t_save_denom)
            ms_value.append(save_value_vec[k]/save_value)
            self.ms_normalization_fact.append(save_value)
                    
        # now meas_numerator is a list constructed as follows: [ 'EDUCT_NAME1/ISOTOPE1',  'EDUCT_NAME2/ISOTOPE2',...]
        # meas_value are the measurement values for the numerators of meas_numerator
        # meas_var is the variance for the above vectors.
        res_num = []
        res_denom = []
        # now we need to translate the pool and isotope to our flux variables.    
        
        for (mn,md, mv, mvar) in zip(ms_numerator, ms_denom, ms_value, ms_var):
            #fname_isobin = [k.split("/") for k in mn]            
            iso_bin_group = []            
            for k in mn[1:]:                
                iso_bin_group+=expand_x_values(k)
            fname=mn[0]
            iso_dem_group=[]
            fdnom_name=md[0]
            for k in md[1:]:
                iso_dem_group+=expand_x_values(k)
            
##            tmp_num_eq = "*+"
##            tmp_dem_eq = "*+"
            tmp_num_eq = ""
            tmp_dem_eq = ""
            #Found=False
            for k in iso_bin_group:
                tmp_num_eq += "*+MEAS_FLUX_"+fname+"/"+k
            for k in iso_dem_group:
                tmp_dem_eq += "*+MEAS_FLUX_"+fname+"/"+k
            
##                
##            for k in self.eq_list:
##                #if k[0] == fname and (Found==False) and k[3] in iso_bin_group:                
##                    if k[0] == fname and k[3] in iso_bin_group:
##                        #Found=True
##                        if fname in self.excell_pools:
##                            if len(k[1])>0:
##                                tmp_num_eq += "*+".join(k[1]) + "*+" #+q #q.split("/")[0]+"/"+iso_bin
##                            else:
##                                tmp_num_eq += "*+".join(k[2]) + "*+" #+q #q.split("/")[0]+"/"+iso_bin
##                        else:                            
##                            tmp_num_eq += "*+".join(k[1]) + "*+" #+q #q.split("/")[0]+"/"+iso_bin           
##    #            for k in self.eq_list:
##                    if k[0] == fdnom_name and k[3] in iso_dem_group:
##                        if fdnom_name in self.excell_pools:
##                            if len(k[1])>0:
##                                tmp_dem_eq += "*+".join(k[1]) + "*+" #+q #q.split("/")[0]+"/"+iso_bin
##                            else:
##                                tmp_dem_eq += "*+".join(k[2]) + "*+" #+q #q.split("/")[0]+"/"+iso_bin
##                        else:
##                            tmp_dem_eq += "*+".join(k[1]) + "*+" #+q #q.split("/")[0]+"/"+iso_bin
##                    
            res_num.append(tmp_num_eq)
            res_denom.append(tmp_dem_eq)
##            res_num.append(tmp_num_eq[:-2])
##            res_denom.append(tmp_dem_eq[:-2])
        if len(res_num)>0:
            #print res_num
            #: Mass-spectrometry numerator matrix
            self.ms_meas_num = self.trnasform_eq_into_matrix(res_num)
            #: Mass-spectrometry dominator matrix
            self.ms_meas_denom = self.trnasform_eq_into_matrix(res_denom)
            #: Mass-spectrometry measurement values vector
            self.ms_meas_value = sparse.lil_matrix((len(ms_value),1))
        
            #self.labelmeas_value = zeros((len(meas_value[1:]),1))
            for (x,y) in enumerate(ms_value):
                if float(y) == 0:
                    y='1e-7'
                self.ms_meas_value[x,0] = float(y)
            self.ms_meas_value.tocsr()
            #self.labelmeas_value = sparse.dok_matrix(self.labelmeas_value)
            self.ms_meas_var = sparse.lil_matrix((len(ms_var),1) )
            for (x,y) in enumerate(ms_var):
                self.ms_meas_var[(x,0)] = float(y)
            #: Mass-spectrometry measurement variance vector
            self.ms_meas_var.tocsr()
            self.ms_meas_var = self.ms_meas_var * self.meas_flux_const_value
            self.ms_meas_value = self.ms_meas_value.T.toarray()[0]
        else:
            self.ms_meas_num = []
            self.ms_meas_denom = []
            self.ms_meas_value = []
            self.ms_meas_var = []
        print " done."
        sys.stdout.flush()
        
    def create_U_matrix(self):
        '''
        This function constructs the U matrix, which transforms the fluxomers vector 
        into the metaoblic fluxes vector. hence::
            M{u = U*x}
        where::
            u - the metabolic flux vector
            x - the fluxomers vector        
        '''
        if debug_level > 1:
            print "Creating the U matrix...",
            sys.stdout.flush()
        eq = []
        for f in self.flux_nw_vec:
            m = "*+" + f + "/"+"\x00"*(self.flux_nw_vec[f])
            eq.append("*"+"*".join(expand_x_values(m)))
        self.U = sparse.csr_matrix(self.trnasform_eq_into_matrix(eq))        
        print " done."
        sys.stdout.flush()
        
    def create_inp_eq_matrix(self):
        '''
        This function parses the LABEL_INPUT segment.
        The output of the function is the matrices and vectors:
            1. L{self.labelinp_num} - The flux value measurements numerator matrix
            2. L{self.labelinp_denom} - The flux value measurements denominator matrix
            3. L{self.labelinp_value} - The flux value measurements values vector

        The objective function will be to minimize::
            M{ ||diag(self.labelinp_var) * [self.labelinp_num - diag(self.labelinp_value) * self.labelinp_denom]*x ||^2 }
            
        @note: since not supplied by the FTBL files, the value of self.labelinp_value is a global assigned constant. 
        
        '''
        if debug_level > 1:
            print "Creating the input label measurements matrices (labelinp_*)... ",
            sys.stdout.flush()
        # Find out what labels we have
        # labels is a list with the names of the measured labels                
        labelinpseg = self.find_segment("LABEL_INPUT")[1:]
        labels = []
        meas_eq = []
        meas_value = []
        meas_label = []
        cur_label = ""
        for linenum, line in enumerate(labelinpseg):
            if line[0] != "":
                cur_label = line[0]
            if len(line) >= 3:
                meas_label.append( cur_label )
                meas_eq.append( line[1][1:] )
                meas_value.append(line[2])
        
        # now meas_eq is a list of the equality isotopes 
        # meas_value are the measurement values for the numerators of meas_numerator
        # meas_label is the name of the label for which meas_eq holds
        
        res_num  = []
        res_denom = []
        # now we need to translate the pool and isotope to our flux variables.
        for fname, iso_bin in zip(meas_label, meas_eq):
            tmp_num_eq = ""
            tmp_dem_eq = ""
            Found=False
            a = []            
            for k in self.eq_list:
                for q in k[1]:
                    if k[0] == fname and Found==False: 
                        Found=True                        
                        tmp_num_eq += q.split("/")[0]+"/"+iso_bin
                        a.append(q.split("/")[0])
            res_num.append("*+"+"*+".join(expand_x_values(tmp_num_eq)))
            a = set(a)
            tmp_dem_eq_a = []
            for x in a:                
                m = "*+" + x + "/"+"\x00"*(self.flux_nw_vec[x])
                tmp_dem_eq_a.append("*"+"*".join(expand_x_values(m)))
            res_denom.append("".join(tmp_dem_eq_a))
        #: Input label measurement numerator matrix
        self.labelinp_num = self.trnasform_eq_into_matrix(res_num)
        #: Input label measurement denominator matrix
        self.labelinp_denom = self.trnasform_eq_into_matrix(res_denom)
        #: Input label measurement value vector
        self.labelinp_value = sparse.lil_matrix((len(meas_value),1) )
        
        for (x,y) in enumerate(meas_value):
            self.labelinp_value[x,0] = float(y)
        self.labelinp_value.tocsr()
        
        print " done."
        sys.stdout.flush()

    def create_label_transition_mat(self):
        '''
        This function creates the propogation transition matrix::
            \M{x(t) = P x(t-1)} .

        P is provided using the following sub matrices:
            1. L{self.trans_mat_a} - the M{H_1} matrix (in CSR format).
            2. L{self.trans_mat_b} - the M{H_2} matrix (in CSR format).
            3. L{self.trans_mat_a_gamma} - the M{g_1} matrix (in CSR format).
            4. L{self.trans_mat_a_beta} - the M{g_2} matrix (in CSR format).
            5. L{self.trans_mat_b_beta} - the M{g_3} matrix (in CSR format)

        In addition, this function creates the beta (ratios) vector:
            1. L{self.beta_vec} - the text names of the beta vector elements
            2. L{self.beta_vec_M} - the numerator for construction of the beta vec.
            3. L{self.beta_vec_N} - the denominator for constuction of the beta vec.
        eventually, we have::
            M{beta_vec = (self.beta_vec_M * u) / (self.beta_vec_N * u)}

        '''
        print "Creating the transition matrices... ",
        sys.stdout.flush()
        # First line is the header:
        netseg = self.find_segment('NETWORK')        
        IOvec = []
        for i  in netseg[0][1:]:
            if i != "":
                IOvec.append( i[0] ) # educt, #product

        fluxes_data = self.fluxes_data
        flux_nw_vec = self.flux_nw_vec
        
        fluxes = self.fluxes
        
        # Create the beta vector:
        beta_vec = []
        
        for f in fluxes_data:
            cur_width = 0            
            for ind,a in enumerate(IOvec):                
                if a[0] == "E" and f[0][ind+1] != "": #if it's an educt (going into this pool)                    
                    beta_vec.append(f[0][ind+1] + "/" + f[0][0]) # beta_vec = "POOLNAME/FLUXNAME"
        
        beta_vec = list(set(beta_vec)) # is this correct..?
        beta_vec.sort()
        
        for f in self.flux_nw_vec:
            beta_vec.append(f) # beta has all the fluxes (and later we'll add all the ratios)
        #: Beta vector names
        self.beta_vec = dict([[beta_vec[k],k] for k in range(len(beta_vec))])
        #: Beta vector numerator matrix
        self.beta_vec_M = sparse.lil_matrix((len(self.beta_vec),len(self.flux_nw_vec)+1))
        #: Beta vector denominator matix
        self.beta_vec_N = sparse.lil_matrix((len(self.beta_vec),len(self.flux_nw_vec)))
        
        z = [k.split("/") for k in beta_vec]

        fl=flux_nw_vec.keys()
        for i,k in enumerate(self.beta_vec):
            t = [q[-1] for q in z if q[0] == z[i][0]]            
            for p in t:
                self.beta_vec_N[i,fl.index(p)]=1

            self.beta_vec_M[i,fl.index(z[i][-1])]=1            

        self.beta_vec_M.tocsr()
        self.beta_vec_N.tocsr()
        l=len(self.fluxes)
        visited_fluxes = []

        if display_prog:
            progbar = progress_bar.ProgressBar(prog_width)

        sys.stdout.flush()
        #progbar = progressbar.ProgressBar().start()

        # coo matrix initialization
        # Coo matrices are used here in order to speed up the construction process of the matrices.
        trans_mat_a_row = []
        trans_mat_a_col = []
        trans_mat_a_beta_row = []
        trans_mat_a_beta_col = []
        trans_mat_a_gamma_row = []
        trans_mat_a_gamma_col = []
        trans_mat_b_row = []
        trans_mat_b_col = []
        trans_mat_b_beta_row = []
        trans_mat_b_beta_col = []
        
        for i,e in enumerate(self.eq_list):        
            # enable if propogation prints are needed: 
            #if i%20 == 0:
                #print str(100*float(i)/len(self.eq_list)) + "% done."
            if display_prog:
                progbar.update(100*float(i)/len(self.eq_list))
            #print len(e[1]), len(e[2])
            k = [p.split("/")[0] for p in e[1]]            
            for f in e[1]:                
                fspl = f.split("/")[0]
                f_ind = self.fluxes[f]
                if k.count(fspl) == 1: # if that's simple pool (hence the whole flux is coming out of it)
                    for p in e[2]:
                        trans_mat_a_row.append(f_ind)
                        trans_mat_a_col.append(self.fluxes[p]) # -> trans_mat_a calculates the input flux to the specific pool
                        trans_mat_a_beta_row.append(f_ind)
                        trans_mat_a_beta_col.append(self.beta_vec[e[0]+"/"+fspl]) # beta -> the ratio of flux fspl in the current pool
                        trans_mat_a_gamma_row.append(f_ind) # gamma is just 1
                        trans_mat_a_gamma_col.append(len(self.beta_vec)) # gamma is just 1
                        
                    trans_mat_b_row.append(f_ind) # b should be just 1 here...
                    trans_mat_b_col.append(len(self.fluxes))
                    trans_mat_b_beta_row.append(f_ind)
                    trans_mat_b_beta_col.append(len(self.beta_vec))
                else:
                #if k.count(f.split('/')[0]) > 1:
                    # if it's 1->2 or 2->2 (hence the flux is not determined solely by 1 equation)
                    if f not in visited_fluxes:
                        visited_fluxes.append(f)
                        for p in e[2]:
                            trans_mat_a_row.append(f_ind)
                            trans_mat_a_col.append(self.fluxes[p])
                            trans_mat_a_beta_row.append(f_ind)
                            trans_mat_a_beta_col.append(self.beta_vec[e[0]+"/"+fspl])
                            try:
                                trans_mat_a_gamma_row.append(f_ind)
                                trans_mat_a_gamma_col.append(self.beta_vec[fspl])
                            except:
                                print fspl, self.beta_vec[fspl], len(self.beta_vec)
                                sys.exit(0)
                                aklsjdkj
                    else:
                        for p in e[2]:
                            trans_mat_b_row.append(f_ind)
                            trans_mat_b_col.append(self.fluxes[p])
                            trans_mat_b_beta_row.append(f_ind)
                            trans_mat_b_beta_col.append(self.beta_vec[e[0]+"/"+fspl])
        # now let's put 1's for the input fluxes...:
        # converting and making sure the max element of a_beta, a_gamma and b_beta is 1        
        trans_mat_a = sparse.csr_matrix( ([1]*len(trans_mat_a_row), (trans_mat_a_row, trans_mat_a_col) ),shape=(len(self.fluxes),len(self.fluxes)), dtype=float64)
        trans_mat_a_beta = sparse.coo_matrix(([1]*len(trans_mat_a_beta_row), (trans_mat_a_beta_row, trans_mat_a_beta_col) ), shape=(len(self.fluxes),len(self.beta_vec)+1),dtype=float64).todok()
        trans_mat_a_gamma = sparse.coo_matrix(([1]*len(trans_mat_a_gamma_row), (trans_mat_a_gamma_row, trans_mat_a_gamma_col) ), shape=(len(self.fluxes),len(self.beta_vec)+1),dtype=float64).todok()
        trans_mat_b = sparse.csr_matrix(([1]*len(trans_mat_b_row), (trans_mat_b_row, trans_mat_b_col) ), shape=(len(self.fluxes),len(self.fluxes)+1),dtype=float64) # we add one in case we don't need a product...
        trans_mat_b_beta = sparse.coo_matrix(([1]*len(trans_mat_b_beta_row), (trans_mat_b_beta_row, trans_mat_b_beta_col) ),shape=(len(self.fluxes),len(self.beta_vec)+1),dtype=float64).todok()

        a=trans_mat_a_beta.sum(1).tolist()        
        for i,k in enumerate(a):            
            if k==0 or k==[0]:                
                trans_mat_a[i,i]=1
                trans_mat_a_beta[i,len(self.beta_vec)]=1
                trans_mat_a_gamma[i,len(self.beta_vec)]=1
                trans_mat_b[i,len(self.fluxes)] = 1
                trans_mat_b_beta[i,len(self.beta_vec)] = 1

        #: The M{H_1} matrix (in CSR format).
        self.trans_mat_a = trans_mat_a.tocsr()
        #self.trans_mat_a = trans_mat_a
        #: The M{g_2} matrix (in CSR format).
        self.trans_mat_a_beta = trans_mat_a_beta.tocsr()
        #: The M{g_1} matrix (in CSR format).
        self.trans_mat_a_gamma = trans_mat_a_gamma.tocsr()
        #: The M{H_2} matrix (in CSR format).
        self.trans_mat_b = trans_mat_b.tocsr()
        #self.trans_mat_b = trans_mat_b
        #: The M{g_3} matrix (in CSR format).
        self.trans_mat_b_beta = trans_mat_b_beta.tocsr()
        
        if display_prog:
            progbar.update(100)
        print " done."
        sys.stdout.flush()

    def create_lu_trans_matrix(self):
        """
        This function computes the LU decomposition of the propogation transition matrices.
        We find a factorization for::
            L{self.trans_mat_a} and L{self.trans_mat_b}
        by finding the LU factorizaiton of the matix constructed out of concatanation of the two above.

        The result are the matrices:
            1. L{self.al1} - the left side of the pseudo LU of self.trans_mat_a
            2. L{self.al2} - the left side of the pseudo LU of self.trans_mat_b
            3. L{self.au} - the right side of the pseudo LU factorizaiton of both self.trans_mat_a and self.trans_mat_b .
        """
        print "Calculating the LU decomposition of the transition matrices... ",
        sys.stdout.flush()
        #tmp_trans_mat = concatenate( (self.trans_mat_a,self.trans_mat_b[:,:-1]) )
        tmp_trans_mat = sparse.vstack( (self.trans_mat_a,self.trans_mat_b[:,:-1]) ).tocsr()
        #sp_tmp_trans_mat = sparse.csc_matrix(tmp_trans_mat)
#        keeprows = nonzero(abs(tmp_trans_mat).sum(1))[0]
        
        L, U, P, Q, R, do_recip = umfpack.lu( tmp_trans_mat )
        
        umfpack.free()
        tP=sparse.csc_matrix( ([1]*len(P),(P,range(len(P)))), (len(P),len(P)) )
        tQ=sparse.csc_matrix( ([1]*len(Q),(Q,range(len(Q)))), (len(Q),len(Q)) )
        
        if do_recip: # LU = PRAQ
            #P=P*sparse.spdiags(R,[0],len(R),len(R))
            self.al = sparse.spdiags(1.0/R,[0],len(R),len(R))*tP*L
            #self.al = tP*L    
        else: # LU=P(R^-1)AQ        
            self.al = sparse.spdiags(R,[0],len(R),len(R))*tP*L
            #self.al = tP*L
        #: The right side of the pseudo LU factorizaiton of both self.trans_mat_a and self.trans_mat_b .
        self.au = U*tQ.T
        #self.au = U
        #: The left side of the pseudo-LU factorizaiton of self.trans_mat_a
        self.al1 = self.al[:len(self.fluxes),:]
        #: For easy of computation, self.al1au = self.al1 * self.au
        self.al1au = self.al1*self.au
        #: The left side of the pseudo-LU factorizaiton of self.trans_mat_b
        self.al2 = self.al[len(self.fluxes):,:]
        #: For easy of computation, self.al2au = self.al2 * self.au
        self.al2au = self.al2*self.au
        self.al2_0 = self.trans_mat_b[:,len(self.fluxes)].tocsc()
        print " done."
        sys.stdout.flush()

    def evaluate(self):
        """
        This function runs the main optimization process.
        It uses the L{scipy.optimize.fmin_slsqp} for the actual optimization process, and prints out the results in a TABed formatted table.
        """
        print "----------------------------------"
        print "\nOptimizing network:", self.ProjName, "\n"
        print "----------------------------------"

        if self.flux_upper_bound == 0: # choose automatically
            self.flux_upper_bound = max(self.flux_meas_values)[0]*5
            
        sys.stdout.flush()        
        find_init_point(self)
        iu = self.init_u # input u
        
        if run_optimization == False:
            return
        
        start_time = time.time()        
        print "Starting the optimization process:"
##        r = optimize.fmin_slsqp(func=get_normG, x0=iu, f_eqcons=lambda x,y:self.S.todense()*matrix(x).T, bounds = [(0,10)]*len(self.init_u), fprime=get_normG_grad, fprime_eqcons=lambda x,y:self.S.todense(), iprint=3, args=[self])
##        r = optimize.fmin_slsqp(func=get_normG, x0=iu, f_eqcons=lambda x,y:self.S.todense()*matrix(x).T,
##                                bounds = [(0,10)]*len(self.init_u), fprime=get_normG_grad,
##                                fprime_eqcons=lambda x,y:self.S.todense(), iprint=3, args=[self])

##        r = optimize.fmin_slsqp(func=get_normG, x0=iu, f_eqcons=lambda x,y:self.S.todense()*matrix(x).T - self.S_b,
##                        bounds = [(0,10)]*len(self.init_u), fprime=get_normG_grad,
##                        fprime_eqcons=lambda x,y:self.S.todense(), iprint=3, args=[self], iter=400, acc=1e-12, epsilon=1e-12 )

        # This is the main optimization process call function.        
##        return        
        #return
        r = optimize.fmin_slsqp(func=get_normG, x0=iu, f_eqcons=lambda x,y:(self.S.todense()*matrix(x).T - self.S_b),
                        bounds = [(0,self.flux_upper_bound)]*len(self.init_u), fprime=get_normG_grad,
                        fprime_eqcons=lambda x,y:self.S.todense(), iprint=4, args=[self], iter=400, epsilon=1e-12, acc=1e-10)
##        dres = get_normG_grad(r,self)
        self.res = r        
        #self.dres = dres
        end_time=time.time()
        print "----------------------------------"
        print "Time for convergence: ", end_time-start_time, ' sec'
        print "----------------------------------"
        print "Results:"
        print_fluxes_table(self, r)
##        # let's sort it...        
##        tmp = self.flux_nw_vec.keys()
##        tmp.sort()
##        for a in tmp:
##            if a[-2:] != '_s':
##                print a,'\t', r[self.get_metabolic_flux_index(a)]#, dres[self.get_metabolic_flux_index(a)]
##                
        #for ind,k in enumerate(self.flux_nw_vec.keys()):
        #    print k, '\t', r[ind]
##        print "MS Results:"
##        calc_13c_ms_residum(self, self.tv)
        
def print_fluxes_table(self, r):
    print "Flux name\tFlux value"
    tmp = self.flux_nw_vec.keys()
    tmp.sort()
    for a in tmp:
        if a[-2:] != '_s':
            print a,'\t', r[self.get_metabolic_flux_index(a)]#, dres[self.get_metabolic_flux_index(a)]
    
def print_tG_table(self, r):
    a = ((self.tG.todense()>0)*r)    
    b = ((self.tG.todense()!=0)*r)
    c = self.tG*r    
    for k in range(len(a)):        
        print a[k]/b[k],self.tB[k], c[k], c[k]-self.tB[k]
    print "--------------"

def calc_13c_ms_residum(self, r):
    a = ((self.ms_meas_num.todense()>0)*r)    
    b = ((self.ms_meas_denom.todense()!=0)*r)
    tsum = 0
    for k in range(len(a)):        
        print a[k]/b[k]*self.ms_normalization_fact[k],self.ms_meas_value[k], ((a[k]/b[k]*self.ms_normalization_fact[k]-self.ms_meas_value[k])/0.02)**2
        tsum +=((a[k]/b[k]*self.ms_normalization_fact[k]-self.ms_meas_value[k])/0.02)**2    
    print "--------------"
    print "Total sum of squres:", tsum

def one_div_vec(vec):    
    """
    This function returns a vector which is 1/vec.
    @param vec: input vector
    @type vec: scipy.array
    @rtype: scipy.array
    @return: 1/vec (element wise division).
    """
    return scipy.divide(ones(vec.shape),vec.toarray())

def sdiag(a):    
    """
    Fast sparse diagonal matrix creator.
    Simply returns a sparse matrix with a as its diagonal.
    @type a: array
    @param a: input array
    @rtype: scipy.sparse matrix
    @return: sparse matrix with a in its diagonal.
    """
    return scipy.sparse.spdiags(scipy.matrix(a).T,0,len(a),len(a))    

def expand_x_values(cteq):
    '''
    This function transforms input with (\x00)'s into all the possible sets of inputs with 0's and 1's replacing the (\x00)s.
    @type cteq: string
    @param cteq: Input string. (\x00)'s will be replaced with 1's and 0's.
    @rtype: list
    @return: List of possible variations of cteq, with all the possible 0's and 1's replacements for \x00's in the input string.
    '''
    if cteq[0] == "*":
        in_teq_s = cteq.split("*")[1:]
    else:
        in_teq_s = cteq.split("*")
        
    f_eq = []
    for teq in in_teq_s:
        eq = []
        if teq.find("\x00") != -1: # if we need to change something:
            for k in range(2**teq.count("\x00")):
                s = ""
                curbin = num2bin(k,teq.count("\x00"))
                for m in teq:
                    if m == "\x00":
                        s = s + curbin[0]
                        curbin = curbin[1:]
                    else:
                        s = s + m
                eq.append(s)
        else:
            eq.append(teq)
        f_eq = f_eq + list(eq)
    return f_eq

def create_ms_isotopomers_dict(no_of_atoms, indices_to_change):
    """
    This function returns all the possible permutations of no_of_atoms atoms, seperated by the number of 1's in them.
    (indices_to_change) specifies which of the atoms can be changed (and the counting is then done only on these atoms).
    For example, if we call::
        create_ms_isotopomers_dict(4,[0,1,2])
    we get::
        [['000x'], ['001x', '010x', '100x'], ['011x', '101x', '110x'], ['111x']]
    The first element represents all 0's vec (for indices_to_change),
    the second a vector with only one 1 element, the third with two 1's elements, and the fourth with three 1's elements. 
    
    This function is used for mass-spectrometry measurements analysis.
    
    @param no_of_atoms: Number of atoms in the counted molecule.
    @type no_of_atoms: number
    @param indices_to_change: List of indicies on which we want to count.
    @type indices_to_change: list
    @return: List of lists - all possible permutations for the molecule, sorted by number of ones.
    @rtype: list
    """
    full_label = no_of_atoms * "x"
    label_out = []
    ones_counter = []
    for k in range(2**len(indices_to_change)):
        ones_counter.append(bin(k)[2:].count('1')) # how many ones are in the binary representation of k?
    
    for k in range(len(indices_to_change)+1):
        tmp = []
        for ind,z in enumerate(ones_counter):
            if z==k:
                tmp.append(bin(ind)[2:].rjust(len(indices_to_change),'0'))
        # tmp containts all the possible isotopomers with k labeled atoms.
        tmp2 = []
        
        for z in tmp:
            t = ""
            q=0
            for ind in range(len(full_label)):                    
                if ind in indices_to_change:
                    if z[q]=='1':
                        t+='1'
                    else:
                        t+='0'
                    q=q+1
                else:
                    t+=full_label[ind]
            tmp2.append(t)
        label_out.append(tmp2)
    return label_out  

def num2bin(num,length):
    """
    Converts number to binary format with at (length) bits.
    @param num: The number to be converted
    @type num: number
    @param length: Length of the output binary number.
    @type length: number
    @return: Binary string representing the input number. Zeros are left appended if necessary in order to maintain the required length.
    @rtype: string
    """
    r = ""
    while num > 0:
        r = str(num % 2) + r
        num = num / 2
    return r.rjust(length,'0')

def save_text_vec(filename, vec):
    """
    This fucntion saves the input vector in text TAB seperated format.
    Used for MATLAB debugging purposes. 
    """
    f = open(filename,"w")
    for k in vec:        f.write(k+'\n')
    f.close()

def matlab_save_sparsed_matrix(filename, mat):
    """
    This fucntion saves the input matrix in text TAB seperated format loadable by MATLAB.
    Used for MATLAB debugging purposes. 
    """
    f = open(filename,'w')
    for i in mat.iteritems():
        f.write(str(i[0][0]+1)+'\t'+str(i[0][1]+1)+'\t'+str(i[1])+'\n')
    if mat[mat.shape[0]-1,mat.shape[1]-1] == 0:
        f.write(str(mat.shape[0])+'\t'+str(mat.shape[1])+'\t'+'0'+'\n')
        
    f.close()

def sort_dict_by_values(d):
    """
    This function sorts the input dictionary by its values.
    """ 
    return sorted(d.items(), key=lambda (k,v): (v,k))

def null(A, eps=1e-15):
    """
    This function finds the null space of the incoming matrix A.
    """
    u, s, vh = scipy.linalg.svd(A,full_matrices=True)
    s = hstack( (matrix(s),zeros((1,A.shape[1]-len(s)))) )
    null_mask = (s <= eps).A1
    null_space = scipy.compress(null_mask, vh, axis=0)	    
    return scipy.transpose(array(null_space))

def find_init_point(self):
    """
    This function generates the initial point given for the optimization process.
    It is based upon the standard init point finding methods of iterior point algorithms: 
    In order to find a valid solution for M{S*u = 0}, M{U* u <= s} we solve::
        min(s)
    
        s.t.:
    
         [I,-I]*u + [0*i,10*i] >= s
         Su = S_b    
    We then use this initial point with the propogation equation, resulting with the first point for the algorithm.
    """
    print('Generating initial guess...'),
    self.len_v = len(self.flux_nw_vec)
    self.len_f = len(self.fluxes)

    init_u = scipy.linalg.lstsq(self.S.todense(),self.S_b)[0]
    
    init_s = abs(min(init_u))
    
    up_bound = self.flux_upper_bound
    
    ineq_mat = vstack( (identity(self.len_v),-identity(self.len_v)) )
    ineq_mat = hstack( (ineq_mat, 1*ones( (ineq_mat.shape[0],1) ) ) )
    ineq_vec = vstack( (zeros( (self.len_v,1)),up_bound * ones( (self.len_v,1) ) ) )

    S_with_x = hstack( (self.S.todense(),zeros( (self.S.shape[0],1) )) )

    # min( ||meas||)
    #
    # s.t.:
    #
    #  [I,-I]*u + [0*i,10*i] >= s
    #  Su = S_b
    
    f_prime = vstack( (zeros( (self.len_v,1) ),ones( (1,1) )) )
    init_x = vstack( (init_u, init_s) )
##    g_vs_s = 0.1 # scaling factor for the fluxes measurements
    
    g_vs_s = 1 # scaling factor for the fluxes measurements
    def func(x):
        a = self.flux_meas_mat*(sdiag(1/self.U.sum(1)).todense()*self.U).T*matrix(x[:-1]).T        
        return g_vs_s*(a.T*a - 2*a.T*self.flux_meas_values) + x[-1]
    def func_prime(x):
        a = (self.flux_meas_mat*(sdiag(1/self.U.sum(1)).todense()*self.U).T)
        return hstack( ( g_vs_s*(2*matrix(x[:-1])*(a.T*a) -2*self.flux_meas_values.T*a),ones((1,1))) ).tolist()[0]
    
    r = optimize.fmin_slsqp(func=func, #lambda x:x[self.len_v],
                            x0=init_x,                        
                            fprime = func_prime, #lambda x:f_prime,
                            f_ieqcons = lambda x: (ineq_mat*matrix(x).T + ineq_vec).tolist(),
                            fprime_ieqcons = lambda x: ineq_mat,
                            f_eqcons = lambda x: (S_with_x*matrix(x).T - self.S_b).tolist(),
                            fprime_eqcons = lambda x: S_with_x,                        
                            iprint=0, epsilon=1e-8,acc=1e-8)
    
    init_u = matrix(r[:-1]).T
    
    self.init_u=init_u
    
    u=self.init_u    

    u_1 = vstack((u,1))

    t = (self.beta_vec_M*u_1) / (self.beta_vec_N*u)
    t[-len(u):] = 1/u
    t_1 = vstack((t,1))
    uz = nonzero(u==0)
    t_1[-len(u)-1+uz[0]]=0
    

    tmp_dtm = 1/(self.beta_vec_N*u)
    d_tm = sparse.spdiags(tmp_dtm.A1,0,len(t),len(t))*self.beta_vec_M

    tmp_dtn = -(self.beta_vec_M*u_1)/((self.beta_vec_N*u).A1**2)
    d_tn = sparse.spdiags(tmp_dtn.A1,0,len(t),len(t))*self.beta_vec_N

    z = multiply((self.trans_mat_a_gamma*t_1), multiply((self.trans_mat_a_beta*t_1), (self.trans_mat_b_beta*t_1)) )
    
    # Generate init vector:
    v = zeros( (self.len_f,1) )
    for k in range(self.len_v):
        #v[self.U.getrow(k).indices[-1]]=u[k]
        v[self.U[k,:].indices[-1]]=u[k]
    
    meas_flux_values = self.labelinp_denom*v
    v = zeros( (self.len_f,1) )
    for k in range(shape(self.labelinp_num)[0]):
        tmpind = nonzero(self.labelinp_num[k,:].toarray()[0] > 0)
        if len(tmpind)>1:
            assert('Incorrect flux measurement.')            
        v[(tmpind[0])] = self.labelinp_value[k,0]*meas_flux_values[k,0]
    # upt/00000 = 3962
    
    v=matrix(v)
    self.orr_v = v
    #print self.U*v
    last_v=v
    
    tmp1 = self.al1*self.au
    tmp2 = self.al2*self.au

    ttmp1 = sparse.spdiags(z.T,[0],len(z),len(z))*self.al1
    ttmp2 = self.al2

    for k in range(800):        
        v = multiply(z, multiply( matrix(tmp1*v), (self.al2_0+tmp2*v )))
        if (k==0): # Let's see which rows are constant (input fluxes which are not set)
            ax = ttmp1*v
            bx = self.al2_0+ttmp2*v
            
            J = self.au*(sdiag(bx)*ttmp1 + sdiag(ax)*ttmp2) - sparse.identity(len(v))
            self.keeprows = nonzero(abs(J).sum(1))[0]        
        q = self.init_u / (self.U*v)        
####        print q
        q[ nonzero(self.U*v==0)]=0
        v = multiply(v,self.U.T*q)        
        if scipy.linalg.norm(v-last_v) < 1e-14:
            break
##        print scipy.linalg.norm(v-last_v)
        last_v=v
    ##print scipy.linalg.norm(self.U*v - self.init_u)
    self.sv=v
    self.init_v = v
    
    print('done.')

def get_normG(u, self):
    '''
    Main MFA optimization objective function value. 
    This function calculates || self.tG * x - self.tB || for a given u vector 
    by finding the valid x vector and then substituting it in the objective function.
    '''
#    tic()
##    print "****** Starting get_normG ******"
##    print u
    if len(shape(u))==1:
        u=matrix(u).T
    
    self.lastu = u
    u_1 = vstack((u,1)) #[u;1]
    t = (self.beta_vec_M*u_1) / (self.beta_vec_N*u)
    
    t[-len(u):] = 1/u
    t_1 = vstack((t,1)) #[t;1]

    # if one the the u's is zero we need to zero the 1/u element in t as well.    
    t_1 = nan_to_num(t_1)
    
##    t_1[-len(u)-1+uz[0]]=0
    
    #print t_1
    # Calc g(u) (here it is called z):
    z = multiply((self.trans_mat_a_gamma*t_1), multiply((self.trans_mat_a_beta*t_1), (self.trans_mat_b_beta*t_1)) )

    # Generate init vector:
    q = u/(self.U*self.sv)
    
##    q[uz[0]] = 0
    q = nan_to_num(q)
    v = self.au*(multiply(self.sv,(self.U.T*q)))    

    last_v = v

    tmp1 = sparse.spdiags(z.T,[0],len(z),len(z))*self.al1
    tmp2 = self.al2

    keeprows = self.keeprows
##    print '------ U*v ---------'
##    for k in range(len(u)):
##        print (self.U*v)[k], u[k]
##    print '-----------------------'
##    adsasdad
    for k in range(400):
        
        ax = tmp1*v
        bx = self.al2_0+tmp2*v
        nv = multiply(ax,bx)
##        q = u / (self.U*nv)
##        q[ nonzero(self.U*nv==0)]=0
##        nv = multiply(nv,self.U.T*q)
##        print self.U*nv
        p=scipy.linalg.norm(self.au*nv-self.au*last_v)                
        last_v=nv
        
        nv=self.au*nv
        if (abs(p) > 1e-4 ) or k==0:#(mod(k,50)==0): # let's see if we need to use Netwon's step            
            f = nv - v            
            ttmp1=tmp1.copy()
            ttmp2=tmp2.copy()
            sparse.sparsetools.csr_scale_rows(len(v),len(v),ttmp1.indptr,ttmp1.indices,ttmp1.data,array(bx.T)[0])
            sparse.sparsetools.csr_scale_rows(len(v),len(v),ttmp2.indptr,ttmp2.indices,ttmp2.data,array(ax.T)[0])
            J = self.au*(ttmp1 + ttmp2) - sparse.identity(len(v))
            #toc()
##            keeprows = nonzero(power(J,2).sum(1))[0]
            #keeprows = nonzero(J.sum(1))[0]
            #keeprows = nonzero(abs(J).sum(1))[0]
         
            r = sparse.linalg.dsolve.spsolve(J[keeprows,keeprows],f[keeprows])

            v[keeprows,0]-=r.T
        else:            
            v=nv


        if abs(p) < 1e-12:
            break
##    print "normG:",k,p
    #    print scipy.linalg.norm(v-last_v)
##    print p
    tv=multiply(ax,bx)
##    print '------ U*tv ---------'
##    for k in range(len(u)):
##        print (self.U*tv)[k], u[k]
##    print '-----------------------'
    
##    a = ((self.tG.todense()>0)*tv)    
##    b = ((self.tG.todense()!=0)*tv)
##    c = self.tG*tv    
##    for k in range(len(a)):        
##        print a[k]/b[k],self.tB[k], c[k], c[k]-self.tB[k]
##    print "--------------"
    val = scipy.linalg.norm(self.tG*tv-self.tB)
    self.tv = tv
    self.tv_val = val

##    print "****** Exiting get_normG ******"
    return val

#def get_normG_grad(self, u, gen_gradient=True):
def get_normG_grad(u, self):
    '''
    Main MFA optimization objective function gradient calculation. 
    This function calculates d (|| self.tG * x - self.tB ||) / du  for a given u vector.
    '''
##    print "****** Starting get_normG_grad ******"
    if len(shape(u))==1:
        u=matrix(u).T
    if any(self.lastu != u):
        get_normG(u,self)
        
    u_1 = vstack((u,1)) #[u;1]
    t = (self.beta_vec_M*u_1) / (self.beta_vec_N*u)
    t[-len(u):] = 1/u
    t_1 = vstack((t,1)) #[t;1]

    # if one the the u's is zero we need to zero the 1/u element in t as well.    
    uz = nonzero(u<=1e-7)
    t_1[-len(u)-1+uz[0]]=0
    t[-len(u)+uz[0]]=0
    
    # Calc beta_vec_M derivative
    tmp_dtm = 1/(self.beta_vec_N*u)
    d_tm = sparse.spdiags(tmp_dtm.T,0,len(t),len(t))*self.beta_vec_M

    # Calc beta_vec_N derivative
    tmp_dtn = -(self.beta_vec_M*u_1)/power((self.beta_vec_N*u),2)
    d_tn = sparse.spdiags(tmp_dtn.T,0,len(t),len(t))*self.beta_vec_N
    
    # Calc g(u) (here it is called z):
    z = multiply((self.trans_mat_a_gamma*t_1), multiply((self.trans_mat_a_beta*t_1), (self.trans_mat_b_beta*t_1)) )
    
    # Generate init vector:
##    q = u/(self.U*self.sv)
##    q[uz[0]]=0
    #v = self.au*(multiply(self.sv,(self.U.T*q)))
    v = self.tv
    tv=v
##    z = self.tv_z    
##    t_1 = self.tv_t_1
##    d_tm = self.tv_d_tm
##    d_tn = self.tv_d_tn
    
    ax = self.al1au*tv
    bx = self.al2au*tv + self.al2_0
    #ax = self.al1*self.au*tv
    #bx = self.al2*self.Gau*tv + self.al2_0
    #tmpones=ones(1,len(v))        
    for k in range(self.labelinp_value.shape[0]):            
        tmpind = nonzero(self.labelinp_num[0,:])[1]
        if len(tmpind) > 1:
            assert('Multi-fluxomers label input data are not supported yet.')
            print('Multi-fluxomers label input data are not supported yet.')            
        ax[tmpind] = 0
        bx[tmpind] = 0
        
    
    #self.tau = tau
    #J = self.au*sparse.spdiags(tz.T,[0],len(z),len(z))*( sparse.spdiags(bx.T,[0],len(v),len(v))*self.al1 + sparse.spdiags(ax.T,[0],len(v),len(v))*self.al2) - sparse.eye(len(v),len(v))

    ttmp1=self.al1.copy()
    ttmp2=self.al2.copy()
    sparse.sparsetools.csr_scale_rows(len(v),len(v),ttmp1.indptr,ttmp1.indices,ttmp1.data,array(bx.T)[0])
    sparse.sparsetools.csr_scale_rows(len(v),len(v),ttmp1.indptr,ttmp1.indices,ttmp1.data,array(z.T)[0])
    sparse.sparsetools.csr_scale_rows(len(v),len(v),ttmp2.indptr,ttmp2.indices,ttmp2.data,array(ax.T)[0])
    sparse.sparsetools.csr_scale_rows(len(v),len(v),ttmp2.indptr,ttmp2.indices,ttmp2.data,array(z.T)[0])
    J = self.au*(ttmp1 + ttmp2) - sparse.identity(len(v))
    
##    J = self.au*sparse.spdiags(z.T,[0],len(z),len(z))*( sparse.spdiags(bx.T,[0],len(v),len(v))*self.al1 + \
##                                                        sparse.spdiags(ax.T,[0],len(v),len(v))*self.al2) - \
##                                                        sparse.identity(len(v))

    
    #z = multiply((self.trans_mat_a_gamma*t_1), multiply((self.trans_mat_a_beta*t_1), (self.trans_mat_b_beta*t_1)) )
    tmp_dz = (sparse.spdiags(multiply((self.trans_mat_a_beta*t_1), (self.trans_mat_b_beta*t_1)).T,[0],len(v),len(v))*self.trans_mat_a_gamma[:,:-1]*sparse.spdiags(-power(t,2).T,[0],len(t),len(t)))[:,-len(u):]        
    d_tm=d_tm[:,:-1]        
    dz = sparse.spdiags(multiply((self.trans_mat_a_gamma*t_1), (self.trans_mat_b_beta*t_1)).T,[0],len(v),len(v)) * self.trans_mat_a_beta[:,:-1]*(d_tm + d_tn) + sparse.spdiags(multiply((self.trans_mat_a_gamma*t_1), (self.trans_mat_a_beta*t_1)).T,[0],len(v),len(v)) * self.trans_mat_b_beta[:,:-1]*(d_tm + d_tn) + tmp_dz        
    #P = (sparse.spdiags(multiply(ax,bx).T,[0],len(v),len(v))*dz).todense()        
    
    P = (sparse.spdiags(multiply(ax,bx).T,[0],len(v),len(v))*dz).todense()    
    
    for k in range(self.labelinp_value.shape[0]):
        tmpind = nonzero(self.labelinp_num[k,:])[1]
        if len(tmpind) > 1:
            assert('Multi-fluxomers label input data are not supported yet.')
            print('Multi-fluxomers label input data are not supported yet.') 
        #tmpind2 = nonzero(self.U*(self.labelinp_denom[k,:]).T)[0]
        tmpind2=self.labelinp_num_fluxes[k]
        P[tmpind[0],tmpind2] = self.labelinp_value[k,0]    
        
    asd = P
    P = self.au*P
    #print max(J)
    
    keeprows = nonzero(abs(J).sum(1))[0]
    #keeprows = nonzero(J.sum(1))[0]    
    J = J[keeprows,keeprows]
    #P = P[keeprows,:]
    
##    tL, tU, tP, tQ, tR, do_recip = umfpack.lu( J )
##    umfpack.free()
##    tP=sparse.csc_matrix( ([1]*len(tP),(tP,range(len(tP)))), (len(tP),len(tP)) )
##    tQ=sparse.csc_matrix( ([1]*len(tQ),(tQ,range(len(tQ)))), (len(tQ),len(tQ)) )
##    
##    if do_recip: # LU = PRAQ    
##        L = sparse.spdiags(1.0/tR,[0],len(tR),len(tR))*tP*tL        
##    else: # LU=P(R^-1)AQ        
##        L = sparse.spdiags(tR,[0],len(tR),len(tR))*tP*tL
##    U = tU*tQ.T
    
    tJ = sparse.linalg.dsolve.factorized(J)
    
    #tJ = sparse.linalg.dsolve.splu(J)
    #b = P[:,g].T.toarray()[0]
    d_xu = zeros( (shape(P)) )
    
    for g in range(shape(P)[1]):
##        tr = sparse.linalg.dsolve.spsolve(L,P[:,g]).T
##        tr2 = sparse.linalg.dsolve.spsolve(U,tr).T
        #tr = sparse.linalg.dsolve.spsolve(J,P[:,g]).T
        #tr = tJ( -array(P[keeprows.T,g]).T[0] )
        tr = tJ( - array(P[keeprows.T,g]).T[0] )
        d_xu[keeprows,g] = tr
    
    d_xu = asd + (sparse.spdiags(bx.T,[0],len(v),len(v))*sparse.spdiags(z.T,[0],len(v),len(v))*self.al1*d_xu + sparse.spdiags(ax.T,[0],len(v),len(v))*sparse.spdiags(z.T,[0],len(v),len(v))*self.al2*d_xu)
##    print (self.tG*tv - self.tB)[-13]
    dval = 0.5*(self.tv_val)**(-1)*(2*tv.T*self.tG.T*self.tG*d_xu - 2*matrix(self.tB).T*self.tG*d_xu)
##    print max(dval)
##    print max(real(dval.A[0])), min(real(dval.A[0]))
##    print real(dval.A[0])[23]
##    print "****** Exiting normG_grad ******"

    # In case the gradient is huge, let's normalize the result:

##    if any(max(dval)>1e6):
##        dval=dval/scipy.linalg.norm(dval)*scipy.linalg.norm(v)
##    dval=dval/scipy.linalg.norm(dval)*scipy.linalg.norm(v)

##    self.dval = dval
##    #dval=dval/scipy.linalg.norm(dval)*scipy.linalg.norm(v)
    #print scipy.linalg.norm(dval), max(max(dval))
##    dval=dval/1e7
##    dval=dval+1e-2
##    dval[dval<1e-1]=0
    return (real(dval.A[0]))


def load_data(fname):
#    global ftbl_file
    if debug_level > 1:
        print "Loading data..."
    f = open(fname)
    ftbl_file=pickle.load(f)
    if debug_level > 1:
        print " done."
    return ftbl_file

    
##def anaylze_correction_equation(formula):
##    """
##    This function returns the correction matrix for the chemical input formula
##    for compensating for naturally occuring non-carbon isotopomers in the input molecule.
##    Currently, the correction applies to the natural distribution of C,H,N,O,Si and S.
##    The isotopic compositions were based on values
##    reported by Rosman and Taylor (1998), and the distribution summary can be found
##    in van Winden (2002).
##    
##        Rosman, K.J.R., Taylor, P.D.P. (1998), Isotopic compositions of the elements
##        1997, Pure Appl. Chem., 70(1)-217.
##
##        van Winden, W. A., Wittmann, C., Heinzle, E., Heijnen, J. J. (2002), Correcting
##        mass isotopomer distributions for naturally occurring isotopes. Biotechnol. Bioeng.
##        80(4)-477.
##
##    C_dist = [0.9893, 0.0107]
##    H_dist = [0.999885, 0.000115]
##    N_dist = [0.99632, 0.00368]
##    O_dist = [0.99757, 0.00038, 0.00205]
##    Si_dist = [0.922297, 0.046832, 0.030872]
##    S_dist = [0.9493, 0.0076, 0.0429, 0.0002]
##
##    The code is very much based upon the code of Lake-Ee Quek, 
##        "OpenFLUX: efficient modelling software for 13C-based metabolic flux analysis"
##        Lake-Ee Quek , Christoph Wittmann , Lars K Nielsen  and Jens O Kromer
##        Microbial Cell Factories 2009, 8:25doi:10.1186/1475-2859-8-25
##    
##    This function is used for mass-spectrometry measurements analysis.
##    
##    @param formula: text chemical equation representing the investigated molecule without its carbon backbone. 
##    @return: correction matrix (res_mat*fluxomer = meas_value)
##    @rtype: matrix
##    """
##    import re
##    import itertools
##    CorIsos = {} # Correction dictionary
##    CorIsos['C'] = [0.9893, 0.0107]
##    CorIsos['H'] = [0.999885, 0.000115]
##    CorIsos['N'] = [0.99632, 0.00368]
##    CorIsos['O'] = [0.99757, 0.00038, 0.00205]
##    CorIsos['Si'] = [0.922297, 0.046832, 0.030872]
##    CorIsos['S'] = [0.9493, 0.0076, 0.0429, 0.0002]
##
##    f = re.findall(r'\d+|\D+', formula)
##    for k in range(0,len(f),2):
##        tmp_vec = zeros( (len(CorIsos[f[k]])*int(f[k+1]),1) )
##        tmp_dic = range(len(CorIsos[f[k]]))        
##        #print f[k], CorIsos[f[k]], int(f[k+1]), tmp_dic                
##        for z in itertools.product(tmp_dic,repeat=int(f[k+1])):
##            tmp_vec[sum(z)]+=reduce(lambda y,x:CorIsos[f[k]][x],z)
##
##    # THIS FUNCTION WAS NEVER FINISHED!!!
##    
def tic():
    global thetime
    thetime = time.time()
def toc():
    global thetime
    y=time.time()
    print y-thetime
    thetime = y


import sys
    
def main(argv):
    global debug_level
    global Evaluate_Only
    global run_optimization
    global upper_bound
    global virt_flux_value
    # parse command line options        
    try:
        opts, args = getopt.getopt(argv, "hi:t:o:b:v:", ["help","fia=","ftbl=","output_file=","upperbound=","virtflux="])
    except getopt.error, msg:
        print msg
        print "for help use --help"        
        sys.exit(2)
    
    Analyze_Only = False
    Evaluate_Only = False
    run_optimization = True
    i_or_t = False
    virt_flux_value = 1e-7 # default value
    upper_bound = 0 # flux upper bound default value
    
    output_file = ""                                   
    # process options
    for o, a in opts:
        if o in ("-h", "--help"):
            print __doc__
            sys.exit(0)
            
        if o in ("-i", "--fia"):
            ProjName = a
            inpfiletype = 'fia'
            i_or_t = True
            
        if o in ("-t", "--ftbl"):
            ProjName = a
            inpfiletype = 'ftbl'
            i_or_t = True
            
        if o in ("-o", "--output_file"):
            output_file = a
                 
        if o in ("-a", "--analyze_only"):            
            Analyze_Only = True

        if o in ("-e", "--evaluate_only"):            
            Evaluate_Only = True
            
        if o in ("-v", "--virtflux"):            
            virt_flux_value = eval(a)
            
        if o in ("-b", "--upperbound"):            
            upper_bound = eval(a)
            
    if not i_or_t:
        print __doc__
        sys.exit(2)    

    # process arguments
    for arg in args:
        print __doc__
        sys.exit(2)    

    print "\n FIA - Fluxomers Iterative Algorithm \n"        

    if output_file == "":
          output_file = ProjName + ".out"
                                   
    writer = MyWriter(sys.stdout, output_file)
    sys.stdout = writer

    ProjPathName = ProjName + "/"  # Used only for Matlab debug purposes
    debug_level = 2
    print time.asctime()
    print "Analyzing network:", ProjName, "\n"
    print "Saving output in:", output_file, "\n"
    sys.stdout.flush()
    gen_trans_mat = True
        
    ftbl_file = FtblFile(ProjName, inpfiletype)

    #print "*************************************"
    print "\nAll done.\n"
    sys.stdout.flush()


debug_level=2
####ProjName="examples/ecoli.ftbl"
####ProjName="examples/ecoli_3.ftbl"
####ProjName="examples/ecoli_as_theirs.ftbl"
####ProjName="examples/nature_ecoli.ftbl"
###ProjName="../OpenFLUX/CGLUT Example/cglut_orr.fia"
##ProjName="examples/openflux_example3.fia"
####ProjName="examples/openflux_example3.fia"
####ProjName="examples/openflux_example4.fia"
####ProjName="examples/oe5.fia"
####ProjName="examples/ex1.fia"
####ProjName="examples/oe8.fia"
####ProjName="examples/ex14.fia"
####ProjName="../OpenFLUX/OpenFLUX_v2.1_Windows/temp/temp.fia"
####ProjName="../comparing to examples/t.ftbl"
####ProjName="examples/OpenFlux_example1.fia"
####ProjName="examples/openflux_example2.fia"
#######ProjName="tmp.ftbl"
##virt_flux_value = 1e-7 # default value
##upper_bound = 500 # flux upper bound default value
##Evaluate_Only = False
##run_optimization = True
####inpfiletype="ftbl"
##inpfiletype="fia"
##ftbl_file = FtblFile(ProjName, inpfiletype)

if __name__ == '__main__':
    main(sys.argv[1:])
