function sol=solve_1d(a,b,c,d,tol)
% Solves the problem 
% (Pi) min (a(i) x -b(i))^2/(c|x(i)|^2+d) +log(c|x(i)|^2+d) s.t. x(i) is complex
% for every i=1,....,length(a)
%
% Based on the paper
% Amir Beck and Yonina Eldar, "Structured Total Maximum Likelihood:An Alternative to Structured Total Least-Squares"
% -----------------------------------------------------------------------
% Copyright (2009): Amir Beck and Yonina Eldar
% 
% FISTA is distributed under the terms of 
% the GNU General Public License 2.0.
% 
% Permission to use, copy, modify, and distribute this software for
% any purpose without fee is hereby granted, provided that this entire
% notice is included in all copies of any software which is or includes
% a copy or modification of this software and in all copies of the
% supporting documentation for such software.
% This software is being provided "as is", without any express or
% implied warranty.  In particular, the authors do not make any
% representation or warranty of any kind concerning the merchantability
% of this software or its fitness for any particular purpose."
%-------------------------------------------------------------
%
% input:    a ............................. a complex valued vector
%                    b ............................. a complex valued  vector (of the same length as a) 
%                    c ............................. a real number
%                    d ............................. a real number
%                    tol ........................ a tolerance parameter 
% output: sol.........................  the i-th coordinate of sol is the optimal solution of Pi


epsilon=1e-8;

if (sum(abs(a)<epsilon)~=0)
    error('the function requires that the vector a consists of nonzero elements')
end

n=length(a);
% The upper and lower bounds for the  1D solver are set

ub=d/c*(exp(abs(b).^2/d)-1);
ub=min(ub,(abs(b)./(abs(a))).^2);
lb=zeros(size(a));

% Zeroing the iteration numers

iter1=0;
iter2=0;

% construction of the signvector. The optimal solution has the same sign as
% conj(a)*b if it is non zero. Otherwise the sign is chosen to be 1.

signvector=sign(conj(a).*b);
P=find(signvector==0);
signvector(P)=1;

% Invoking a simple unimodal solver on a problem of
% the form  (|a|sqrt(x)-|b|)^2/(cx+d)+log(cx+d)

disp([' #ITER      #1D-PROBLEMS      #MAX-INTERVAL']) 
while (sum((ub-lb)>tol)>0)
    P=find((ub-lb)>tol);
    xm=2/3*lb(P)+1/3*ub(P);
    xp=1/3*lb(P)+2/3*ub(P);
    
    fm=(abs(a(P)).*sqrt(xm)-abs(b(P))).^2./(c*xm+d)+log(c*xm+d);
    fp=(abs(a(P)).*sqrt(xp)-abs(b(P))).^2./(c*xp+d)+log(c*xp+d);
    comp=(fm<=fp);
    ub(P)=comp.*xp+(1-comp).*ub(P);
    lb(P)=comp.*lb(P)+(1-comp).*xm;
    
    fprintf(1,'      %3d                   %6d                    %0.3e\n',iter1,length(P),max(ub-lb))
    iter1=iter1+1;
end

    
% Computing the solution vector

sol=((sqrt(ub)+sqrt(lb))/2).*signvector;
lb=sqrt(lb);
ub=sqrt(ub);
